/*!\file:  adjoint_core.cpp
 * \brief compute inverse method adjoint state
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"


void adjoint_core(FemModel* femmodel){
	
	
	/*parameters: */
	int   verbose=0;
	char* solverstring=NULL;
	bool  isstokes=false;
	bool  conserve_loads=true;
	int   dim;
	int   solution_type;
	
	/*intermediary: */
	Vec u_g=NULL;
	Mat K_ff0=NULL;
	Mat K_fs0=NULL;

	Vec du_g=NULL;
	Vec du_f=NULL;

	Vec adjoint_f=NULL;
	Vec adjoint_g=NULL;

	/*retrieve parameters:*/
	femmodel->parameters->FindParam(&solverstring,SolverStringEnum);
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);

	/*set analysis type to compute velocity: */
	if(isstokes)femmodel->SetCurrentAnalysis(DiagnosticStokesAnalysisEnum);
	else femmodel->SetCurrentAnalysis(DiagnosticHorizAnalysisEnum);
	
	_printf_("%s\n","      recover solution for this stiffness and right hand side:");
	solver_diagnostic_nonlinear(NULL,&K_ff0,&K_fs0, femmodel,conserve_loads); 

	_printf_("%s\n","      buid Du, difference between observed velocity and model velocity:");
	Dux(&du_g, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

	_printf_("%s\n","      reduce adjoint load from g-set to f-set:");
	Reduceloadfromgtofx(&du_f, du_g, femmodel->Gmn, K_fs0, femmodel->ys, femmodel->nodesets,true); //true means that ys0 flag is activated: all spcs show 0 displacement
	
	/*free some ressources: */
	VecFree(&du_g);MatFree(&K_fs0);

	_printf_("%s\n","      solve for adjoint vector:");
	Solverx(&adjoint_f, K_ff0, du_f, NULL, solverstring);
	
	/*free some ressources: */
	VecFree(&du_f); MatFree(&K_ff0);
	
	_printf_("%s\n","      merge back to g set:");
	Mergesolutionfromftogx(&adjoint_g, adjoint_f,femmodel->Gmn,femmodel->ys,femmodel->nodesets,true);//true means that ys0 flag is activated: all spc are 0
	
	/*free some ressources: */
	VecFree(&adjoint_f);

	/*Update inputs using adjoint solution, and same type of setup as diagnostic solution: */
	if(isstokes)femmodel->SetCurrentAnalysisAlias(DiagnosticStokesAnalysisEnum,AdjointAnalysisEnum);
	else femmodel->SetCurrentAnalysisAlias(DiagnosticHorizAnalysisEnum,AdjointAnalysisEnum);
	
	InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,adjoint_g);

	if(verbose)_printf_("saving results:\n");
	if(solution_type==AdjointSolutionEnum){
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,AdjointxEnum);
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,AdjointyEnum);
		if(dim==3) InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,AdjointzEnum);
	}
	
	/*Free ressources:*/
	xfree((void**)&solverstring);
	VecFree(&u_g);
	MatFree(&K_ff0);
	MatFree(&K_fs0);
	VecFree(&du_g);
	VecFree(&du_f);
	VecFree(&adjoint_f);
	VecFree(&adjoint_g);

}
