/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

void control_core(FemModel* femmodel){

	int     i,n;
	
	/*parameters: */
	int     verbose=0;
	int     control_type;
	int     control_steady;
	int     nsteps;
	double  eps_cm;
	double  tolx;
	double  cm_min;
	double  cm_max;
	int     cm_gradient;
	int     dim;

	double* fit=NULL;
	double* optscal=NULL;
	double* maxiter=NULL;
	double* cm_jump=NULL;

		
	/*intermediary: */
	double  search_scalar;
	OptArgs optargs;
	OptPars optpars;

	/*output: */
	double* J=NULL;

	/*some preliminary work to be done if running full-Stokes analysis: */
	stokescontrolinit(femmodel);

	/*Recover parameters used throughout the solution:{{{1*/
	femmodel->parameters->FindParam(&nsteps,NStepsEnum);
	femmodel->parameters->FindParam(&control_type,ControlTypeEnum);
	femmodel->parameters->FindParam(&fit,NULL,NULL,FitEnum);
	femmodel->parameters->FindParam(&optscal,NULL,NULL,OptScalEnum);
	femmodel->parameters->FindParam(&maxiter,NULL,NULL,MaxIterEnum);
	femmodel->parameters->FindParam(&cm_jump,NULL,NULL,CmJumpEnum);
	femmodel->parameters->FindParam(&eps_cm,EpsCmEnum);
	femmodel->parameters->FindParam(&tolx,TolXEnum);
	femmodel->parameters->FindParam(&cm_min,CmMinEnum);
	femmodel->parameters->FindParam(&cm_max,CmMaxEnum);
	femmodel->parameters->FindParam(&cm_gradient,CmGradientEnum);
	femmodel->parameters->FindParam(&control_steady,ControlSteadyEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	/*}}}*/

	/*Initialize misfit: */
	J=(double*)xmalloc(nsteps*sizeof(double));
		
	/*Initialize some of the BrentSearch arguments: */
	optargs.femmodel=femmodel;
	optpars.xmin=0; optpars.xmax=1; optpars.tolerance=tolx ;
	
	/*Start looping: */
	for(n=0;n<nsteps;n++){

		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,fit[n],FitEnum);
		
		/*In case we are running a steady state control method, compute new temperature field using new parameter * distribution: */
		if (control_steady) steadystate_core(femmodel);
	
		_printf_("%s\n","      computing gradJ...");
		gradient_core(femmodel);

		/*Return gradient if asked: */
		if (cm_gradient){
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,GradientEnum);
			goto cleanup_and_return;
		}

		_printf_("%s\n","      optimizing along gradient direction");
		optargs.n=n; optpars.maxiter=(int)maxiter[n]; optpars.cm_jump=cm_jump[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);

		_printf_("%s","      updating parameter using optimized search scalar...");
		InputAXPYx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,control_type,search_scalar*optscal[n],ControlParameterEnum);

		_printf_("%s","      constraining the new distribution...");    
		InputControlConstrainx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,control_type,cm_min,cm_max);
		
		_printf_("%s","      save new parameter...");
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,control_type,ControlParameterEnum);
		
		_printf_("%s%i%s%g\n","      value of misfit J after optimization #",n+1,": ",J[n]);
		
		if(controlconvergence(J,fit,eps_cm,n))goto convergence_point;

		/*Temporary saving every 5 control steps: */
		if (((n+1)%5)==0){
			_printf_("%s","      saving temporary results...");
			controlrestart(femmodel,J);
		}
	}


	convergence_point:
	_printf_("%s","      preparing final velocity solution");
	if (control_steady) steadystate_core(femmodel);
	else diagnostic_core(femmodel);

	/*some results not computed by steadystate_core or diagnostic_core: */
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,control_type); //the parameter itself!
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VxEnum); 
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VyEnum); 
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VelEnum); 
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,GradientEnum); 
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,AdjointxEnum); 
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,AdjointyEnum); 
	if(dim==3)InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VzEnum); 
	femmodel->results->AddObject(new DoubleVecExternalResult(femmodel->results->Size()+1,JEnum,J,nsteps,1,0));
	femmodel->results->AddObject(new StringExternalResult(femmodel->results->Size()+1,ControlTypeEnum,EnumAsString(control_type),1,0));

	cleanup_and_return:
	/*Free ressources: */
	xfree((void**)&control_type);
	xfree((void**)&fit);
	xfree((void**)&optscal);
	xfree((void**)&maxiter);
	xfree((void**)&cm_jump);
	xfree((void**)&J);
}
