/*!\file Beam.c
 * \brief: implementation of the Beam object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"

/*Beam constructors and destructor*/
/*FUNCTION Beam::Beam(){{{1*/
Beam::Beam(){
	this->inputs=NULL;
	this->parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION Beam::~Beam(){{{1*/
Beam::~Beam(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Beam::copy{{{1*/
Object* Beam::copy() {

	int i;
	Beam* beam=NULL;

	beam=new Beam();

	/*copy fields: */
	beam->id=this->id;
	if(this->inputs){
		beam->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		beam->inputs=new Inputs();
	}
	/*point parameters: */
	beam->parameters=this->parameters;

	/*pointers: */
	beam->nodes=(Node**)xmalloc(2*sizeof(Node*)); 
	for(i=0;i<2;i++)beam->nodes[i]=this->nodes[i];
	beam->matice=this->matice;
	beam->matpar=this->matpar;

	return beam;
}
/*}}}*/
/*FUNCTION Beam::DeepEcho{{{1*/
void Beam::DeepEcho(void){

	int i;

	printf("Beam:\n");
	printf("   id: %i\n",id);
	for(i=0;i<2;i++){
		nodes[i]->DeepEcho();
	}
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Beam::Demarshall{{{1*/
void  Beam::Demarshall(char** pmarshalled_dataset){
	ISSMERROR("not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Echo {{{1*/
void Beam::Echo(void){

	int i;

	printf("Beam:\n");
	printf("   id: %i\n",id);
	for(i=0;i<2;i++){
		nodes[i]->DeepEcho();
	}
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   parameters\n");
	parameters->Echo();
	printf("   inputs\n");
	inputs->Echo();

	return;
}
/*}}}*/
/*FUNCTION Beam::Enum{{{1*/
int Beam::Enum(void){

	return BeamEnum;

}
/*}}}*/
/*FUNCTION Beam::Id{{{1*/
int    Beam::Id(void){ return id; }
/*}}}*/
/*FUNCTION Beam::Marshall{{{1*/
void  Beam::Marshall(char** pmarshalled_dataset){
	ISSMERROR("not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MarshallSize{{{1*/
int   Beam::MarshallSize(){
	ISSMERROR("not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MyRank{{{1*/
int    Beam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Update virtual functions definitions:*/
/*FUNCTION Beam::InputUpdateFromVector(double* vector, int name, int type);{{{1*/
void  Beam::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;
	switch(type){

		case VertexEnum:

			/*New PentaVertexInpu*/
			double values[2];

			/*Get values on the 6 vertices*/
			for (int i=0;i<2;i++){
				values[i]=vector[nodes[i]->GetVertexDof()];
			}

			/*update input*/
			this->inputs->AddInput(new BeamVertexInput(name,values));
			return;

		default:

			ISSMERROR("type %i (%s) not implemented yet",type,EnumAsString(type));
	}
}
/*}}}*/
/*FUNCTION Beam::InputUpdateFromVector(int* vector, int name, int type);{{{1*/
void  Beam::InputUpdateFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::InputUpdateFromVector(bool* vector, int name, int type);{{{1*/
void  Beam::InputUpdateFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::InputUpdateFromSolution {{{1*/
void  Beam::InputUpdateFromSolution(double* solution){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Element virtual functions definitions:*/
/*FUNCTION Beam::ComputeBasalStress{{{1*/
void  Beam::ComputeBasalStress(Vec eps){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Beam::ComputePressure{{{1*/
void  Beam::ComputePressure(Vec p_g){

	int i;
	const int numgrids=2;
	int doflist[numgrids];
	double pressure[numgrids];
	double surface[numgrids];
	double rho_ice,g;
	double xyz_list[numgrids][3];
	double gauss[numgrids][numgrids]={{1,0},{0,1}};
	int analysis_type,sub_analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	parameters->FindParam(&sub_analysis_type,AnalysisTypeEnum);

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
        
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*recover value of surface at gauss points (0,1) and (1,0): */
	inputs->GetParameterValues(&surface[0],&gauss[0][0],2,SurfaceEnum);
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(p_g,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Beam::ComputeStrainRate{{{1*/
void  Beam::ComputeStrainRate(Vec eps){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Beam::Configure {{{1*/
void  Beam::Configure(Elements* elementsin,Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){

	ISSMERROR(" not supported yet!");

}
/*}}}*/
/*FUNCTION Beam::CostFunction{{{1*/
double Beam::CostFunction(void){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::CreateKMatrix{{{1*/
void  Beam::CreateKMatrix(Mat Kgg){

	int analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHutterAnalysisEnum) {
		CreateKMatrixDiagnosticHutter( Kgg);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}

}
/*}}}*/
/*FUNCTION Beam::CreatePVector{{{1*/
void  Beam::CreatePVector(Vec pg){

	int analysis_type,sub_analysis_type;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	parameters->FindParam(&sub_analysis_type,AnalysisTypeEnum);
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHutterAnalysisEnum) {
		CreatePVectorDiagnosticHutter( pg);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}

}
/*}}}*/
/*FUNCTION Beam::Du{{{1*/
void  Beam::Du(Vec){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetBedList{{{1*/
void  Beam::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetMatPar{{{1*/
void* Beam::GetMatPar(){

	return matpar;
}
/*}}}*/
/*FUNCTION Beam::GetNodes{{{1*/
void  Beam::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=NULL;

	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Beam::GetOnBed{{{1*/
bool   Beam::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetShelf{{{1*/
bool   Beam::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetSolutionFromInputs(Vec solution);{{{1*/
void  Beam::GetSolutionFromInputs(Vec solution){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetThicknessList{{{1*/
void  Beam::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetVectorFromInputs(Vec vector,int NameEnum){{{1*/
void  Beam::GetVectorFromInputs(Vec vector,int NameEnum){

	int i;
	const int numvertices=2;
	int doflist1[numvertices];

	/*Find NameEnum input in the inputs dataset, and get it to fill in the vector: */
	for(i=0;i<this->inputs->Size();i++){
		Input* input=(Input*)this->inputs->GetObjectByOffset(i);
		if(input->EnumType()==NameEnum){
			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			this->GetDofList1(&doflist1[0]);
			input->GetVectorFromInputs(vector,&doflist1[0]);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Beam::Gradj{{{1*/
void  Beam::Gradj(Vec gradient,int control_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GradjB{{{1*/
void  Beam::GradjB(Vec gradient){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GradjDrag{{{1*/
void  Beam::GradjDrag(Vec gradient){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::InputAXPY(int YEnum, double scalar, int XEnum);{{{1*/
void  Beam::InputAXPY(int YEnum, double scalar, int XEnum){

	Input* xinput=NULL;
	Input* yinput=NULL;

	/*Find x and y inputs: */
	xinput=(Input*)this->inputs->GetInput(XEnum);
	yinput=(Input*)this->inputs->GetInput(YEnum);

	/*some checks: */
	if(!xinput || !yinput)ISSMERROR("%s%s%s%s%s"," input ",EnumAsString(XEnum)," or input ",EnumAsString(YEnum)," could not be found!");
	if(xinput->Enum()!=yinput->Enum())ISSMERROR("%s%s%s%s%s"," input ",EnumAsString(XEnum)," and input ",EnumAsString(YEnum)," are not of the same type!");

	/*Scale: */
	yinput->AXPY(xinput,scalar);
}
/*}}}*/
/*FUNCTION Beam::InputControlConstrain(int control_type, double cm_min, double cm_max){{{1*/
void  Beam::InputControlConstrain(int control_type, double cm_min, double cm_max){

	Input* input=NULL;

	/*Find input: */
	input=(Input*)this->inputs->GetInput(control_type);
	
	/*Do nothing if we  don't find it: */
	if(!input)return;

	/*Constrain input using cm_min and cm_max: */
	input->Constrain(cm_min,cm_max);

}
/*}}}*/
/*FUNCTION Beam::InputConvergence(int* pconverged, double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){{{1*/
void  Beam::InputConvergence(int* pconverged,double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){

	int     i;
	Input** new_inputs=NULL;
	Input** old_inputs=NULL;
	int     converged=1;

	new_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the new inputs
	old_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the old inputs
	
	for(i=0;i<num_enums/2;i++){
		new_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+0]);
		old_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+1]);
		if(!new_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumAsString(enums[2*i+0]));
		if(!old_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumAsString(enums[2*i+0]));
	}

	/*ok, we've got the inputs (new and old), now loop throught the number of criterions and fill the eps array:*/
	for(i=0;i<num_criterionenums;i++){
		IsInputConverged(eps+i,new_inputs,old_inputs,num_enums/2,criterionenums[i]);
		if(eps[i]>criterionvalues[i]) converged=0; 
	}

	/*Assign output pointers:*/
	*pconverged=converged;

}
/*}}}*/
/*FUNCTION Beam::InputDuplicate(int original_enum,int new_enum){{{1*/
void  Beam::InputDuplicate(int original_enum,int new_enum){

	Input* original=NULL;
	Input* copy=NULL;

	/*Make a copy of the original input: */
	original=(Input*)this->inputs->GetInput(original_enum);
	copy=(Input*)original->copy();

	/*Change copy enum to reinitialized_enum: */
	copy->ChangeEnum(new_enum);

	/*Add copy into inputs, it will wipe off the one already there: */
	inputs->AddObject((Input*)copy);
}
/*}}}*/
/*FUNCTION Beam::InputScale(int enum_type,double scale_factor){{{1*/
void  Beam::InputScale(int enum_type,double scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Beam::InputToResult(int enum_type,int step,double time){{{1*/
void  Beam::InputToResult(int enum_type,int step,double time){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MassFlux{{{1*/
double Beam::MassFlux( double* segment){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MaxAbsVx(double* pmaxabsvx, bool process_units);{{{1*/
void  Beam::MaxAbsVx(double* pmaxabsvx, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vx_values[numgrids];
	double  maxabsvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*now, compute maximum:*/
	maxabsvx=fabs(vx_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vx_values[i])>maxabsvx)maxabsvx=fabs(vx_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Beam::MaxAbsVy(double* pmaxabsvy, bool process_units);{{{1*/
void  Beam::MaxAbsVy(double* pmaxabsvy, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vy_values[numgrids];
	double  maxabsvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*now, compute maximum:*/
	maxabsvy=fabs(vy_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vy_values[i])>maxabsvy)maxabsvy=fabs(vy_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Beam::MaxAbsVz(double* pmaxabsvz, bool process_units);{{{1*/
void  Beam::MaxAbsVz(double* pmaxabsvz, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vz_values[numgrids];
	double  maxabsvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*now, compute maximum:*/
	maxabsvz=fabs(vz_values[0]);
	for(i=1;i<numgrids;i++){
		if (fabs(vz_values[i])>maxabsvz)maxabsvz=fabs(vz_values[i]);
	}

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Beam::MaxVel(double* pmaxvel, bool process_units);{{{1*/
void  Beam::MaxVel(double* pmaxvel, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vx_values[numgrids];
	double  vy_values[numgrids];
	double  vz_values[numgrids];
	double  vel_values[numgrids];
	double  maxvel;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);
	if(dim==3) inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*now, compute maximum of velocity :*/
	if(dim==2){
		for(i=0;i<numgrids;i++)vel_values[i]=sqrt(pow(vx_values[i],2)+pow(vy_values[i],2));
	}
	else{
		for(i=0;i<numgrids;i++)vel_values[i]=sqrt(pow(vx_values[i],2)+pow(vy_values[i],2)+pow(vz_values[i],2));
	}

	/*now, compute maximum:*/
	maxvel=vel_values[0];
	for(i=1;i<numgrids;i++){
		if (vel_values[i]>maxvel)maxvel=vel_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvel=maxvel;

}
/*}}}*/
/*FUNCTION Beam::MaxVx(double* pmaxvx, bool process_units);{{{1*/
void  Beam::MaxVx(double* pmaxvx, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vx_values[numgrids];
	double  maxvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*now, compute maximum:*/
	maxvx=vx_values[0];
	for(i=1;i<numgrids;i++){
		if (vx_values[i]>maxvx)maxvx=vx_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvx=maxvx;

}
/*}}}*/
/*FUNCTION Beam::MaxVy(double* pmaxvy, bool process_units);{{{1*/
void  Beam::MaxVy(double* pmaxvy, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vy_values[numgrids];
	double  maxvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*now, compute maximum:*/
	maxvy=vy_values[0];
	for(i=1;i<numgrids;i++){
		if (vy_values[i]>maxvy)maxvy=vy_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvy=maxvy;

}
/*}}}*/
/*FUNCTION Beam::MaxVz(double* pmaxvz, bool process_units);{{{1*/
void  Beam::MaxVz(double* pmaxvz, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vz_values[numgrids];
	double  maxvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*now, compute maximum:*/
	maxvz=vz_values[0];
	for(i=1;i<numgrids;i++){
		if (vz_values[i]>maxvz)maxvz=vz_values[i];
	}

	/*Assign output pointers:*/
	*pmaxvz=maxvz;

}
/*}}}*/
/*FUNCTION Beam::MinVel(double* pminvel, bool process_units);{{{1*/
void  Beam::MinVel(double* pminvel, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vx_values[numgrids];
	double  vy_values[numgrids];
	double  vz_values[numgrids];
	double  vel_values[numgrids];
	double  minvel;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);
	if(dim==3) inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*now, compute minimum of velocity :*/
	if(dim==2){
		for(i=0;i<numgrids;i++)vel_values[i]=sqrt(pow(vx_values[i],2)+pow(vy_values[i],2));
	}
	else{
		for(i=0;i<numgrids;i++)vel_values[i]=sqrt(pow(vx_values[i],2)+pow(vy_values[i],2)+pow(vz_values[i],2));
	}

	/*now, compute minimum:*/
	minvel=vel_values[0];
	for(i=1;i<numgrids;i++){
		if (vel_values[i]<minvel)minvel=vel_values[i];
	}

	/*Assign output pointers:*/
	*pminvel=minvel;

}
/*}}}*/
/*FUNCTION Beam::MinVx(double* pminvx, bool process_units);{{{1*/
void  Beam::MinVx(double* pminvx, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vx_values[numgrids];
	double  minvx;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vx_values[0],&gaussgrids[0][0],numgrids,VxEnum);

	/*now, compute minimum:*/
	minvx=vx_values[0];
	for(i=1;i<numgrids;i++){
		if (vx_values[i]<minvx)minvx=vx_values[i];
	}

	/*Assign output pointers:*/
	*pminvx=minvx;

}
/*}}}*/
/*FUNCTION Beam::MinVy(double* pminvy, bool process_units);{{{1*/
void  Beam::MinVy(double* pminvy, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vy_values[numgrids];
	double  minvy;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vy_values[0],&gaussgrids[0][0],numgrids,VyEnum);

	/*now, compute minimum:*/
	minvy=vy_values[0];
	for(i=1;i<numgrids;i++){
		if (vy_values[i]<minvy)minvy=vy_values[i];
	}

	/*Assign output pointers:*/
	*pminvy=minvy;

}
/*}}}*/
/*FUNCTION Beam::MinVz(double* pminvz, bool process_units);{{{1*/
void  Beam::MinVz(double* pminvz, bool process_units){

	int i;
	int dim;
	const int numgrids=2;
	double  gaussgrids[numgrids][2]={{0,1},{1,0}};
	double  vz_values[numgrids];
	double  minvz;

	/*retrieve dim parameter: */
	parameters->FindParam(&dim,DimEnum);

	/*retrive velocity values at nodes */
	inputs->GetParameterValues(&vz_values[0],&gaussgrids[0][0],numgrids,VzEnum);

	/*now, compute minimum:*/
	minvz=vz_values[0];
	for(i=1;i<numgrids;i++){
		if (vz_values[i]<minvz)minvz=vz_values[i];
	}

	/*Assign output pointers:*/
	*pminvz=minvz;

}
/*}}}*/
/*FUNCTION Beam::Misfit{{{1*/
double Beam::Misfit(void){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::PatchFill(int* pcount, Patch* patch){{{1*/
void  Beam::PatchFill(int* pcount, Patch* patch){

	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){{{1*/
void  Beam::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){

	ISSMERROR(" not supported yet!");
	
}
/*}}}*/
/*FUNCTION Beam::ProcessResultsUnits(void){{{1*/
void  Beam::ProcessResultsUnits(void){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::SurfaceArea{{{1*/
double Beam::SurfaceArea(void){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Update(int index, IoModel* iomodel,int analysis_counter,int analysis_type);{{{1*/
void Beam::Update(int index, IoModel* iomodel,int analysis_counter,int analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Beam specific routines: */
/*FUNCTION Beam::CreateKMatrixDiagnosticHutter{{{1*/

void  Beam::CreateKMatrixDiagnosticHutter(Mat Kgg){
	
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    Ke_gg[numdofs][numdofs]={{0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0}};
	int       numberofdofspernode;
	bool onbed;
	bool onsurface;
	int connectivity[2];
	double one0,one1;
	
	connectivity[0]=nodes[0]->GetConnectivity();
	connectivity[1]=nodes[1]->GetConnectivity();
	
	one0=1/(double)connectivity[0];
	one1=1/(double)connectivity[1];
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	GetDofList(&doflist[0],&numberofdofspernode);

	if (onbed){
		Ke_gg[0][0]=one0;
		Ke_gg[1][1]=one0;
		Ke_gg[2][0]=-2*one1;
		Ke_gg[2][2]=2*one1;
		Ke_gg[3][1]=-2*one1;
		Ke_gg[3][3]=2*one1;
	}
	else if (onsurface){
		Ke_gg[2][0]=-one1;
		Ke_gg[2][2]=one1;
		Ke_gg[3][1]=-one1;
		Ke_gg[3][3]=one1;
	}
	else{ //node is on two horizontal layers and beams include the values only once, so the have to use half of the connectivity
		Ke_gg[2][0]=-2*one1;
		Ke_gg[2][2]=2*one1;
		Ke_gg[3][1]=-2*one1;
		Ke_gg[3][3]=2*one1;
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Beam::CreatePVectorDiagnosticHutter{{{1*/

void Beam::CreatePVectorDiagnosticHutter( Vec pg){

	int i,j,k;
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    pe_g[4]={0,0,0,0};
	double    pe_g_gaussian[4]={0,0,0,0};
	int       found=0;
	int       dofs[1]={0};
	double    xyz_list[numgrids][3];
	double    z_list[numgrids];
	double    constant_part;
	int       numberofdofspernode;

	/*gaussian points: */
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double   gauss_coord;
	double*  gauss_weights=NULL;
	double   gauss_weight;
	double   gauss1[2]={1,0};
	int      ig;
	double   l1l2[2];
	double   slope[2];
	double   slope2;

	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;
	double   surface,thickness;
	
	bool onbed;
	bool onsurface;
	int  connectivity[2];

	/*recover doflist: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*recover parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();

	//recover extra inputs
	inputs->GetParameterValue(&slope[0],&gauss1[0],SurfaceSlopeXEnum);
	inputs->GetParameterValue(&slope[1],&gauss1[0],SurfaceSlopeYEnum);
	inputs->GetParameterValue(&surface,&gauss1[0],SurfaceEnum);
	inputs->GetParameterValue(&thickness,&gauss1[0],ThicknessEnum);

	connectivity[0]=nodes[0]->GetConnectivity();
	connectivity[1]=nodes[1]->GetConnectivity();

	//Get all element grid data:
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	for(i=0;i<numgrids;i++)z_list[i]=xyz_list[i][2];
	
	//compute slope2 slopex and slopy
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//%compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	GaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	//Start  looping on the number of gaussian points:
	for(ig=0;ig<num_gauss;ig++){

		//Pick up the gaussian point and its weight:
		gauss_weight=gauss_weights[ig];
		gauss_coord=segment_gauss_coord[ig];

		//compute constant_part
		GetParameterValue(&z_g, &z_list[0],gauss_coord);

		//Get Jacobian determinant:
		GetJacobianDeterminant(&Jdet, &z_list[0],gauss_coord);
		
		/*Add contribution*/
		if (onsurface){
			for(j=0;j<NDOF2;j++){
				pe_g_gaussian[NDOF2+j]=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight/(double)connectivity[1];
			}
		}
		else{//connectivity is too large, should take only half on it
			for(j=0;j<NDOF2;j++){
				pe_g_gaussian[NDOF2+j]=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight*2/(double)connectivity[1];
			}
		}
		
		//add pe_gaussian vector to pe:
		for(j=0;j<numdofs;j++){
			pe_g[j]+=pe_g_gaussian[j];
		}
	} //for(ig=0;ig<num_gauss;ig++)

	//Deal with lower surface
	if (onbed){

		//compute ub
		constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*thickness;
		ub=constant_part*slope[0];
		vb=constant_part*slope[1];

		//Add to pe: 
		pe_g[0]+=ub/(double)connectivity[0];
		pe_g[1]+=vb/(double)connectivity[0];
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&segment_gauss_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Beam::GetDofList{{{1*/
void  Beam::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	for(i=0;i<2;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Beam::GetDofList1{{{1*/
void  Beam::GetDofList1(int* doflist){

	int i;
	for(i=0;i<2;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Beam::GetJacobianDeterminant{{{1*/
void Beam::GetJacobianDeterminant(double* pJdet,double* z_list, double gauss_coord){


	double Jdet;

	Jdet=1.0/2.0*(z_list[1]-z_list[0]);

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}
	
	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Beam::GetNodalFunctions{{{1*/
void Beam::GetNodalFunctions(double* l1l2, double gauss_coord){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2[0]=-0.5*gauss_coord+0.5;

	/*Second nodal function: */
	l1l2[1]=+0.5*gauss_coord+0.5;
}
/*}}}*/
/*FUNCTION Beam::GetParameterValue{{{1*/
void Beam::GetParameterValue(double* pvalue, double* value_list,double gauss_coord){

	double l1l2[2];
	
	GetNodalFunctions(&l1l2[0],gauss_coord);

	*pvalue=l1l2[0]*value_list[0]+l1l2[1]*value_list[1];
}
/*}}}*/
/*FUNCTION Beam::IsInput{{{1*/
bool Beam::IsInput(int name){
	if (name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum){
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Beam::SetClone {{{1*/
void  Beam::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
