/*
 * \file DataSet.c
 * \brief: implementation of the DataSet class, and derived classes Inputs and Parameters
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Constructors/Destructors*/
/*FUNCTION DataSet::DataSet(){{{1*/
DataSet::DataSet(){
	
	sorted=0;
	sorted_ids=NULL;
	id_offsets=NULL;

}
/*}}}*/
/*FUNCTION DataSet::DataSet(int dataset_enum){{{1*/
DataSet::DataSet(int dataset_enum){
	enum_type=dataset_enum;
	
	sorted=0;
	sorted_ids=NULL;
	id_offsets=NULL;

}
/*}}}*/
/*FUNCTION DataSet::Copy{{{1*/
DataSet*   DataSet::Copy(void){


	DataSet* copy=NULL;
	vector<Object*>::iterator object;
	Object* object_copy=NULL;

	copy=new DataSet(enum_type);

	copy->sorted=sorted;
	copy->presorted=presorted;
	if(sorted_ids){
		copy->sorted_ids=(int*)xmalloc(objects.size()*sizeof(int));
		memcpy(copy->sorted_ids,sorted_ids,objects.size()*sizeof(int));
	}
	if(id_offsets){
		copy->id_offsets=(int*)xmalloc(objects.size()*sizeof(int));
		memcpy(copy->id_offsets,id_offsets,objects.size()*sizeof(int));
	}

	/*Now we need to deep copy the objects: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		object_copy = (*object)->copy();
		copy->AddObject(object_copy);
	}
	return copy;
}
/*}}}*/
/*FUNCTION DataSet::~DataSet{{{1*/
DataSet::~DataSet(){
	clear();
	xfree((void**)&sorted_ids);
	xfree((void**)&id_offsets);
}
/*}}}*/

/*I/O*/
/*FUNCTION DataSet::Marshall{{{1*/
char* DataSet::Marshall(){

	vector<Object*>::iterator object;
	int                       object_size;
	int                       marshalled_dataset_size=0;
	char*                     marshalled_dataset=NULL;
	char*                     old_marshalled_dataset=NULL;

	/*First get size of marshalled dataset: */
	object_size=(int)objects.size();

	marshalled_dataset_size=MarshallSize();
	
	/*Allocate marshalled dataset: */
	marshalled_dataset=(char*)xmalloc(marshalled_dataset_size*sizeof(char)); 

	/*Keep track of old_marshalled_dataset: */
	old_marshalled_dataset=marshalled_dataset;

	/*Store internals of dataset first: */
	memcpy(marshalled_dataset,&object_size,sizeof(int)); marshalled_dataset+=sizeof(int);
	memcpy(marshalled_dataset,&sorted,sizeof(int)); marshalled_dataset+=sizeof(int);
	if(sorted){
		if(object_size)memcpy(marshalled_dataset,sorted_ids,object_size*sizeof(int)); marshalled_dataset+=object_size*sizeof(int);
		if(object_size)memcpy(marshalled_dataset,id_offsets,object_size*sizeof(int)); marshalled_dataset+=object_size*sizeof(int);
	}

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		(*object)->Marshall(&marshalled_dataset);
	}

	/* Ok, marshalled_dataset now points to the end of the original marshalled_dataset pointer 
	 * before  we started the loop on objects. Get object to point right again: */
	marshalled_dataset-=marshalled_dataset_size;

	/*We should be back to old_marshalled_dataset: check and abort if that's not the case, 
	 * because this is a nasty error: */
	if (marshalled_dataset!=old_marshalled_dataset){
		ISSMERROR("final marshalled dataset is different from initial one!"); 
		abort();
	}

	/*Return: */
	return marshalled_dataset;
}
/*}}}*/
/*FUNCTION DataSet::MarshallSize{{{1*/
int DataSet::MarshallSize(){

	vector<Object*>::iterator object;
	int                      marshalled_dataset_size=0;


	for ( object=objects.begin() ; object < objects.end(); object++ ){
		marshalled_dataset_size+= (*object)->MarshallSize();
	}

	marshalled_dataset_size+=sizeof(int); //objects size
	marshalled_dataset_size+=sizeof(int); //sorted size
	if(sorted){
		marshalled_dataset_size+=(int)objects.size()*sizeof(int); //sorted ids
		marshalled_dataset_size+=(int)objects.size()*sizeof(int); //id offsets
	}

	return marshalled_dataset_size;
}
/*}}}*/
/*FUNCTION DataSet::Demarshall{{{1*/
DataSet* DataSetDemarshall(char* marshalled_dataset){

	return DataSetDemarshallRaw(&marshalled_dataset);

}
/*}}}*/
/*FUNCTION DataSet::DemarshallRaw{{{1*/
DataSet* DataSetDemarshallRaw(char** pmarshalled_dataset){

	int i;

	DataSet* dataset=NULL;
	int      numobjects=0;
	int      enum_type;
	Object*  object=NULL;
	int      sorted;
	int*     sorted_ids=NULL;
	int*     id_offsets=NULL;
	char*    marshalled_dataset=NULL;

	/*recover marshalled_dataset pointer: */
	marshalled_dataset=*pmarshalled_dataset;

	/*initialize dataset: */
	dataset=new DataSet();

	/*Get internals first: */
	memcpy(&numobjects,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);
	memcpy(&sorted,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);
	if(sorted){
		if(numobjects){
			sorted_ids=(int*)xmalloc(numobjects*sizeof(int));
			id_offsets=(int*)xmalloc(numobjects*sizeof(int));
			memcpy(sorted_ids,marshalled_dataset,numobjects*sizeof(int)); marshalled_dataset+=numobjects*sizeof(int);
			memcpy(id_offsets,marshalled_dataset,numobjects*sizeof(int)); marshalled_dataset+=numobjects*sizeof(int);
		}
		dataset->SetSorting(sorted_ids,id_offsets);
	}

	for(i=0;i<numobjects;i++){

		/*get enum type of object: */
		memcpy(&enum_type,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);

		if(enum_type==NodeEnum){
			Node* node=NULL;
			node=new Node();
			node->Demarshall(&marshalled_dataset);
			dataset->AddObject(node);
		}
		else if(enum_type==VertexEnum){
			Vertex* vertex=NULL;
			vertex=new Vertex();
			vertex->Demarshall(&marshalled_dataset);
			dataset->AddObject(vertex);
		}
		else if(enum_type==DoubleParamEnum){
			DoubleParam* doubleparam=NULL;
			doubleparam=new DoubleParam();
			doubleparam->Demarshall(&marshalled_dataset);
			dataset->AddObject(doubleparam);
		}
		else if(enum_type==TriaEnum){
			Tria* tria=NULL;
			tria=new Tria();
			tria->Demarshall(&marshalled_dataset);
			dataset->AddObject(tria);
		}
		else if(enum_type==TriaVertexInputEnum){
			TriaVertexInput* triavertexinput=NULL;
			triavertexinput=new TriaVertexInput();
			triavertexinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(triavertexinput);
		}
		else if(enum_type==PentaVertexInputEnum){
			PentaVertexInput* pentavertexinput=NULL;
			pentavertexinput=new PentaVertexInput();
			pentavertexinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(pentavertexinput);
		}
		else if(enum_type==SingVertexInputEnum){
			SingVertexInput* singvertexinput=NULL;
			singvertexinput=new SingVertexInput();
			singvertexinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(singvertexinput);
		}
		else if(enum_type==BeamVertexInputEnum){
			BeamVertexInput* beamvertexinput=NULL;
			beamvertexinput=new BeamVertexInput();
			beamvertexinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(beamvertexinput);
		}
		else if(enum_type==TriaVertexElementResultEnum){
			TriaVertexElementResult* triavertexelementresult=NULL;
			triavertexelementresult=new TriaVertexElementResult();
			triavertexelementresult->Demarshall(&marshalled_dataset);
			dataset->AddObject(triavertexelementresult);
		}
		else if(enum_type==PentaVertexElementResultEnum){
			PentaVertexElementResult* pentavertexelementresult=NULL;
			pentavertexelementresult=new PentaVertexElementResult();
			pentavertexelementresult->Demarshall(&marshalled_dataset);
			dataset->AddObject(pentavertexelementresult);
		}
		else if(enum_type==SingEnum){
			Sing* sing=NULL;
			sing=new Sing();
			sing->Demarshall(&marshalled_dataset);
			dataset->AddObject(sing);
		}
		else if(enum_type==BeamEnum){
			Beam* beam=NULL;
			beam=new Beam();
			beam->Demarshall(&marshalled_dataset);
			dataset->AddObject(beam);
		}
		else if(enum_type==PentaEnum){
			Penta* penta=NULL;
			penta=new Penta();
			penta->Demarshall(&marshalled_dataset);
			dataset->AddObject(penta);
		}
		else if(enum_type==MaticeEnum){
			Matice* matice=NULL;
			matice=new Matice();
			matice->Demarshall(&marshalled_dataset);
			dataset->AddObject(matice);
		}
		else if(enum_type==MatparEnum){
			Matpar* matpar=NULL;
			matpar=new Matpar();
			matpar->Demarshall(&marshalled_dataset);
			dataset->AddObject(matpar);
		}
		else if(enum_type==SpcEnum){
			Spc* spc=NULL;
			spc=new Spc();
			spc->Demarshall(&marshalled_dataset);
			dataset->AddObject(spc);
		}
		else if(enum_type==PengridEnum){
			Pengrid* pengrid=NULL;
			pengrid=new Pengrid();
			pengrid->Demarshall(&marshalled_dataset);
			dataset->AddObject(pengrid);
		}
		else if(enum_type==PenpairEnum){
			Penpair* penpair=NULL;
			penpair=new Penpair();
			penpair->Demarshall(&marshalled_dataset);
			dataset->AddObject(penpair);
		}
		else if(enum_type==IcefrontEnum){
			Icefront* icefront=NULL;
			icefront=new Icefront();
			icefront->Demarshall(&marshalled_dataset);
			dataset->AddObject(icefront);
		}
		else if(enum_type==NumericalfluxEnum){
			Numericalflux* numericalflux=NULL;
			numericalflux=new Numericalflux();
			numericalflux->Demarshall(&marshalled_dataset);
			dataset->AddObject(numericalflux);
		}
		else if(enum_type==RgbEnum){
			Rgb* rgb=NULL;
			rgb=new Rgb();
			rgb->Demarshall(&marshalled_dataset);
			dataset->AddObject(rgb);
		}
		else if(enum_type==RiftfrontEnum){
			Riftfront* riftfront=NULL;
			riftfront=new Riftfront();
			riftfront->Demarshall(&marshalled_dataset);
			dataset->AddObject(riftfront);
		}
		else if(enum_type==DoubleInputEnum){
			DoubleInput* doubleinput=NULL;
			doubleinput=new DoubleInput();
			doubleinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(doubleinput);
		}
		else if(enum_type==IntInputEnum){
			IntInput* intinput=NULL;
			intinput=new IntInput();
			intinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(intinput);
		}
		else if(enum_type==BoolInputEnum){
			BoolInput* boolinput=NULL;
			boolinput=new BoolInput();
			boolinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(boolinput);
		}
		else if(enum_type==IntParamEnum){
			IntParam* intparam=NULL;
			intparam=new IntParam();
			intparam->Demarshall(&marshalled_dataset);
			dataset->AddObject(intparam);
		}
		else if(enum_type==BoolParamEnum){
			BoolParam* boolparam=NULL;
			boolparam=new BoolParam();
			boolparam->Demarshall(&marshalled_dataset);
			dataset->AddObject(boolparam);
		}
		else if(enum_type==StringParamEnum){
			StringParam* stringparam=NULL;
			stringparam=new StringParam();
			stringparam->Demarshall(&marshalled_dataset);
			dataset->AddObject(stringparam);
		}
		else{
			ISSMERROR("could not recognize enum type: %i (%s)",enum_type,EnumAsString(enum_type));
		}

	}

	/*Assign output pointers:*/
	*pmarshalled_dataset=marshalled_dataset;
	
	return dataset;
}
/*}}}*/

/*Specific methods*/
/*FUNCTION DataSet::AddObject{{{1*/
int  DataSet::AddObject(Object* object){

	objects.push_back(object);

	return 1;
}
/*}}}*/
/*FUNCTION DataSet::clear{{{1*/
void  DataSet::clear(){

	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		delete (*object);
	}
	objects.clear();
}
/*}}}*/
/*FUNCTION DataSet::DeleteObject{{{1*/
int  DataSet::DeleteObject(Object* object){

	vector<Object*>::iterator iterator;

	if(object){
		iterator = find(objects.begin(), objects.end(),object);
		delete *iterator;
		objects.erase(iterator);
	}

}
/*}}}*/
/*FUNCTION DataSet::DeepEcho{{{1*/
void DataSet::DeepEcho(){


	vector<Object*>::iterator object;

	if(this==NULL)ISSMERROR(" trying to echo a NULL dataset");

	_printf_("DataSet echo: %i objects\n",objects.size());

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call deep echo on object: */
		(*object)->DeepEcho();

	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::Echo{{{1*/
void DataSet::Echo(){


	vector<Object*>::iterator object;

	if(this==NULL)ISSMERROR(" trying to echo a NULL dataset");

	_printf_("DataSet echo: %i objects\n",objects.size());

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		(*object)->Echo();

	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::GetEnum{{{1*/
int  DataSet::GetEnum(){
	return enum_type;
}
/*}}}*/
/*FUNCTION DataSet::GetEnum(int offset){{{1*/
int   DataSet::GetEnum(int offset){

	return objects[offset]->Enum();

}
/*}}}*/
/*FUNCTION DataSet::GetObjectByOffset{{{1*/
Object* DataSet::GetObjectByOffset(int offset){

	return objects[offset];

}
/*}}}*/
/*FUNCTION DataSet::GetObjectById{{{1*/
Object* DataSet::GetObjectById(int* poffset,int eid){

	int id_offset;
	int offset;
	int i;

	if(!sorted)ISSMERROR(" trying to binary search on a non-sorted dataset!");

	/*Carry out a binary search on the sorted_ids: */
	if(!binary_search(&id_offset,eid, sorted_ids,objects.size())){
		ISSMERROR("could not find object with id %i in DataSet %s",eid,EnumAsString(enum_type));
	}

	/*Convert  the id offset into sorted offset: */
	offset=id_offsets[id_offset];

	/*Assign output pointers if requested:*/
	if (poffset)*poffset=offset;

	/*Return object at offset position in objects :*/
	return objects[offset];
}
/*}}}*/
/*FUNCTION DataSet::Presort{{{1*/
void DataSet::Presort(){

	/*vector of objects is already sorted, just allocate the sorted ids and their
	 * offsets:*/
	int i;

	if(objects.size()){

		/*Delete existing ids*/
		xfree((void**)&sorted_ids);
		xfree((void**)&id_offsets);

		/*Allocate new ids*/
		sorted_ids=(int*)xmalloc(objects.size()*sizeof(int));
		id_offsets=(int*)xmalloc(objects.size()*sizeof(int));

		/*Build id_offsets and sorted_ids*/
		for(i=0;i<objects.size();i++){
			id_offsets[i]=i;
			sorted_ids[i]=objects[i]->Id();
		}
	}

	/*set sorted flag: */
	sorted=1;
}
/*}}}*/
/*FUNCTION DataSet::SetSorting{{{1*/
void DataSet::SetSorting(int* in_sorted_ids,int* in_id_offsets){

	sorted=1;
	sorted_ids=in_sorted_ids;
	id_offsets=in_id_offsets;
}
/*}}}*/
/*FUNCTION DataSet::Size{{{1*/
int  DataSet::Size(void){

	return objects.size();
}
/*}}}*/
/*FUNCTION DataSet::Sort{{{1*/
void DataSet::Sort(){

	/*Only sort if we are not already sorted: */
	if(!sorted){
		ISSMERROR(" not implemented yet!");
	}
}
/*}}}*/
/*FUNCTION DataSet::Configure{{{1*/
void DataSet::Configure(Elements* elements,Loads* loads, Nodes* nodes, Vertices* vertices, Materials* materials,Parameters* parameters){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Configure(elements,loads,nodes,materials,parameters);
		}
		if(EnumIsLoad((*object)->Enum())){
			load=(Load*)(*object);
			load->Configure(elements,loads,nodes,vertices,materials,parameters);
		}

		if((*object)->Enum()==NodeEnum){
			node=(Node*)(*object);
			node->Configure(nodes,vertices);
		}
	}

}
/*}}}*/
