/*
 * \file Inputs.c
 * \brief: implementation of the Inputs class, derived from DataSet class
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Inputs::Inputs(){{{1*/
Inputs::Inputs(){
	return;
}
/*}}}*/
/*FUNCTION Inputs::Inputs(int in_enum){{{1*/
Inputs::Inputs(int in_enum): DataSet(in_enum) {
	//do nothing;
	return;
}
/*}}}*/
/*FUNCTION Inputs::~Inputs(){{{1*/
Inputs::~Inputs(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,gauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type,double defaultvalue){{{1*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type,double defaultvalue){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. Return the default value: */
		*pvalue=defaultvalue;
	}
	else{
		input->GetParameterValue(pvalue,gauss);
	}
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){{{1*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValues(values,gauss_pointers,numgauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue, Node* node, int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,Node* node,int enum_type){

	/*given a node, instead of a gauss point, we want to recover a value: probably in an element!: */

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,node);
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue, Node* node1, Node* node2,int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,Node* node1, Node* node2,double gauss_coord,int enum_type){

	/*given a node, instead of a gauss point, we want to recover a value: probably in an element!: */

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,node1,node2,gauss_coord);
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){{{1*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){

	int i;
	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. Return the default values: */
		for(i=0;i<numgauss;i++) values[i]=defaultvalues[i];
	}
	else{
		input->GetParameterValues(values,gauss_pointers,numgauss);
	}

}
/*}}}*/
/*FUNCTION Inputs::GetParameterAverage(double* pvalue,int enum-type){{{1*/
void Inputs::GetParameterAverage(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterAverage(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(bool* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(bool* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(int* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(int* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){{{1*/
void Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("could not find input with enum type %i (%s)",enum_type,EnumAsString(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterDerivativeValue(derivativevalues,xyz_list,gauss);
}
/*}}}*/
/*FUNCTION Inputs::GetStrainRate2d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum){{{1*/
void Inputs::GetStrainRate2d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum){
	/*Compute the 2d Strain Rate (3 components):
	 *
	 * epsilon=[exx eyy exy]
	 */

	vector<Object*>::iterator object;
	int i;
	Input* vxinput=NULL;
	Input* vyinput=NULL;
	double epsilonvx[3];
	double epsilonvy[3];
	bool   foundvx=false;
	bool   foundvy=false;

	/*Go through inputs and find data for vxenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vxinput=(Input*)(*object); 
		if (vxinput->EnumType()==vxenum){
			foundvx=true;
			break;
		}
	}
	/*Go through inputs and find data for vyenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vyinput=(Input*)(*object); 
		if (vyinput->EnumType()==vyenum){
			foundvy=true;
			break;
		}
	}

	/*Check that both inputs have been found*/
	if (!foundvx || !foundvy){
		ISSMERROR("Could not find input with enum %i (%s) or enum %i (%s)",vxenum,EnumAsString(vxenum),vyenum,EnumAsString(vyenum));
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vxinput->GetVxStrainRate2d(epsilonvx,xyz_list,gauss);
	vyinput->GetVyStrainRate2d(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<3;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Inputs::GetStrainRate3d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum, int vzenum){{{1*/
void Inputs::GetStrainRate3d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum,int vzenum){
	/*Compute the 3d Strain Rate (6 components):
	 *
	 * epsilon=[exx eyy ezz exy exz eyz]
	 */

	int    i;
	vector<Object*>::iterator object;
	Input* vxinput=NULL;
	Input* vyinput=NULL;
	Input* vzinput=NULL;
	bool   foundvx=false;
	bool   foundvy=false;
	bool   foundvz=false;
	double epsilonvx[6];
	double epsilonvy[6];
	double epsilonvz[6];

	/*Go through inputs and find data for vxenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vxinput=(Input*)(*object); 
		if (vxinput->EnumType()==vxenum){
			foundvx=true;
			break;
		}
	}
	/*Go through inputs and find data for vyenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vyinput=(Input*)(*object); 
		if (vyinput->EnumType()==vyenum){
			foundvy=true;
			break;
		}
	}
	/*Go through inputs and find data for vzenum, not for Pattyn*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vzinput=(Input*)(*object); 
		if (vzinput->EnumType()==vzenum){
			foundvz=true;
			break;
		}
	}

	/*Check that all inputs have been found*/
	if (!foundvx || !foundvy || !foundvz){
		ISSMERROR("Could not find input with enum %i (%s), enum %i (%s) or  enum %i (%s)",vxenum,EnumAsString(vxenum),vyenum,EnumAsString(vyenum),vzenum,EnumAsString(vzenum));
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vxinput->GetVxStrainRate3d(epsilonvx,xyz_list,gauss);
	vyinput->GetVyStrainRate3d(epsilonvy,xyz_list,gauss);
	vzinput->GetVzStrainRate3d(epsilonvz,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<6;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i]+epsilonvz[i];

}
/*}}}*/
/*FUNCTION Inputs::GetStrainRate3dPattyn(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum){{{1*/
void Inputs::GetStrainRate3dPattyn(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum){
	/*Compute the 3d Blatter/PattynStrain Rate (5 components):
	 *
	 * epsilon=[exx eyy exy exz eyz]
	 *
	 * with exz=1/2 du/dz
	 *      eyz=1/2 dv/dz
	 *
	 * the contribution of vz is neglected
	 */

	int    i;
	vector<Object*>::iterator object;
	Input* vxinput=NULL;
	Input* vyinput=NULL;
	bool   foundvx=false;
	bool   foundvy=false;
	double epsilonvx[5];
	double epsilonvy[5];

	/*Go through inputs and find data for vxenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vxinput=(Input*)(*object); 
		if (vxinput->EnumType()==vxenum){
			foundvx=true;
			break;
		}
	}
	/*Go through inputs and find data for vyenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vyinput=(Input*)(*object); 
		if (vyinput->EnumType()==vyenum){
			foundvy=true;
			break;
		}
	}

	/*Check that all inputs have been found*/
	if (!foundvx || !foundvy){
		ISSMERROR("Could not find input with enum %i (%s) or enum %i (%s)",vxenum,EnumAsString(vxenum),vyenum,EnumAsString(vyenum));
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vxinput->GetVxStrainRate3dPattyn(epsilonvx,xyz_list,gauss);
	vyinput->GetVyStrainRate3dPattyn(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<5;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Inputs::AddInput{{{1*/
int  Inputs::AddInput(Input* in_input){

	/*First, go through dataset of inputs and check whether any input 
	 * with the same name is already in. If so, erase the corresponding 
	 * object before adding this new one: */
	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==in_input->EnumType()){
			this->DeleteObject(input);
			break;
		}
	}
	this->AddObject(in_input);
}
/*}}}*/
/*FUNCTION Inputs::GetInput{{{1*/
Input* Inputs::GetInput(int enum_name){

	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==enum_name){
			return input;
		}
	}
	return NULL;
}
/*}}}*/
/*FUNCTION Inputs::ChangeEnum{{{1*/
void  Inputs::ChangeEnum(int oldenumtype,int newenumtype){

	/*Go through dataset of inputs and look for input with 
	 * same enum as input enum, once found, just change its name */
	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Delete existing input of newenumtype if it exists*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		input=(Input*)(*object); 

		if (input->EnumType()==newenumtype){
			this->DeleteObject(input);
			break;
		}
	}

	/*Change enum_type of input of oldenumtype*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==oldenumtype){
			input->ChangeEnum(newenumtype);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Inputs::SpawnBeamInputs{{{1*/
Inputs* Inputs::SpawnBeamInputs(int* indices){

	/*Intermediary*/
	vector<Object*>::iterator object;
	Input* inputin=NULL;
	Input* inputout=NULL;

	/*Output*/
	Inputs* newinputs=new Inputs();

	/*Go through inputs and call Spawn function*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Create new input*/
		inputin=(Input*)(*object); 
		inputout=inputin->SpawnBeamInput(indices);

		/*Add input to new inputs*/
		newinputs->AddObject(inputout);
	}

	/*Assign output pointer*/
	return newinputs;
}
/*}}}*/
/*FUNCTION Inputs::SpawnSingInputs{{{1*/
Inputs* Inputs::SpawnSingInputs(int index){

	/*Intermediary*/
	vector<Object*>::iterator object;
	Input* inputin=NULL;
	Input* inputout=NULL;

	/*Output*/
	Inputs* newinputs=new Inputs();

	/*Go through inputs and call Spawn function*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Create new input*/
		inputin=(Input*)(*object); 
		inputout=inputin->SpawnSingInput(index);

		/*Add input to new inputs*/
		newinputs->AddObject(inputout);
	}

	/*Assign output pointer*/
	return newinputs;
}
/*}}}*/
/*FUNCTION Inputs::SpawnTriaInputs{{{1*/
Inputs* Inputs::SpawnTriaInputs(int* indices){

	/*Intermediary*/
	vector<Object*>::iterator object;
	Input* inputin=NULL;
	Input* inputout=NULL;

	/*Output*/
	Inputs* newinputs=new Inputs();

	/*Go through inputs and call Spawn function*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Create new input*/
		inputin=(Input*)(*object); 
		inputout=inputin->SpawnTriaInput(indices);

		/*Add input to new inputs*/
		newinputs->AddObject(inputout);
	}

	/*Assign output pointer*/
	return newinputs;
}
/*}}}*/
