/*!\file Pengrid.c
 * \brief: implementation of the Pengrid object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../include/include.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
	
/*Object constructors and destructor*/
/*FUNCTION Pengrid::constructor {{{1*/
Pengrid::Pengrid(){
	this->inputs=NULL;
	this->parameters=NULL;
	
	/*not active, not zigzagging: */
	active=0;
	zigzag_counter=0;

}
/*}}}1*/
/*FUNCTION Pengrid::Pengrid(int index, int id, IoModel* iomodel,int analysis_type){{{1*/
Pengrid::Pengrid(int id, int index, IoModel* iomodel, int in_analysis_type){ //i is the element index

	int i,j;
	int pengrid_node_id;
	int pengrid_matpar_id;
	int pengrid_element_id;

	/*Some checks if debugging activated*/
	ISSMASSERT(iomodel->singlenodetoelementconnectivity);
	ISSMASSERT(index>=0 && index<iomodel->numberofvertices);

	/*id: */
	this->id=id;
	this->analysis_type=in_analysis_type;
	
	/*hooks: */
	pengrid_node_id=index+1;
	pengrid_element_id=iomodel->singlenodetoelementconnectivity[index];
	pengrid_matpar_id=iomodel->numberofelements+1; //refers to the constant material parameters object

	this->hnode.Init(&pengrid_node_id,1);
	this->helement.Init(&pengrid_element_id,1);
	this->hmatpar.Init(&pengrid_matpar_id,1);

	//initialize inputs: none needed
	this->inputs=new Inputs();

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	//let's not forget internals
	this->active=0;
	this->zigzag_counter=0;

}
/*}}}*/
/*FUNCTION Icefront::copy {{{1*/
Object* Pengrid::copy() {
	
	Pengrid* pengrid=NULL;

	pengrid=new Pengrid();

	/*copy fields: */
	pengrid->id=this->id;
	pengrid->analysis_type=this->analysis_type;
	if(this->inputs){
		pengrid->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		pengrid->inputs=new Inputs();
	}
	/*point parameters: */
	pengrid->parameters=this->parameters;

	/*now deal with hooks and objects: */
	pengrid->hnode.copy(&this->hnode);
	pengrid->helement.copy(&this->helement);
	pengrid->hmatpar.copy(&this->hmatpar);

	//let's not forget internals
	pengrid->active=this->active=0;
	pengrid->zigzag_counter=this->zigzag_counter=0;

	return pengrid;

}
/*}}}*/
/*FUNCTION Pengrid::destructor {{{1*/
Pengrid::~Pengrid(){
	return;
}
/*}}}1*/
		
/*Object management*/
/*FUNCTION Pengrid::Configure {{{1*/
void  Pengrid::Configure(Elements* elementsin,Loads* loadsin,DataSet* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this load, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnode.configure(nodesin);
	helement.configure(elementsin);
	hmatpar.configure(materialsin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}1*/
/*FUNCTION Pengrid::DeepEcho{{{1*/
void Pengrid::DeepEcho(void){

	printf("Pengrid:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumAsString(analysis_type));
	hnode.DeepEcho();
	helement.DeepEcho();
	hmatpar.DeepEcho();
	printf("   active %i\n",this->active);
	printf("   zigzag_counter %i\n",this->zigzag_counter);
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
}
/*}}}*/
/*FUNCTION Pengrid::Demarshall {{{1*/
void  Pengrid::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	memcpy(&active,marshalled_dataset,sizeof(active));marshalled_dataset+=sizeof(active);
	memcpy(&zigzag_counter,marshalled_dataset,sizeof(zigzag_counter));marshalled_dataset+=sizeof(zigzag_counter);

	/*demarshall hooks: */
	hnode.Demarshall(&marshalled_dataset);
	helement.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Pengrid::Echo {{{1*/
void Pengrid::Echo(void){
	this->DeepEcho();
}
/*}}}1*/
/*FUNCTION Pengrid::Enum {{{1*/
int Pengrid::Enum(void){

	return PengridEnum;
}
/*}}}1*/
/*FUNCTION Pengrid::Id {{{1*/
int    Pengrid::Id(void){ return id; }
/*}}}1*/
/*FUNCTION Pengrid::Marshall {{{1*/
void  Pengrid::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Tria: */
	enum_type=PengridEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Tria data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	memcpy(marshalled_dataset,&active,sizeof(active));marshalled_dataset+=sizeof(active);
	memcpy(marshalled_dataset,&zigzag_counter,sizeof(zigzag_counter));marshalled_dataset+=sizeof(zigzag_counter);

	/*Marshall hooks: */
	hnode.Marshall(&marshalled_dataset);
	helement.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Pengrid::MarshallSize {{{1*/
int   Pengrid::MarshallSize(){
	
	return sizeof(id)
		+sizeof(analysis_type)
		+sizeof(active)
		+sizeof(zigzag_counter)
		+hnode.MarshallSize()
		+helement.MarshallSize()
		+hmatpar.MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Pengrid::MyRank {{{1*/
int    Pengrid::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}1*/

/*Object functions*/
/*FUNCTION Pengrid::CreateKMatrix {{{1*/
void  Pengrid::CreateKMatrix(Mat Kgg){

	/*No loads applied, do nothing: */
	return;

}
/*}}}1*/
/*FUNCTION Pengrid::CreatePVector {{{1*/
void  Pengrid::CreatePVector(Vec pg){

	/*No loads applied, do nothing: */
	return;

}
/*}}}1*/
/*FUNCTION Pengrid::GetDofList {{{1*/
void  Pengrid::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	/*dynamic objects pointed to by hooks: */
	Node* node=NULL;

	/*recover objects from hooks: */
	node=(Node*)hnode.delivers();
	
	node->GetDofList(&doflist_per_node[0],&numberofdofspernode);
	for(j=0;j<numberofdofspernode;j++){
		doflist[j]=doflist_per_node[j];
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Pengrid::PenaltyConstrain {{{1*/
void  Pengrid::PenaltyConstrain(int* punstable){

	int analysis_type,sub_analysis_type;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);

	if ((analysis_type==DiagnosticAnalysisEnum) && ((sub_analysis_type==StokesAnalysisEnum))){

		/*No penalty to check*/
		return;

	}
	else if (analysis_type==ThermalAnalysisEnum){
		
		PenaltyConstrainThermal(punstable);
		
	}
	else if (analysis_type==MeltingAnalysisEnum){
			
		/*No penalty to check*/
		return;

	}
	else{
		ISSMERROR("%s%i%s%i%s","analysis: ",analysis_type," and sub_analysis_type: ",sub_analysis_type," not supported yet");
	}

}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyConstrainThermal {{{1*/
void  Pengrid::PenaltyConstrainThermal(int* punstable){

	//   The penalty is stable if it doesn't change during to successive iterations.   

	int    found=0;
	const int numgrids=1;
	double pressure;
	double temperature;
	double beta,t_pmp;
	double meltingpoint;
	int    new_active;
	int    unstable=0;
	int    reset_penalties=0;
	int    stabilize_constraints;
	double* gauss=NULL;

	/*pointers: */
	Node* node=NULL;
	Penta* penta=NULL;
	Matpar*  matpar=NULL;

	/*recover pointers: */
	node=(Node*)hnode.delivers();
	penta=(Penta*)helement.delivers();
	matpar=(Matpar*)hmatpar.delivers();
	
	/*check that pengrid is not a clone (penalty to be added only once)*/
	if (node->IsClone()){
		unstable=0;
		*punstable=unstable;
		return;
	}

	//First recover pressure and temperature values, using the element: */
	gauss=penta->GaussFromNode(node);
	penta->inputs->GetParameterValue(&pressure,gauss,PressureEnum);
	penta->inputs->GetParameterValue(&temperature,gauss,TemperatureEnum);

	//Recover our data:
	meltingpoint=matpar->GetMeltingPoint();
	beta=matpar->GetBeta();
	parameters->FindParam(&stabilize_constraints,StabilizeConstraintsEnum);
	
	//Compute pressure melting point
	t_pmp=meltingpoint-beta*pressure;

	//Figure out if temperature is over melting_point, in which case, this penalty needs to be activated.

	if (temperature>t_pmp){
		new_active=1;
	}
	else{
		new_active=0;
	}


	//Figure out stability of this penalty
	if (active==new_active){
		unstable=0;
	}
	else{
		unstable=1;
		if(stabilize_constraints)zigzag_counter++;
	}

	/*If penalty keeps zigzagging more than 5 times: */
	if(stabilize_constraints){
		if(zigzag_counter>stabilize_constraints){
			unstable=0;
			active=1;
		}
	}

	//Set penalty flag
	active=new_active;

	//*Assign output pointers:*/
	*punstable=unstable;

	/*Clean up*/
	xfree((void**)&gauss);
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateMatrix {{{1*/
void  Pengrid::PenaltyCreateKMatrix(Mat Kgg,double kmax){

	int analysis_type;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	if (analysis_type==DiagnosticStokesAnalysisEnum){
		PenaltyCreateKMatrixDiagnosticStokes( Kgg,kmax);
	}
	else if (analysis_type==ThermalAnalysisEnum){
		PenaltyCreateKMatrixThermal( Kgg,kmax);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		PenaltyCreateKMatrixMelting( Kgg,kmax);
	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}

}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateKMatrixDiagnosticStokes {{{1*/
void  Pengrid::PenaltyCreateKMatrixDiagnosticStokes(Mat Kgg,double kmax){
	
	const int numgrids=1;
	const int NDOF4=4;
	const int numdof=numgrids*NDOF4;
	int       doflist[numdof];
	int       numberofdofspernode;

	int dofs1[1]={0};
	int dofs2[1]={1};
	double slope[2];
	int found=0;
	double Ke[4][4]={0.0};
	double penalty_offset;

	/*pointers: */
	Node* node=NULL;
	Penta* penta=NULL;

	/*Get dof list: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover pointers: */
	node=(Node*)hnode.delivers();
	penta=(Penta*)helement.delivers();

	//recover slope: */
	penta->inputs->GetParameterValue(&slope[0],node,BedSlopeXEnum);
	penta->inputs->GetParameterValue(&slope[1],node,BedSlopeYEnum);
	
	/*recover parameters: */
	parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);

	//Create elementary matrix: add penalty to constrain wb (wb=ub*db/dx+vb*db/dy)
	Ke[2][0]=-slope[0]*kmax*pow((double)10.0,penalty_offset);
	Ke[2][1]=-slope[1]*kmax*pow((double)10.0,penalty_offset);
	Ke[2][2]=kmax*pow((double)10,penalty_offset);
	
	/*Add Ke to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke,ADD_VALUES);
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateKMatrixMelting {{{1*/
void  Pengrid::PenaltyCreateKMatrixMelting(Mat Kgg,double kmax){

	int found=0;
	const int numgrids=1;
	const int NDOF1=1;
	const int numdof=numgrids*NDOF1;
	double Ke[numdof][numdof]={0.0};
	int     dofs1[1]={0};
	int       doflist[numdof];
	int      numberofdofspernode;
	double  meltingpoint;
	double* gauss=NULL;

	double pressure;
	double temperature;
	double beta,t_pmp;
	double penalty_offset;

	/*pointers: */
	Node* node=NULL;
	Penta* penta=NULL;
	Matpar*  matpar=NULL;

	/*recover pointers: */
	node=(Node*)hnode.delivers();
	penta=(Penta*)helement.delivers();
	matpar=(Matpar*)hmatpar.delivers();

	/*check that pengrid is not a clone (penalty to be added only once)*/
	if (node->IsClone()) return;

	//First recover pressure and temperature values, using the element: */
	gauss=penta->GaussFromNode(node);
	penta->inputs->GetParameterValue(&pressure,gauss,PressureEnum);
	penta->inputs->GetParameterValue(&temperature,gauss,TemperatureEnum);

	/*recover parameters: */
	parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);

	/*Get dof list: */
	GetDofList(&doflist[0],&numberofdofspernode);
	
	//Compute pressure melting point
	meltingpoint=matpar->GetMeltingPoint();
	beta=matpar->GetBeta();
	t_pmp=meltingpoint-beta*pressure;

	//Add penalty load
	if (temperature<t_pmp){ //If T<Tpmp, there must be no melting. Therefore, melting should be  constrained to 0 when T<Tpmp, instead of using spcs, use penalties
		Ke[0][0]=kmax*pow((double)10,penalty_offset);
	}
	
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke,ADD_VALUES);

	/*Clean up*/
	xfree((void**)&gauss);
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateKMatrixThermal {{{1*/
void  Pengrid::PenaltyCreateKMatrixThermal(Mat Kgg,double kmax){

	int found=0;
	
	const int numgrids=1;
	const int NDOF1=1;
	const int numdof=numgrids*NDOF1;
	double Ke[numdof][numdof];
	int       doflist[numdof];
	int       numberofdofspernode;
	double    penalty_offset;

	if(!this->active)return;

	/*recover parameters: */
	parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);

	/*Get dof list: */
	GetDofList(&doflist[0],&numberofdofspernode);

	Ke[0][0]=kmax*pow((double)10,penalty_offset);
	
	/*Add Ke to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke,ADD_VALUES);
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreatePVector {{{1*/
void  Pengrid::PenaltyCreatePVector(Vec pg,double kmax){

	int analysis_type;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	if (analysis_type==ThermalAnalysisEnum){
		PenaltyCreatePVectorThermal( pg,kmax);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		PenaltyCreatePVectorMelting( pg,kmax);
	}
	else if (analysis_type==DiagnosticStokesAnalysisEnum){

		/*No loads applied, do nothing: */
		return;

	}
	else{
		ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumAsString(analysis_type));
	}

}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreatePVectorMelting {{{1*/
void  Pengrid::PenaltyCreatePVectorMelting(Vec pg, double kmax){
	
	const int numgrids=1;
	const int NDOF1=1;
	const int numdof=numgrids*NDOF1;
	int    doflist[numdof];
	double P_terms[numdof]={0.0};
	int    numberofdofspernode;
	int    found=0;
	int    dofs1[1]={0};
	double pressure;
	double temperature;
	double melting_offset;
	double meltingpoint;
	double beta, heatcapacity;
	double latentheat;
	double t_pmp;
	double dt,penalty_offset;
	double* gauss=NULL;

	/*pointers: */
	Node* node=NULL;
	Penta* penta=NULL;
	Matpar*  matpar=NULL;

	/*recover pointers: */
	node=(Node*)hnode.delivers();
	penta=(Penta*)helement.delivers();
	matpar=(Matpar*)hmatpar.delivers();

	/*check that pengrid is not a clone (penalty to be added only once)*/
	if (node->IsClone()) return;

	/*Get dof list: */
	GetDofList(&doflist[0],&numberofdofspernode);

	//First recover pressure and temperature values, using the element: */
	gauss=penta->GaussFromNode(node);
	penta->inputs->GetParameterValue(&pressure,gauss,PressureEnum);
	penta->inputs->GetParameterValue(&temperature,gauss,TemperatureEnum);
	inputs->GetParameterValue(&melting_offset,MeltingOffsetEnum);
	parameters->FindParam(&dt,DtEnum);
	parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);

	meltingpoint=matpar->GetMeltingPoint();
	beta=matpar->GetBeta();
	heatcapacity=matpar->GetHeatCapacity();
	latentheat=matpar->GetLatentHeat();

	//Compute pressure melting point
	t_pmp=meltingpoint-beta*pressure;

	//Add penalty load
	//This time, the penalty must have the same value as the one used for the thermal computation
	//so that the corresponding melting can be computed correctly
	//In the thermal computation, we used kmax=melting_offset, and the same penalty_offset
	if (temperature<t_pmp){ //%no melting
		P_terms[0]=0;
	}
	else{
		if (dt){
			P_terms[0]=melting_offset*pow((double)10,penalty_offset)*(temperature-t_pmp)/dt;
		}
		else{
			P_terms[0]=melting_offset*pow((double)10,penalty_offset)*(temperature-t_pmp);
		}
	}

	/*Add P_terms to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/*Clean up*/
	xfree((void**)&gauss);
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreatePVectorThermal {{{1*/
void  Pengrid::PenaltyCreatePVectorThermal(Vec pg,  double kmax){

	const int numgrids=1;
	const int NDOF1=1;
	const int numdof=numgrids*NDOF1;
	int       doflist[numdof];
	double  P_terms[numdof]={0.0};
	int    numberofdofspernode;
	int    found=0;
	double pressure;
	int dofs1[1]={0};
	double meltingpoint;
	double beta;
	double t_pmp;
	double penalty_offset;
	double* gauss=NULL;

	/*pointers: */
	Node* node=NULL;
	Penta* penta=NULL;
	Matpar*  matpar=NULL;

	/*recover pointers: */
	node=(Node*)hnode.delivers();
	penta=(Penta*)helement.delivers();
	matpar=(Matpar*)hmatpar.delivers();

	if(!this->active)return;

	/*Get dof list: */
	GetDofList(&doflist[0],&numberofdofspernode);

	//First recover pressure  and penalty_offset
	gauss=penta->GaussFromNode(node);
	penta->inputs->GetParameterValue(&pressure,gauss,PressureEnum);
	parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);

	//Compute pressure melting point
	meltingpoint=matpar->GetMeltingPoint();
	beta=matpar->GetBeta();
	t_pmp=meltingpoint-beta*pressure;

	//Add penalty load
	P_terms[0]=kmax*pow((double)10,penalty_offset)*t_pmp;

	/*Add P_terms to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/*Clean up*/
	xfree((void**)&gauss);
}
/*}}}1*/

/*Updates: */
/*FUNCTION Pengrid::UpdateFromDakota {{{1*/
void  Pengrid::UpdateFromDakota(void* inputs){
	ISSMERROR("not supported yet!");
}
/*}}}1*/
/*FUNCTION Pengrid::UpdateInputs {{{1*/
void  Pengrid::UpdateInputs(double* solution){
	ISSMERROR("not supported yet!");
}
/*}}}1*/
/*FUNCTION Pengrid::InputUpdateFromVector(double* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVector(double* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVector(int* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVector(bool* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromConstant(double constant, int name) {{{1*/
void  Pengrid::InputUpdateFromConstant(double constant, int name){
	switch(name){

		case MeltingOffsetEnum:
			inputs->AddInput(new DoubleInput(name,constant));
			return;

	}
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromConstant(int constant, int name) {{{1*/
void  Pengrid::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromConstant(bool constant, int name) {{{1*/
void  Pengrid::InputUpdateFromConstant(bool constant, int name){

	switch(name){

		case ResetPenaltiesEnum:
			if (constant) zigzag_counter=0;
			return;

	}
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromSolution(double* solution) {{{1*/
void  Pengrid::InputUpdateFromSolution(double* solution){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InAnalysis(int analysis_type){{{1*/
bool Pengrid::InAnalysis(int in_analysis_type){
	if (in_analysis_type=this->analysis_type)return true;
	else return false;
}
/*}}}*/
