function bool=isresultconsistent(md,analysis_type)
%ISRESULTCONSISTENT: check that results are consistent
%
%   This routine is called after a computation and check
%   that the values t the penalties are consitent
%
%   Usage:
%      bool=IsModelSelfConsistent(model,analysis_type)

%tolerance we use in litmus tests for the consistency of the model
tolerance=10^-2;

%some checks
if (nargin~=2 | nargout~=1)
	help isresultconsistent
	error(' ');
end

%initialize output as 1 -> success
bool=1;

%do not check results if qmu analysis
if md.qmu_analysis,
	return
end

%DIAGNOSTIC
if analysis_type==DiagnosticSolutionEnum() & md.control_analysis==0,

	if (md.dim==3)
		fields1={'results.DiagnosticAnalysis.vx','results.DiagnosticAnalysis.vy','results.DiagnosticAnalysis.vz','results.DiagnosticAnalysis.vel'};
		fields2={'results.DiagnosticAnalysis.vel'};
	else
		fields1={'results.DiagnosticAnalysis.vx','results.DiagnosticAnalysis.vy','results.DiagnosticAnalysis.vel'};
		fields2={'results.DiagnosticAnalysis.vel'};
	end

	%check size
	if ~testsize(md,fields1,md.numberofgrids),
		bool=0; return
	end

	%no NAN
	if ~testnan(md,fields1),
		bool=0; return
	end

	%check value is real
	if ~testreal(md,fields1),
		bool=0; return
	end

	%check value>=0
	if ~testpositive(md,fields2),
		bool=0; return
	end

end

%CONTROL
if analysis_type==DiagnosticSolutionEnum() & md.control_analysis==1,

	if ~md.cm_gradient,
		fields1={'results.DiagnosticAnalysis.vx','results.DiagnosticAnalysis.vy','results.DiagnosticAnalysis.vel','results.DiagnosticAnalysis.parameter'};
		fields2={'results.DiagnosticAnalysis.vel','results.DiagnosticAnalysis.J'};

		%check size
		if ~testsize(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%no NAN
		if ~testnan(md,fields1),
			bool=0; return
		end

		%check value is real
		if ~testreal(md,fields1),
			bool=0; return
		end

		%check value>=0
		if ~testpositive(md,fields2),
			bool=0; return
		end

		%check inversed parameter
		if ~isnan(md.cm_max),
			if any(md.results.DiagnosticAnalysis.parameter>md.cm_max)
				disp(['''control'' result not consistent: inverse parameter is greater than ' num2str(md.cm_max)]);
				bool=0; return; 
			end
		end
		if ~isnan(md.cm_min),
			if any(md.results.DiagnosticAnalysis.parameter<md.cm_min)
				disp(['''control'' result not consistent: inverse parameter is smaller than ' num2str(md.cm_min)]);
				bool=0; return; 
			end
		end
	else
		fields={'results.DiagnosticAnalysis.gradient'};
		%check size
		if ~testsize(md,fields,md.numberofgrids),
			bool=0; return
		end

		%no NAN
		if ~testnan(md,fields),
			bool=0; return
		end

		%check value is real
		if ~testreal(md,fields),
			bool=0; return
		end
	end
end

%THERMAL
if analysis_type==ThermalSolutionEnum(),

	for iter=1:length(md.results.ThermalAnalysis)

		fields1={['results.ThermalAnalysis(' num2str(iter) ').temperature'],['results.ThermalAnalysis(' num2str(iter) ').melting']};
		fields2={['results.ThermalAnalysis(' num2str(iter) ').temperature']};

		%check size
		if ~testsize(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%no NAN
		if ~testnan(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value is real
		if ~testreal(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value>=0
		if ~testpositive(md,fields2,md.numberofgrids),
			bool=0; return
		end

		%check melting (<=0 via penalties)
		if any(abs(md.results.ThermalAnalysis(iter).melting(md.numberofgrids2d+1:end))>tolerance)
			disp(['''thermal'' result not consistent: increase penalty_melting (negative melting)']);
			bool=0; return; 
		end
	end

end

if (analysis_type==Transient2DSolutionEnum() | analysis_type==Transient3DSolutionEnum()),

	for iter=1:length(md.results.TransientAnalysis)

		if (md.dim==3),
			fields1={['results.TransientAnalysis(' num2str(iter) ').vx'],['results.TransientAnalysis(' num2str(iter) ').vy'],...
				['results.TransientAnalysis(' num2str(iter) ').vz'],['results.TransientAnalysis(' num2str(iter) ').vel'],...
				['results.TransientAnalysis(' num2str(iter) ').bed'],['results.TransientAnalysis(' num2str(iter) ').surface'],...
				['results.TransientAnalysis(' num2str(iter) ').thickness'],...
				['results.TransientAnalysis(' num2str(iter) ').temperature'],['results.TransientAnalysis(' num2str(iter) ').melting']};
			fields2={['results.TransientAnalysis(' num2str(iter) ').vel'],['results.TransientAnalysis(' num2str(iter) ').thickness'],...
				['results.TransientAnalysis(' num2str(iter) ').temperature']};
		else
			fields1={['results.TransientAnalysis(' num2str(iter) ').vx'],['results.TransientAnalysis(' num2str(iter) ').vy'],...
				['results.TransientAnalysis(' num2str(iter) ').vel'],...
				['results.TransientAnalysis(' num2str(iter) ').bed'],['results.TransientAnalysis(' num2str(iter) ').surface'],...
				['results.TransientAnalysis(' num2str(iter) ').thickness']};
			fields2={['results.TransientAnalysis(' num2str(iter) ').vel'],['results.TransientAnalysis(' num2str(iter) ').thickness']};
		end

		%check size
		if ~testsize(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%no NAN
		if ~testnan(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value is real
		if ~testreal(md,fields1,md.numberofgrids),
			bool=0; return
		end

		%check value>=0
		if ~testpositive(md,fields2,md.numberofgrids),
			bool=0; return
		end

		%check melting (<=0 via penalties)
		if (md.dim==3),
			if any(abs(md.results.TransientAnalysis(iter).melting(md.numberofgrids2d+1:end))>tolerance)
				disp(['''thermal'' result not consistent: increase penalty_melting (negative melting)']);
				bool=0; return; 
			end
		end
	end
end
end % end function

function bool=testsize(md,fields,fieldsize)
	%TESTSIZE - test size of a field
	bool=1;
	for i=1:length(fields),
		if length(eval(['md.' fields{i}]))~=fieldsize
			disp(['results not consistent: field ' fields{i} ' size should be ' num2str(md.numberofgrids)]);
			bool=0; return;
		end
	end
end

function bool=testnan(md,fields,nanfield)
	%TESTNAN - test nan values of a field
	bool=1;
	for i=1:length(fields),
		if any(isnan(eval(['md.' fields{i}]))),
			disp(['results not consistent: NaN values in field ' fields{i}]);
			bool=0; return;
		end
	end
end

function bool=testreal(md,fields,realfield)
	%TESTREAL - test real values of a field
	bool=1;
	for i=1:length(fields),
		if any(eval(['~isreal(md.' fields{i} ')'])),
			disp(['results not consistent: complex values in field ' fields{i}]);
			bool=0; return;
		end
	end
end

function bool=testpositive(md,fields,positivefield)
	%TESTPOSITIVE - test positive values of a field
	bool=1;
	for i=1:length(fields),
		if any(eval(['md.' fields{i} '<0'])),
			disp(['results not consistent: negative values in field ' fields{i}]);
			bool=0; return;
		end
	end
end
