/*!\file:  SpawnCoreParallel.cpp
 * \brief: run core ISSM solution using Dakota inputs coming from CPU 0.
 * \sa qmu.cpp DakotaPlugin.cpp
 *
 * This routine needs to be understood simultaneously with qmu.cpp and DakotaPlugin. 
 * SpawnCoreParallel is called by all CPUS, with CPU 0 holding Dakota variable values, along 
 * with variable descriptors. 
 *
 * SpawnCoreParallel takes care of broadcasting the variables and their descriptors across the MPI 
 * ring. Once this is done, we use the variables to modify the inputs for the solution core. 
 * For ex, if "rho_ice" is provided, for ex 920, we include "rho_ice" in the inputs, then 
 * call the core with the modified inputs. This is the way we get Dakota to explore the parameter 
 * spce of the core. 
 *
 * Once the core is called, we process the results of the core, and using the processed results, 
 * we compute response functions. The responses are computed on all CPUS, but they are targeted 
 * for CPU 0, which will get these values back to the Dakota engine. 
 *
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


#include "../../objects/objects.h"
#include "../../io/io.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "./Qmux.h"
#include "../../include/include.h"
#include "../../solutions/solutions.h"
#include "../modules.h"

void SpawnCoreParallel(double* responses, int numresponses, double* variables, char** variables_descriptors,int numvariables, FemModel* femmodel,int counter){

	int i;
	extern int my_rank;
	
	char   **responses_descriptors     = NULL;
	char    *string                    = NULL;
	int      string_length;
	double  *qmu_part                  = NULL;
	int      qmu_npart;
	int      verbose                   = 0;
	int      dummy;
	int      solution_type;

	/*synchronize all cpus, as CPU 0 is probably late (it is starting the entire dakota strategy!) : */
	MPI_Barrier(MPI_COMM_WORLD);
	_printf_("qmu iteration: %i\n",counter);
	
	/*retrieve parameters: */
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&responses_descriptors,&dummy,ResponseDescriptorsEnum);
	if(dummy!=numresponses)ISSMERROR(" parameter numresponses not equal to Dakota provided numresponses: %i vs %i\n",dummy,numresponses);

	/* only cpu 0, running dakota is providing us with variables, variables_descriptors and responses. 
	 * broadcast onto other cpus: */
	DakotaMPI_Bcast(&variables,&variables_descriptors,&numvariables,&numresponses);

	/*Modify core inputs in objects contained in femmodel, to reflect the dakota variables inputs: */
	InputUpdateFromDakotax(femmodel,variables,variables_descriptors,numvariables);

	/*Run the analysis core solution sequence: */
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	if(verbose)_printf_("%s%s%s\n","Starting ",EnumAsString(solution_type)," core:");
	switch(solution_type){
		case DiagnosticAnalysisEnum:
			diagnostic_core(femmodel);
			break;
		case ThermalAnalysisEnum:
			thermal_core(femmodel);
			break;
		case PrognosticAnalysisEnum:
			prognostic_core(femmodel);
			break;
		case Transient2DAnalysisEnum:
			transient2d_core(femmodel);
			break;
		case Transient3DAnalysisEnum:
			transient3d_core(femmodel);
		default:
			ISSMERROR("%s%s%s"," solution_type: ",EnumAsString(solution_type),", not supported yet!");
			break;
	}

	/*compute responses on cpu 0: dummy for now! */
	if(verbose)_printf_("compute dakota responses:\n");
	DakotaResponses(responses,responses_descriptors,numresponses,femmodel);

	/*Free ressources:{{{1*/
	//variables only on cpu != 0
	if(my_rank!=0){
		xfree((void**)&variables);
		for(i=0;i<numvariables;i++){
			string=variables_descriptors[i];
			xfree((void**)&string);
		}
		xfree((void**)&variables_descriptors);
	}
	//responses descriptors
	for(i=0;i<numresponses;i++){
		string=responses_descriptors[i];
		xfree((void**)&string);
	}
	//rest of dynamic allocations.
	xfree((void**)&responses_descriptors);
	xfree((void**)&qmu_part);
	/*}}}*/
}

