/*!\file Numericalflux.c
 * \brief: implementation of the Numericalflux object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../objects.h"

extern int my_rank;

/*Object constructors and destructor*/
/*FUNCTION Numericalflux::Numericalflux(){{{1*/
Numericalflux::Numericalflux(){
	this->inputs=NULL;
	this->parameters=NULL;
}
/*}}}*/
/*}}}*//*FUNCTION Numericalflux::Numericalflux(int id, int i, IoModel* iomodel, int analysis_type) {{{1*/
Numericalflux::Numericalflux(int numericalflux_id,int i, IoModel* iomodel, int in_analysis_type){

	/* Intermediary */
	int  e1,e2;
	int  i1,i2;
	int  j;
	int  pos1,pos2;
	int  num_nodes;
	int  num_elems;

	/*numericalflux constructor data: */
	int   numericalflux_elem_ids[2];
	int   numericalflux_mparid;
	int   numericalflux_node_ids[4];
	int   numericalflux_type;

	/* Get MatPar id */
	numericalflux_mparid=iomodel->numberofelements+1; //matlab indexing

	/*First, see wether this is an internal or boundary edge (if e2=NaN)*/
	if (isnan((double)iomodel->edges[4*i+3])){ //edges are [node1 node2 elem1 elem2]
		/* Boundary edge, only one element */
		e1=(int)iomodel->edges[4*i+2];
		e2=(int)UNDEF;
		num_elems=1;
		num_nodes=2;
		numericalflux_type=BoundaryEnum;
		numericalflux_elem_ids[0]=(int)e1;
	}
	else{
		/* internal edge: connected to 2 elements */
		e1=(int)iomodel->edges[4*i+2];
		e2=(int)iomodel->edges[4*i+3];
		num_elems=2;
		num_nodes=4;
		numericalflux_type=InternalEnum;
		numericalflux_elem_ids[0]=(int)e1;
		numericalflux_elem_ids[1]=(int)e2;
	}

	/*1: Get vertices ids*/
	i1=(int)iomodel->edges[4*i+0];
	i2=(int)iomodel->edges[4*i+1];

	if (numericalflux_type==InternalEnum){

		/*Now, we must get the nodes of the 4 nodes located on the edge*/

		/*2: Get the column where these ids are located in the index*/
		pos1=pos2=UNDEF;
		for(j=0;j<3;j++){
			if (iomodel->elements[3*(e1-1)+j]==i1) pos1=j+1;
			if (iomodel->elements[3*(e2-1)+j]==i1) pos2=j+1;
		}
		ISSMASSERT(pos1!=UNDEF && pos2!=UNDEF);

		/*3: We have the id of the elements and the position of the vertices in the index
		 * we can compute their dofs!*/
		numericalflux_node_ids[0]=3*(e1-1)+pos1;       //ex: 1 2 3
		numericalflux_node_ids[1]=3*(e1-1)+(pos1%3)+1; //ex: 2 3 1
		numericalflux_node_ids[2]=3*(e2-1)+pos2;           //ex: 1 2 3
		numericalflux_node_ids[3]=3*(e2-1)+((pos2+1)%3)+1; //ex: 3 1 2
	}
	else{

		/*2: Get the column where these ids are located in the index*/
		pos1==UNDEF;
		for(j=0;j<3;j++){
			if (iomodel->elements[3*(e1-1)+j]==i1) pos1=j+1;
		}
		ISSMASSERT(pos1!=UNDEF);

		/*3: We have the id of the elements and the position of the vertices in the index
		 * we can compute their dofs!*/
		numericalflux_node_ids[0]=3*(e1-1)+pos1;
		numericalflux_node_ids[1]=3*(e1-1)+(pos1%3)+1;
	}

	/*Ok, we have everything to build the object: */
	this->id=numericalflux_id;
	this->analysis_type=in_analysis_type;

	/*Hooks: */
	this->hnodes.Init(numericalflux_node_ids,num_nodes);
	this->helements.Init(numericalflux_elem_ids,num_elems);

	//intialize  and add as many inputs per element as requested: 
	this->inputs=new Inputs();
	this->inputs->AddInput(new IntInput(TypeEnum,numericalflux_type));

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Numericalflux::copy {{{1*/
Object* Numericalflux::copy() {
	
	Numericalflux* numericalflux=NULL;

	numericalflux=new Numericalflux();

	/*copy fields: */
	numericalflux->id=this->id;
	numericalflux->analysis_type=this->analysis_type;
	if(this->inputs){
		numericalflux->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		numericalflux->inputs=new Inputs();
	}
	/*point parameters: */
	numericalflux->parameters=this->parameters;

	/*now deal with hooks and objects: */
	numericalflux->hnodes.copy(&this->hnodes);
	numericalflux->helements.copy(&this->helements);

	return numericalflux;

}
/*}}}*/
/*FUNCTION Numericalflux::~Numericalflux(){{{1*/
Numericalflux::~Numericalflux(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object marshall*/
/*FUNCTION Numericalflux::Configure {{{1*/
void  Numericalflux::Configure(DataSet* elementsin,DataSet* loadsin,DataSet* nodesin,DataSet* verticesin,DataSet* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	helements.configure(elementsin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Numericalflux::DeepEcho {{{1*/
void Numericalflux::DeepEcho(void){

	printf("Numericalflux:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumAsString(analysis_type));
	hnodes.DeepEcho();
	helements.DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
	
}		
/*}}}*/
/*FUNCTION Numericalflux::Demarshall {{{1*/
void  Numericalflux::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	helements.Demarshall(&marshalled_dataset);
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Echo {{{1*/
void Numericalflux::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION Numericalflux::Enum {{{1*/
int Numericalflux::Enum(void){

	return NumericalfluxEnum;

}
/*}}}*/
/*FUNCTION Numericalflux::Marshall {{{1*/
void  Numericalflux::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Numericalflux: */
	enum_type=NumericalfluxEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Numericalflux data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);

	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	helements.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::MarshallSize{{{1*/
int   Numericalflux::MarshallSize(){

	return sizeof(id)
		+sizeof(analysis_type)
		+hnodes.MarshallSize()
		+helements.MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Numericalflux::MyRank {{{1*/
int    Numericalflux::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Object functions*/
/*FUNCTION Numericalflux::CreateKMatrix {{{1*/
void  Numericalflux::CreateKMatrix(Mat Kgg){

	int type;

	/*recover type: */
	inputs->GetParameterValue(&type,TypeEnum);

	if (type==InternalEnum){
		CreateKMatrixInternal(Kgg);
	}
	else if (type==BoundaryEnum){
		CreateKMatrixBoundary(Kgg);
	}
	else ISSMERROR("type not supported yet");

}
/*}}}*/
/*FUNCTION Numericalflux::CreateKMatrixInternal {{{1*/
void  Numericalflux::CreateKMatrixInternal(Mat Kgg){

	/* local declarations */
	int             i,j;
	int analysis_type,sub_analysis_type;

	/* node data: */
	const int numgrids=4;
	const int NDOF1=1;
	const int numdof=NDOF1*numgrids;
	double    xyz_list[numgrids][3];
	double    normal[2];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* gauss_coords =NULL;
	double  gauss_coord;
	double* gauss_weights=NULL;
	double  gauss_weight;

	/* matrices: */
	double B[numgrids];
	double L[numgrids];
	double DL1,DL2;
	double Ke_gg1[numdof][numdof];
	double Ke_gg2[numdof][numdof];
	double Ke_gg[numdof][numdof]={0.0};
	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double vx,vy;
	double UdotN;
	double dt;
	int    found;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Tria**  trias=NULL;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	trias=(Tria**)helements.deliverp();

	/*recover parameters: */
	if (analysis_type==Prognostic2AnalysisEnum){
		parameters->FindParam(&dt,DtEnum);
	}
	else if (analysis_type==Balancedthickness2AnalysisEnum){
		/*No transient term is involved*/
		dt=1;
	}
	else{
		ISSMERROR("analysis_type %i not supported yet");
	}

	/* Get node coordinates, dof list and normal vector: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);
	GetNormal(&normal[0],xyz_list);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	num_gauss=2;
	GaussSegment(&gauss_coords, &gauss_weights,num_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=gauss_weights[ig];
		gauss_coord =gauss_coords[ig];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet,xyz_list,gauss_coord);

		//Get vx, vy and v.n
		trias[0]->inputs->GetParameterValue(&vx, nodes[0],nodes[1],gauss_coord,VxAverageEnum);
		trias[0]->inputs->GetParameterValue(&vy, nodes[0],nodes[1],gauss_coord,VyAverageEnum);
		
		UdotN=vx*normal[0]+vy*normal[1];
		if (fabs(UdotN)<1.0e-9 && analysis_type==Balancedthickness2AnalysisEnum) printf("Edge number %i has a flux very small (u.n = %g ), which could lead to unaccurate results\n",id,UdotN);

		/*Get L and B: */
		GetL(&L[0],gauss_coord,numdof);
		GetB(&B[0],gauss_coord);

		/*Compute DLs*/
		DL1=gauss_weight*Jdet*dt*UdotN/2;
		DL2=gauss_weight*Jdet*dt*fabs(UdotN)/2;

		/*  Do the triple products: */
		TripleMultiply(&B[0],1,numdof,1,
					&DL1,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg1[0][0],0);
		TripleMultiply(&B[0],1,numdof,1,
					&DL2,1,1,0,
					&B[0],1,numdof,0,
					&Ke_gg2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg2[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	xfree((void**)&gauss_coords);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Numericalflux::CreateKMatrixBoundary {{{1*/
void  Numericalflux::CreateKMatrixBoundary(Mat Kgg){

	/* local declarations */
	int             i,j;
	int analysis_type,sub_analysis_type;

	/* node data: */
	const int numgrids=2;
	const int NDOF1=1;
	const int numdof=NDOF1*numgrids;
	double    xyz_list[numgrids][3];
	double    normal[2];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* gauss_coords =NULL;
	double  gauss_coord;
	double* gauss_weights=NULL;
	double  gauss_weight;

	/* matrices: */
	double L[numgrids];
	double DL;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof];
	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double vx,vy;
	double mean_vx;
	double mean_vy;
	double UdotN;
	double dt;
	int    dofs[1]={0};
	int    found;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Tria**  trias=NULL;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	trias=(Tria**)helements.deliverp();

	/*recover parameters: */
	if (analysis_type==Prognostic2AnalysisEnum){
		parameters->FindParam(&dt,DtEnum);
	}
	else if (analysis_type==Balancedthickness2AnalysisEnum){
		/*No transient term is involved*/
		dt=1;
	}
	else{
		ISSMERROR("analysis_type %i not supported yet");
	}
	
	/* Get node coordinates, dof list and normal vector: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);
	GetNormal(&normal[0],xyz_list);

	/*Check wether it is an inflow or outflow BC*/
	trias[0]->inputs->GetParameterValue(&mean_vx, nodes[0],nodes[1],.5,VxAverageEnum);
	trias[0]->inputs->GetParameterValue(&mean_vy, nodes[0],nodes[1],.5,VyAverageEnum);
	UdotN=mean_vx*normal[0]+mean_vy*normal[1];
	if (UdotN<=0){
		/*(u,n)<0 -> inflow, PenaltyCreatePVector will take care of it*/
		return;
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	num_gauss=2;
	GaussSegment(&gauss_coords, &gauss_weights,num_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=gauss_weights[ig];
		gauss_coord =gauss_coords[ig];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet,xyz_list,gauss_coord);

		//Get vx, vy and v.n
		trias[0]->inputs->GetParameterValue(&vx, nodes[0],nodes[1],gauss_coord,VxAverageEnum);
		trias[0]->inputs->GetParameterValue(&vy, nodes[0],nodes[1],gauss_coord,VyAverageEnum);

		UdotN=vx*normal[0]+vy*normal[1];

		/*Get L*/
		GetL(&L[0],gauss_coord,numdof);

		/*Compute DLs*/
		DL=gauss_weight*Jdet*dt*UdotN;

		/*Do triple product*/
		TripleMultiply(&L[0],1,numdof,1,
					&DL,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	xfree((void**)&gauss_coords);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVector {{{1*/
void  Numericalflux::CreatePVector(Vec pg){

	int type;

	/*recover type: */
	inputs->GetParameterValue(&type,TypeEnum);

	if (type==InternalEnum){
		CreatePVectorInternal(pg);
	}
	else if (type==BoundaryEnum){
		CreatePVectorBoundary(pg);
	}
	else ISSMERROR("type not supported yet");

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVectorInternal{{{1*/
void  Numericalflux::CreatePVectorInternal(Vec pg){

	/*Nothing added to PVector*/
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVectorBoundary{{{1*/
void  Numericalflux::CreatePVectorBoundary(Vec pg){

	/* local declarations */
	int             i,j;
	int analysis_type,sub_analysis_type;

	/* node data: */
	const int numgrids=2;
	const int NDOF1=1;
	const int numdof=NDOF1*numgrids;
	double    xyz_list[numgrids][3];
	double    normal[2];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* gauss_coords =NULL;
	double  gauss_coord;
	double* gauss_weights=NULL;
	double  gauss_weight;

	/* matrices: */
	double L[numgrids];
	double DL;
	double pe_g[numdof]={0.0};
	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double vx,vy;
	double mean_vx;
	double mean_vy;
	double UdotN;
	double thickness;
	double dt;
	int    dofs[1]={0};
	int    found;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Tria**  trias=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	trias=(Tria**)helements.deliverp();

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	this->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);

	/*recover parameters: */
	if (analysis_type==Prognostic2AnalysisEnum){
		parameters->FindParam(&dt,DtEnum);
	}
	else if (analysis_type==Balancedthickness2AnalysisEnum){
		/*No transient term is involved*/
		dt=1;
	}
	else{
		ISSMERROR("analysis_type %i not supported yet");
	}
	
	/* Get node coordinates, dof list and normal vector: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);
	GetNormal(&normal[0],xyz_list);

	/*Check wether it is an inflow or outflow BC*/
	trias[0]->inputs->GetParameterValue(&mean_vx, nodes[0],nodes[1],.5,VxAverageEnum);
	trias[0]->inputs->GetParameterValue(&mean_vy, nodes[0],nodes[1],.5,VyAverageEnum);
	UdotN=mean_vx*normal[0]+mean_vy*normal[1];
	if (UdotN>0){
		/*(u,n)>0 -> outflow, PenaltyCreateKMatrix will take care of it*/
		return;
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	num_gauss=2;
	GaussSegment(&gauss_coords, &gauss_weights,num_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=gauss_weights[ig];
		gauss_coord =gauss_coords[ig];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet,xyz_list,gauss_coord);

		//Get vx, vy and v.n
		trias[0]->inputs->GetParameterValue(&vx, nodes[0],nodes[1],gauss_coord,VxAverageEnum);
		trias[0]->inputs->GetParameterValue(&vy, nodes[0],nodes[1],gauss_coord,VyAverageEnum);
		trias[0]->inputs->GetParameterValue(&thickness, nodes[0],nodes[1],gauss_coord,ThicknessEnum);

		UdotN=vx*normal[0]+vy*normal[1];

		/*Get L*/
		GetL(&L[0],gauss_coord,numdof);

		/*Compute DL*/
		DL= - gauss_weight*Jdet*dt*UdotN*thickness;

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i] += DL* L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&gauss_coords);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*}}}*/
/*FUNCTION Numericalflux::GetB {{{1*/
void Numericalflux::GetB(double* B, double gauss_coord){

	const int numgrids=4;
	double l1l4[numgrids];

	/*Get nodal functions: */
	GetNodalFunctions(&l1l4[0],gauss_coord);

	/*Build B: */
	B[0] = +l1l4[0];
	B[1] = +l1l4[1];
	B[2] = -l1l4[0];
	B[3] = -l1l4[1];
}
/*}}}*/
/*FUNCTION Numericalflux::GetDofList{{{1*/

void  Numericalflux::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	int type;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	/*recover type: */
	inputs->GetParameterValue(&type,TypeEnum);

	if (type==InternalEnum){
		for(i=0;i<4;i++){
			nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
			for(j=0;j<numberofdofspernode;j++){
				doflist[i*numberofdofspernode+j]=doflist_per_node[j];
			}
		}
	}
	else if (type==BoundaryEnum){
		for(i=0;i<2;i++){
			nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
			for(j=0;j<numberofdofspernode;j++){
				doflist[i*numberofdofspernode+j]=doflist_per_node[j];
			}
		}
	}
	else ISSMERROR("type %s not supported yet",type);

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;
}
/*}}}*/
/*FUNCTION Numericalflux::Id {{{1*/
int    Numericalflux::Id(void){
	return id;
}
/*}}}*/
/*FUNCTION Numericalflux::GetJacobianDeterminant{{{1*/
void Numericalflux::GetJacobianDeterminant(double* pJdet,double xyz_list[4][3], double gauss_coord){

	double Jdet,length;

	length=sqrt(pow(xyz_list[1][0] - xyz_list[0][0],2.0) + pow(xyz_list[1][1] - xyz_list[0][1],2.0));
	Jdet=1.0/2.0*length;

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}

	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Numericalflux::GetL {{{1*/
void Numericalflux::GetL(double* L, double gauss_coord, int numdof){

	const int numgrids=4;
	double l1l4[numgrids];

	/*Check numdof*/
	ISSMASSERT(numdof==2 || numdof==4);

	/*Get nodal functions: */
	GetNodalFunctions(&l1l4[0],gauss_coord);

	/*Luild L: */
	L[0] = l1l4[0];
	L[1] = l1l4[1];
	if (numdof==4){
		L[2] = l1l4[2];
		L[3] = l1l4[3];
	}
}
/*}}}*/
/*FUNCTION Numericalflux::GetNodalFunctions{{{1*/
void Numericalflux::GetNodalFunctions(double* l1l4, double gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l4[0]=-0.5*gauss_coord+0.5;
	l1l4[1]=+0.5*gauss_coord+0.5;
	l1l4[2]=-0.5*gauss_coord+0.5;
	l1l4[3]=+0.5*gauss_coord+0.5;

}
/*}}}*/
/*FUNCTION Numericalflux::GetNormal {{{1*/
void Numericalflux:: GetNormal(double* normal,double xyz_list[4][3]){

	/*Build unit outward pointing vector*/
	const int numgrids=4;
	double vector[2];
	double norm;

	vector[0]=xyz_list[1][0] - xyz_list[0][0];
	vector[1]=xyz_list[1][1] - xyz_list[0][1];

	norm=sqrt(pow(vector[0],2.0)+pow(vector[1],2.0));

	normal[0]= + vector[1]/norm;
	normal[1]= - vector[0]/norm;
}
/*}}}*/
/*FUNCTION Numericalflux::GetParameterValue {{{1*/
void Numericalflux::GetParameterValue(double* pp, double* plist, double gauss_coord){

	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter value at gaussian 
	 * point specifie by gauss_l1l2l3: */

	/*nodal functions: */
	double l1l4[4];

	/*output: */
	double p;

	GetNodalFunctions(&l1l4[0],gauss_coord);

	p=l1l4[0]*plist[0]+l1l4[1]*plist[1];

	/*Assign output pointers:*/
	*pp=p;
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrix {{{1*/
void  Numericalflux::PenaltyCreateKMatrix(Mat Kgg,double kmax){

	/*No stiffness loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVector{{{1*/
void  Numericalflux::PenaltyCreatePVector(Vec pg,double kmax){

	/*No penalty loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::InAnalysis(int analysis_type){{{1*/
bool Numericalflux::InAnalysis(int in_analysis_type){
	if (in_analysis_type=this->analysis_type)return true;
	else return false;
}
/*}}}*/
