function md=SetIceShelfBC(md,icefrontfile)
%SETICESHELFBC - Create the boundary conditions for diagnostic and thermal models for a  Ice Shelf with Ice Front
%
%   Neumann BC are used on the ice front (an ANRGUS contour around the ice front
%   must be given in input)
%   Dirichlet BC are used elsewhere for diagnostic
%
%   Usage:
%      md=SetIceShelfBC(md,icefrontfile)
%
%   Example:
%      md=SetIceShelfBC(md,'Front.exp');
%
%   See also: SETICESHEETBC, SETMARINEICESHEETBC

%grid on Dirichlet (boundary and ~icefront)
if ~exist(icefrontfile)
	error(['SetIceShelfBC error message: ice front file ' icefrontfile ' not found']);
end
gridinsideicefront=ContourToMesh(md.elements,md.x,md.y,expread(icefrontfile,1),'node',2);
gridonicefront=double(md.gridonboundary & gridinsideicefront);
pos=find(md.gridonboundary & ~gridonicefront);
md.gridondirichlet_diag=zeros(md.numberofgrids,1);
md.gridondirichlet_diag(pos)=1;

%Dirichlet Values
md.dirichletvalues_diag=zeros(md.numberofgrids,2);
if (length(md.vx_obs)==md.numberofgrids & length(md.vy_obs)==md.numberofgrids)
	disp('      boundary conditions for diagnostic model: spc set as observed velocities');
	md.dirichletvalues_diag(pos,:)=[md.vx_obs(pos) md.vy_obs(pos)];
else
	disp('      boundary conditions for diagnostic model: spc set as zero');
end

%segment on Neumann (Ice Front)
pos=find(gridonicefront(md.segments(:,1)) | gridonicefront(md.segments(:,2)));
md.segmentonneumann_diag=md.segments(pos,:);
md.neumannvalues_diag=NaN*ones(length(md.segmentonneumann_diag),1); %dynamic boundary conditions (water pressure)

%Create zeros melting and accumulation if not specified
if isnan(md.accumulation),
	md.accumulation=zeros(md.numberofgrids,1);
	disp('      no accumulation specified: values set as zero');
end
if isnan(md.melting),
	md.melting=zeros(md.numberofgrids,1);
	disp('      no melting specified: values set as zero');
end

disp('      boundary conditions for prognostic model initialization ');
md.gridondirichlet_prog=zeros(md.numberofgrids,1);
md.dirichletvalues_prog=zeros(md.numberofgrids,1);
md.segmentonneumann_prog=[];
md.neumannvalues_prog=[];

if (length(md.observed_temperature)==md.numberofgrids),
	disp('      boundary conditions for thermal model');
	md.gridondirichlet_thermal=ones(md.numberofgrids,1); %surface temperature
	md.dirichletvalues_thermal=md.observed_temperature;
	if (length(md.geothermalflux)~=md.numberofgrids),
		md.geothermalflux=zeros(md.numberofgrids,1);
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
