/*!\file Node.c
 * \brief: implementation of the Node object
 */

/*Include files: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "./objects.h"
#include "../DataSet/DataSet.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/include.h"
/*}}}*/

/*Object constructors and destructors: {{{1*/
/*FUNCTION Node::Node() default constructor {{{2*/
Node::Node(){
	this->inputs=NULL;
	return;
}
/*}}}*/
/*FUNCTION Node::Node(int id, int vertex_id, int uppernode_id, int numdofs, NodeProperties*) {{{2*/
Node::Node(int node_id,int node_vertex_id, int node_upper_node_id, int node_numdofs):
	indexing(node_numdofs),
    hvertex(&node_vertex_id,1),
    hupper_node(&node_upper_node_id,1){

	this->id=node_id;
	this->inputs=new Inputs();
}
/*}}}*/
/*FUNCTION Node::Node(int id, DofIndexing* indexing, Hook* vertex, Hook* uppernode,Inputs* inputs,int analysis_type){{{2*/
Node::Node(int node_id,DofIndexing* node_indexing, Hook* node_vertex, Hook* node_upper_node,Inputs* node_inputs,int analysis_type):
	    indexing(node_indexing),
		hvertex(node_vertex),
		hupper_node(node_upper_node) {

	    /*all the initialization has been done by the initializer, just fill in the id: */
	    this->id=node_id;
		this->analysis_type=analysis_type;

		if(node_inputs){
			this->inputs=(Inputs*)node_inputs->Copy();
		}
		else{
			this->inputs=new Inputs();
		}
}
/*}}}*/
/*FUNCTION Node::Node(int id, int i, IoModel* iomodel,int analysis_type)          -> Continuous Galerkin{{{2*/
Node::Node(int node_id, int i, IoModel* iomodel,int analysis_type){ //i is the node index

	int k;
	int numdofs;
	int vertex_id;
	int upper_node_id;

	/*id: */
	this->id=node_id; //matlab indexing
	this->analysis_type=analysis_type;

	/*indexing:*/
	DistributeNumDofs(&numdofs,analysis_type); //number of dofs per node

	this->indexing.Init(numdofs);

	/*hooks: */
	vertex_id=this->id; //node and vertex have the same id, as we are running galerkin continuous, with same number of nodes and vertices.

	if (iomodel->dim==3){
		if (isnan(iomodel->uppernodes[i])){
			upper_node_id=this->id; //nodes on surface do not have upper nodes, only themselves.
		}
		else{
			upper_node_id=(int)iomodel->uppernodes[i];
		}
	}
	else{
		/*If we are running 2d, upper_node does not mean much. Just point towards itself!:*/
		upper_node_id=this->id;
	}

	this->hvertex.Init(&vertex_id,1); //node id is the same as the vertex id, continuous galerkin!
	this->hupper_node.Init(&upper_node_id,1);

	/*set single point constraints: */

	/*Diagnostic Horiz*/
	if (analysis_type==DiagnosticHorizAnalysisEnum){
		if (iomodel->dim==3){
			/*We have a  3d mesh, we may have collapsed elements, hence dead grids. Freeze them out: */
			if (!iomodel->deadgrids) ISSMERROR("iomodel->deadgrids is NULL");
			if (iomodel->deadgrids[i]){
				for(k=1;k<=numdofs;k++){
					this->FreezeDof(k);
				}
			}
		}
		/*spc all nodes on hutter*/
		if (!iomodel->gridonhutter) ISSMERROR("iomodel->gridonhutter is NULL");
		if (iomodel->gridonhutter[i]){
			for(k=1;k<=numdofs;k++){
				this->FreezeDof(k);
			}
		}
	}

	/*Diagnostic Stokes*/
	if (analysis_type==DiagnosticStokesAnalysisEnum){
		/*On a 3d mesh, in stokes formualtions, only stokes grids are free, the others are frozen: */
		if (!iomodel->borderstokes) ISSMERROR("iomodel->borderstokes is NULL");
		if (iomodel->borderstokes[i]){
			//freeze everything except pressure
			this->FreezeDof(1);
			this->FreezeDof(2);
			this->FreezeDof(3);
		}
		else if (iomodel->gridonstokes[i]==0){
			for(k=1;k<=numdofs;k++){
				this->FreezeDof(k);
			}
		}
	}

	/*Diagnostic Hutter*/
	if (analysis_type==DiagnosticHutterAnalysisEnum){
		/*Spc all nodes that are not Hutter*/
		if (!iomodel->gridonhutter) ISSMERROR("iomodel->gridonhutter is NULL");
		if (!iomodel->gridonhutter[i]){
			for(k=1;k<=numdofs;k++){
				this->FreezeDof(k);
			}
		}
	}

	/*Prognostic/ Melting/ Slopecompute/ Balancedvelocities/ Balancedthickness*/
	if (
				analysis_type==PrognosticAnalysisEnum || 
				analysis_type==MeltingAnalysisEnum || 
				analysis_type==SlopeAnalysisEnum || 
				analysis_type==BalancedvelocitiesAnalysisEnum || 
				analysis_type==BalancedthicknessAnalysisEnum
				){
		if (iomodel->dim==3){
			/*On a 3d mesh, we may have collapsed elements, hence dead grids. Freeze them out: */
			if (!iomodel->gridonbed) ISSMERROR("iomodel->gridonbed is NULL");
			if (!iomodel->gridonbed[i]){
				for(k=1;k<=numdofs;k++){
					this->FreezeDof(k);
				}
			}
		}
	}
	
	//intialize inputs, and add as many inputs per element as requested: 
	this->inputs=new Inputs();
	if (iomodel->gridonbed)      this->inputs->AddInput(new BoolInput(NodeOnBedEnum,(IssmBool)iomodel->gridonbed[i]));
	if (iomodel->gridonsurface)  this->inputs->AddInput(new BoolInput(NodeOnSurfaceEnum,(IssmBool)iomodel->gridonsurface[i]));
	if (iomodel->gridoniceshelf) this->inputs->AddInput(new BoolInput(NodeOnIceShelfEnum,(IssmBool)iomodel->gridoniceshelf[i]));
	if (iomodel->gridonicesheet) this->inputs->AddInput(new BoolInput(NodeOnIceSheetEnum,(IssmBool)iomodel->gridonicesheet[i]));
	if (iomodel->numbernodetoelementconnectivity) this->inputs->AddInput(new IntInput(NumberNodeToElementConnectivityEnum,iomodel->numbernodetoelementconnectivity[i]));

}
/*}}}*/
/*FUNCTION Node::Node(int id, int i, int j, IoModel* iomodel,int analysis_type)   -> Discontinuous Galerkin{{{2*/
Node::Node(int node_id,int i,int j,IoModel* iomodel,int analysis_type){
	/* i -> index of the vertex in C indexing
	 * j -> index of the node in C indexing*/

	int numdofs;
	int vertex_id;
	int upper_node_id;

	/*id: */
	this->id=node_id; //matlab indexing
	this->analysis_type=analysis_type;

	/*indexing:*/
	DistributeNumDofs(&numdofs,analysis_type); //number of dofs per node

	this->indexing.Init(numdofs);

	
	/*hooks: */
	vertex_id=i+1; //matlab indexing

	if (iomodel->dim==3){
		if (isnan(iomodel->uppernodes[i])){
			upper_node_id=this->id; //nodes on surface do not have upper nodes, only themselves.
		}
		else{
			upper_node_id=(int)iomodel->uppernodes[i];
		}
	}
	else{
		/*If we are running 2d, upper_node does not mean much. Just point towards itself!:*/
		upper_node_id=this->id;
	}

	this->hvertex.Init(&vertex_id,1);
	this->hupper_node.Init(&upper_node_id,1);
	
	//intialize inputs, and add as many inputs per element as requested: 
	this->inputs=new Inputs();
	if (iomodel->gridonbed) this->inputs->AddInput(new BoolInput(NodeOnBedEnum,(IssmBool)iomodel->gridonbed[i]));
	if (iomodel->gridonsurface) this->inputs->AddInput(new BoolInput(NodeOnSurfaceEnum,(IssmBool)iomodel->gridonsurface[i]));
	if (iomodel->gridoniceshelf) this->inputs->AddInput(new BoolInput(NodeOnIceShelfEnum,(IssmBool)iomodel->gridoniceshelf[i]));
	if (iomodel->gridonicesheet) this->inputs->AddInput(new BoolInput(NodeOnIceSheetEnum,(IssmBool)iomodel->gridonicesheet[i]));


}
/*}}}*/
/*FUNCTION Node::~Node(){{{2*/
Node::~Node(){
	delete inputs;
	return;
}
/*}}}*/
/*}}}*/
/*Object management: {{{1*/
/*FUNCTION Node::Configure {{{2*/
void  Node::Configure(DataSet* nodesin,DataSet* verticesin){

	int i;

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hvertex.configure(verticesin);
	hupper_node.configure(nodesin);

}
/*FUNCTION Node::copy {{{2*/
Object* Node::copy() {
		
	return new Node(this->id,&this->indexing, &this->hvertex,&this->hupper_node,this->inputs,this->analysis_type);

}

/*}}}*/
/*FUNCTION Node::DeepEcho{{{2*/
void Node::DeepEcho(void){

	printf("Node:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumAsString(analysis_type));
	indexing.DeepEcho();
	printf("Vertex:\n");
	hvertex.DeepEcho();
	printf("Upper node:\n");
	//Do not Deepecho the upper_node otherwise DeepEcho will go crazy!
	hupper_node.Echo();
	printf("   inputs\n");
	inputs->DeepEcho();

}
/*}}}*/
/*FUNCTION Node::Demarshall{{{2*/
void  Node::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	
	/*demarshall objects: */
	indexing.Demarshall(&marshalled_dataset);
	hvertex.Demarshall(&marshalled_dataset);
	hupper_node.Demarshall(&marshalled_dataset);

	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Node::Echo{{{2*/
void Node::Echo(void){

	printf("Node:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumAsString(analysis_type));
	indexing.Echo();
	hvertex.Echo();
	hupper_node.Echo();
	printf("   inputs\n");
	inputs->DeepEcho();


}
/*}}}*/
/*FUNCTION Node::Enum{{{2*/
int Node::Enum(void){

	return NodeEnum;

}
/*}}}*/
/*FUNCTION Node::GetDof {{{2*/
int   Node::GetDof(int dofindex){

	return indexing.doflist[dofindex];

}
/*}}}*/
/*FUNCTION Node::GetDofList1{{{2*/
int  Node::GetDofList1(void){

	Vertex* vertex=NULL;

	vertex=(Vertex*)this->hvertex.delivers();

	return vertex->dof;
}
/*}}}*/
/*FUNCTION Node::GetDofList{{{2*/
void  Node::GetDofList(int* outdoflist,int* pnumberofdofspernode){

	int i;
	for(i=0;i<this->indexing.numberofdofs;i++){
		outdoflist[i]=indexing.doflist[i];
	}
	/*Assign output pointers:*/
	*pnumberofdofspernode=this->indexing.numberofdofs;
}
/*}}}*/
/*FUNCTION Node::Id{{{2*/
int    Node::Id(void){ return id; }
/*}}}*/
/*FUNCTION Node::GetVertexId {{{2*/
int   Node::GetVertexId(void){

	Vertex*  vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->id;
}
/*}}}*/
/*FUNCTION Node::GetVertexDof {{{2*/
int   Node::GetVertexDof(void){

	Vertex*  vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->dof;
}
/*}}}*/
/*FUNCTION Node::Marshall{{{2*/
void  Node::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Node: */
	enum_type=NodeEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Node data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	
	/*marshall objects: */
	indexing.Marshall(&marshalled_dataset);
	hvertex.Marshall(&marshalled_dataset);
	hupper_node.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*Free ressources:*/
	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Node::MarshallSize{{{2*/
int   Node::MarshallSize(){

	return sizeof(id)+
		indexing.MarshallSize()+
		hvertex.MarshallSize()+
		hupper_node.MarshallSize()+
		inputs->MarshallSize()+
		sizeof(analysis_type)+
		sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Node::SetVertexDof {{{2*/
void   Node::SetVertexDof(int in_dof){

	Vertex*  vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	vertex->dof=in_dof;

}
/*}}}*/
/*FUNCTION Node::InAnalysis(int analysis_type){{{1*/
bool Node::InAnalysis(int in_analysis_type){
	if (in_analysis_type=this->analysis_type)return true;
	else return false;
}
/*}}}*/
/*Object numerics: {{{1*/
/*FUNCTION Node::ApplyConstraints{{{2*/
void  Node::ApplyConstraint(Vec yg,int dof,double value){

	int index;

	/*First, dof should be added in the s set, describing which 
	 * dofs are constrained to a certain value (dirichlet boundary condition*/

	DofInSSet(dof-1);

	/*Second, we should add value into yg, at dof corresponding to doflist[dof], unless
	 *  we are a clone!*/

	if(!indexing.clone){

		index=indexing.doflist[dof-1]; //matlab indexing

		VecSetValues(yg,1,&index,&value,INSERT_VALUES);

	}

}
/*}}}*/
/*FUNCTION Node::CreateVecSets {{{2*/
void  Node::CreateVecSets(Vec pv_g,Vec pv_m,Vec pv_n,Vec pv_f,Vec pv_s){

	double gvalue=1.0; //all nodes are in the g set;
	double value;

	int i;

	for(i=0;i<this->indexing.numberofdofs;i++){

		/*g set: */
		VecSetValues(pv_g,1,&indexing.doflist[i],&gvalue,INSERT_VALUES);
		
		/*m set: */
		value=(double)this->indexing.m_set[i];
		VecSetValues(pv_m,1,&indexing.doflist[i],&value,INSERT_VALUES);

		/*n set: */
		value=(double)this->indexing.n_set[i];
		VecSetValues(pv_n,1,&indexing.doflist[i],&value,INSERT_VALUES);

		/*f set: */
		value=(double)this->indexing.f_set[i];
		VecSetValues(pv_f,1,&indexing.doflist[i],&value,INSERT_VALUES);

		/*s set: */
		value=(double)this->indexing.s_set[i];
		VecSetValues(pv_s,1,&indexing.doflist[i],&value,INSERT_VALUES);

	}


}
/*}}}*/
/*FUNCTION Node::DofInMSet{{{2*/
void  Node::DofInMSet(int dof){

	/*Put dof for this node into the m set (m set is for rigid body modes)*/

	this->indexing.m_set[dof]=1; //m and n are mutually exclusive (m for rigid body modes)
	this->indexing.n_set[dof]=0;
	this->indexing.f_set[dof]=0; //n splits into f (for which we solve) and s (single point constraints)
	this->indexing.s_set[dof]=0;
}
/*}}}*/
/*FUNCTION Node::DofInSSet {{{2*/
void  Node::DofInSSet(int dof){

	/*Put dof for this node into the s set (ie, this dof will be constrained 
	 * to a fixed value during computations. */

	this->indexing.m_set[dof]=0; //m and n are mutually exclusive (m for rigid body modes)
	this->indexing.n_set[dof]=1;
	this->indexing.f_set[dof]=0; //n splits into f (for which we solve) and s (single point constraints)
	this->indexing.s_set[dof]=1;
}
/*}}}*/
/*FUNCTION Node::DofIsInMSet{{{2*/
int  Node::DofIsInMSet(int dof){

	if (this->indexing.m_set[dof])return 1;
	else return 0;

}
/*}}}*/
/*FUNCTION Node::FieldAverageOntoVertices{{{2*/
void  Node::FieldAverageOntoVertices(Vec fieldsum,Vec connectivity,double* field){

	/*Intermediary*/
	int vertexdof;
	int index;
	double value;

	/*Skip if clone*/
	if (indexing.clone==1) return;

	/*Check dofpernode*/
	if (indexing.numberofdofs!=1) ISSMERROR("only one dof suppoerted yet");

	/*Get value at node*/
	index=indexing.doflist[0];
	value=field[index];

	/*Add values to the two vectors*/
	vertexdof=this->GetVertexDof();
	VecSetValue(fieldsum,    vertexdof,value,ADD_VALUES);
	VecSetValue(connectivity,vertexdof,1.0,  ADD_VALUES);
}
/*}}}*/
/*FUNCTION Node::FieldDepthAverageAtBase{{{2*/
void  Node::FieldDepthAverageAtBase(Vec field,double* field_serial,char* fieldname){

	/* node data: */
	int          vertexdof;
	int          dofx,dofy;
	int          isnodeonsurface;

	bool onbed;
	bool onsurface;
	
	Node* node=NULL;
	Node* upper_node=NULL;
	double z1,z2,dz;
	double thickness;

	/*recover parameters: */
	inputs->GetParameterValue(&onbed,NodeOnBedEnum);
	inputs->GetParameterValue(&onsurface,NodeOnSurfaceEnum);

	/*Are we on the base, not on the surface, and not on a clone node?:*/
	
	if(onbed==1 & indexing.clone==0 & onsurface==0){
			
		vertexdof=this->GetVertexDof();

		/*this node is on the bed. We are going to, follow the upper nodes until we reach the surface. At each upper node, 
		 * we'll grab the * field for this node, and add it to overall field: */

		if(strcmp(fieldname,"velocity")==0){

			/*field is a velocity, 2 dofs per node: */
			double velocity2[2];
			double velocity1[2];
			double velocity_average[2];
			double sum[2];

			sum[0]=0;
			sum[1]=0;
			thickness=0;

			/*get dofs for this base node velocity: we know there are two dofs in field_serial */
			dofx=2*vertexdof;
			dofy=2*vertexdof+1;

			node=this;
			for(;;){

				if (node->IsOnSurface())break;

				vertexdof=node->GetVertexDof();
				
				velocity1[0]=field_serial[2*vertexdof];
				velocity1[1]=field_serial[2*vertexdof+1];
				z1=node->GetZ();

				upper_node=node->GetUpperNode();
				vertexdof=upper_node->GetVertexDof();
			
				velocity2[0]=field_serial[2*vertexdof];
				velocity2[1]=field_serial[2*vertexdof+1];
				z2=upper_node->GetZ();

				dz=(z2-z1);
				thickness+=dz;
				velocity_average[0]=(velocity1[0]+velocity2[0])/2.0;
				velocity_average[1]=(velocity1[1]+velocity2[1])/2.0;

				sum[0]+=velocity_average[0]*dz;
				sum[1]+=velocity_average[1]*dz;

				/* get next node: */
				node=node->GetUpperNode();
			}

			sum[0]=sum[0]/thickness;
			sum[1]=sum[1]/thickness;

			/* Plfield velocity_average*deltaH/H into base of field: */
			VecSetValues(field,1,&dofx,&sum[0],INSERT_VALUES);
			VecSetValues(field,1,&dofy,&sum[1],INSERT_VALUES);
		}
		else{
			/*field is regular, 1 dof per node: */
			double field2;
			double field1;
			double field_average;
			double sum;

			sum=0;
			thickness=0;

			/*get dofs for this base node velocity: we know there are two dofs in field_serial */
			dofx=vertexdof;

			node=this;
			for(;;){

				if (node->IsOnSurface()) break;

				vertexdof=node->GetVertexDof();
				
				field1=field_serial[vertexdof];
				z1=node->GetZ();

				upper_node=node->GetUpperNode();
				vertexdof=upper_node->GetVertexDof();
			
				field2=field_serial[vertexdof];
				z2=upper_node->GetZ();

				dz=(z2-z1);
				thickness+=dz;
				field_average=(field1+field2)/2.0;

				sum+=field_average*dz;

				/* get next node: */
				node=node->GetUpperNode();
			}

			sum=sum/thickness;

			/* Plug field_average*deltH/H into base of field: */
			VecSetValues(field,1,&dofx,&sum,INSERT_VALUES);
		}
	}
}
/*}}}*/
/*FUNCTION Node::VecExtrude {{{2*/
void  Node::VecExtrude(Vec vector,double* vector_serial){
		
	/* node data: */
	int   numberofdofspernode;
	Node* node=NULL;
	int   i;

	bool onbed;

	/*recover parameters: */
	inputs->GetParameterValue(&onbed,NodeOnBedEnum);

	/*Is this node on bed? :*/
	if (onbed){

		/* node data: */
		int          dof1;
		double       vectorel;

		//initilaize node and get dof1
		node=this;
		dof1=node->GetVertexDof();

		/*get vector for this base node: */
		vectorel=vector_serial[dof1];

		//go throvectorn all nodes which sit on top of this node, until we reach the surface, 
		//and plvector  vector in vector
		for(;;){

			dof1=node->GetVertexDof();
			VecSetValues(vector,1,&dof1,&vectorel,INSERT_VALUES);

			if (node->IsOnSurface())break;
			/*get next node: */
			node=node->GetUpperNode();
		}
	} 
}
/*}}}*/
/*FUNCTION Node::FreezeDof{{{2*/
void  Node::FreezeDof(int dof){
	
	DofInSSet(dof-1); //with 0 displacement for this dof.

}
/*}}}*/
/*FUNCTION Node::GetConnectivity {{{2*/
int Node::GetConnectivity(){
	int connectivity;

	/*recover parameters: */
	inputs->GetParameterValue(&connectivity,NumberNodeToElementConnectivityEnum);

	return connectivity;
}
/*}}}*/
/*FUNCTION Node::GetNumberOfDofs{{{2*/
int   Node::GetNumberOfDofs(){
	
	return this->indexing.numberofdofs;

}
/*}}}*/
/*FUNCTION Node::GetSigma {{{2*/
double Node::GetSigma(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->sigma;
}
/*}}}*/
/*FUNCTION Node::GetUpperNode {{{2*/
Node* Node::GetUpperNode(){
	Node* upper_node=NULL;
	upper_node=(Node*)hupper_node.delivers();
	return upper_node;
}
/*}}}*/
/*FUNCTION Node::GetX {{{2*/
double Node::GetX(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->x;
}
/*}}}*/
/*FUNCTION Node::GetY {{{2*/
double Node::GetY(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->y;
}
/*}}}*/
/*FUNCTION Node::GetZ {{{2*/
double Node::GetZ(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->z;
}
/*}}}*/
/*FUNCTION Node::IsClone {{{2*/
int   Node::IsClone(){
	
	return indexing.clone;

}
/*}}}*/
/*FUNCTION Node::IsOnBed {{{2*/
int   Node::IsOnBed(){

	bool onbed;

	/*recover parameters: */
	inputs->GetParameterValue(&onbed,NodeOnBedEnum);

	return onbed;
}
/*}}}*/
/*FUNCTION Node::IsOnSheet {{{2*/
int   Node::IsOnSheet(){

	bool onsheet;

	/*recover parameters: */
	inputs->GetParameterValue(&onsheet,NodeOnIceSheetEnum);

	return onsheet;
}		
/*}}}*/
/*FUNCTION Node::IsOnShelf {{{2*/
int   Node::IsOnShelf(){
	
	bool onshelf;

	/*recover parameters: */
	inputs->GetParameterValue(&onshelf,NodeOnIceShelfEnum);

	return onshelf;
}
/*}}}*/
/*FUNCTION Node::IsOnSurface {{{2*/
int   Node::IsOnSurface(){

	bool onsurface;

	/*recover parameters: */
	inputs->GetParameterValue(&onsurface,NodeOnSurfaceEnum);

	return onsurface;
}
/*}}}*/
/*FUNCTION Node::MyRank{{{2*/
int    Node::MyRank(void){ 
	extern int my_rank;

	return my_rank; 
}
/*}}}*/
/*FUNCTION Node::UpdateFromInputs {{{2*/
void  Node::UpdateFromInputs(void* vinputs){
	
	/*nothing updated for now*/

}
/*}}}*/
/*FUNCTION Node::InputUpdateFromVector(double* vector, int name, int type){{{2*/
void  Node::InputUpdateFromVector(double* vector, int name, int type){

	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Node::InputUpdateFromVector(int* vector, int name, int type){{{2*/
void  Node::InputUpdateFromVector(int* vector, int name, int type){

	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Node::InputUpdateFromVector(bool* vector, int name, int type){{{2*/
void  Node::InputUpdateFromVector(bool* vector, int name, int type){

	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Node::InputUpdateFromConstant(double constant, int name){{{2*/
void  Node::InputUpdateFromConstant(double constant, int name){

	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Node::InputUpdateFromConstant(int constant, int name){{{2*/
void  Node::InputUpdateFromConstant(int constant, int name){

	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Node::InputUpdateFromConstant(bool constant, int name){{{2*/
void  Node::InputUpdateFromConstant(bool constant, int name){

	/*Nothing updated yet*/
}
/*}}}*/
/*}}}*/
/* DofObject routines: {{{1*/
/*FUNCTION Node::DistributeDofs{{{2*/
void  Node::DistributeDofs(int* pdofcount){

	int i;
	extern int my_rank;
	int dofcount;

	dofcount=*pdofcount;
	
	if(indexing.clone){
		/*This node is a clone! Don't distribute dofs, it will get them from another cpu!*/
		return;
	}

	/*This node should distribute dofs, go ahead: */
	for(i=0;i<this->indexing.numberofdofs;i++){
		indexing.doflist[i]=dofcount+i;
	}
	dofcount+=this->indexing.numberofdofs;

	/*Assign output pointers: */
	*pdofcount=dofcount;

}
/*}}}*/
/*FUNCTION Node::OffsetDofs{{{2*/
void  Node::OffsetDofs(int dofcount){
	
	int i;
	extern int my_rank;
	
	if(indexing.clone){
		/*This node is a clone, don't offset the dofs!: */
		return;
	}

	/*This node should offset the dofs, go ahead: */
	for(i=0;i<this->indexing.numberofdofs;i++){
		indexing.doflist[i]+=dofcount;
	}
}
/*}}}*/
/*FUNCTION Node::ShowTrueDofs{{{2*/
void  Node::ShowTrueDofs(int* truedofs){

	int j;
	extern int my_rank;
	
	/*Are we a clone? : */
	if(indexing.clone)return;

	/*Ok, we are not a clone, just plug our dofs into truedofs: */
	for(j=0;j<this->indexing.numberofdofs;j++){
		*(truedofs+this->indexing.numberofdofs*(id-1)+j)=indexing.doflist[j];
	}

}
/*}}}*/
/*FUNCTION Node::UpdateCloneDofs{{{2*/
void  Node::UpdateCloneDofs(int* alltruedofs){

	int j;
	extern int my_rank;
	
	/*If we are not a clone, don't update, we already have dofs!: */
	if(indexing.clone==0)return;

	/*Ok, we are a clone node, but we did not create the dofs for this node. 
	 * Therefore, our doflist is garbage right now. Go pick it up in the alltruedofs: */
	for(j=0;j<this->indexing.numberofdofs;j++){
		indexing.doflist[j]=*(alltruedofs+this->indexing.numberofdofs*(id-1)+j);
	}
}
/*}}}*/
/*FUNCTION Node::SetClone {{{2*/
void  Node::SetClone(int* minranks){

	extern int my_rank;

	if (minranks[id-1]==my_rank){
		indexing.clone=0;
	}
	else{
		/*!there is a cpu with lower rank that has the same node, 
		therefore, I am a clone*/
		indexing.clone=1; 	
	}

}
/*}}}*/
/*FUNCTION Node::CreatePartition{{{2*/
void  Node::CreatePartition(Vec partition){ 

	int      idxm;
	double   value;

	idxm=(id-1);
	value=(double)this->indexing.doflist[0];
	ISSMASSERT(value>=0);

	VecSetValues(partition,1,&idxm,&value,INSERT_VALUES);

	return;
}
/*}}}*/
/*}}}*/
