/*!\file: solver_thermal_nonlinear.cpp
 * \brief: core of the thermal solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void solver_thermal_nonlinear(Vec* ptg,double* pmelting_offset,FemModel* fem){

	/*solution : */
	Vec tg=NULL; 
	Vec tf=NULL; 
	Vec tf_old=NULL; 
	double melting_offset;

	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kgg_nopenalty=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pg_nopenalty=NULL;
	Vec pf=NULL;

	int converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int numberofnodes;
	int min_thermal_constraints;
	bool reset_penalties;

	/*parameters:*/
	int kflag,pflag;
	char* solver_string=NULL;
	int verbose=0;
	bool lowmem=0;

	/*Recover parameters: */
	kflag=1; pflag=1;

	fem->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	fem->parameters->FindParam(&solver_string,SolverStringEnum);
	fem->parameters->FindParam(&verbose,VerboseEnum);
	fem->parameters->FindParam(&lowmem,LowmemEnum);
	fem->parameters->FindParam(&min_thermal_constraints,MinThermalConstraintsEnum);

	count=1;
	converged=0;

	for(;;){

		if(verbose)_printf_("%s\n","starting direct shooting method");

		if(count==1) reset_penalties=1; else reset_penalties=0;
		InputUpdateFromConstantx( fem->elements,fem->nodes, fem->vertices, fem->loads, fem->materials, fem->parameters,reset_penalties,ResetPenaltiesEnum);

		//*Generate system matrices
		if (!lowmem){

			/*Compute Kgg_nopenalty and pg_nopenalty once for all: */
			if (count==1){
				SystemMatricesx(&Kgg_nopenalty, &pg_nopenalty,fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters,kflag,pflag);
			}

			/*Copy K_gg_nopenalty into Kgg, same for pg: */
			MatDuplicate(Kgg_nopenalty,MAT_COPY_VALUES,&Kgg);
			VecDuplicatePatch(&pg,pg_nopenalty);

			//apply penalties each time
			PenaltySystemMatricesx(Kgg, pg,&melting_offset,fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters,kflag,pflag);
		}
		else{
			SystemMatricesx(&Kgg, &pg,fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters,kflag,pflag);
			//apply penalties
			PenaltySystemMatricesx(Kgg, pg,&melting_offset,fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters,kflag,pflag);
		}

		/*!Reduce matrix from g to f size:*/
		Reducematrixfromgtofx(&Kff,&Kfs,Kgg,fem->Gmn,fem->nodesets);

		/*Free ressources: */
		MatFree(&Kgg);
	
		if (verbose) _printf_("   reducing load from g to f set\n");
		/*!Reduce load from g to f size: */
		Reduceloadfromgtofx(&pf, pg, fem->Gmn, Kfs, fem->ys, fem->nodesets);

		//no need for pg and Kfs anymore 
		VecFree(&pg); 
		MatFree(&Kfs);

		/*Solve: */
		if(verbose)_printf_("%s\n","solving");
		VecFree(&tf);
		Solverx(&tf, Kff, pf,tf_old, solver_string);
		VecFree(&tf_old); VecDuplicatePatch(&tf_old,tf);
	
		//no need for Kff and pf anymore
		MatFree(&Kff);VecFree(&pf);VecFree(&tg);

		if (verbose) _printf_("   merging solution from f to g set\n");
		//Merge back to g set
		Mergesolutionfromftogx(&tg, tf,fem->Gmn,fem->ys,fem->nodesets);

		//Deal with penalty loads
		if (verbose) _printf_("   penalty constraints\n");
		PenaltyConstraintsx(&constraints_converged, &num_unstable_constraints, fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters);
		
		//Update inputs using new solution:
		UpdateInputsFromVectorx( fem->elements,fem->nodes, fem->vertices, fem->loads, fem->materials, fem->parameters,tg,TemperatureEnum,VertexEnum);
		InputUpdateFromSolutionx(fem->elements,fem->nodes, fem->vertices, fem->loads, fem->materials, fem->parameters,tg);

		if (!converged){
			if(verbose)_printf_("%s%i\n","   #unstable constraints = ",num_unstable_constraints);
			if (num_unstable_constraints <= min_thermal_constraints)converged=1;
		}
		count++;
		
		if(converged==1)break;
	}

	/*Free ressources: */
	MatFree(&Kgg_nopenalty);
	VecFree(&pg_nopenalty);
	VecFree(&tf);
	VecFree(&tf_old);
	delete solver_string;

	/*Assign output pointers: */
	*ptg=tg;
	*pmelting_offset=melting_offset;
}
