/*!\file: solver_diagnostic_nonlinear.cpp
 * \brief: core of the diagnostic solution for non linear materials
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"
#include "../solutions/solutions.h"
#include "./solvers.h"

void solver_diagnostic_nonlinear(Vec* pug,Mat* pKff0,Mat* pKfs0, FemModel* fem,bool conserve_loads){


	/*solution : */
	Vec ug=NULL; 
	Vec uf=NULL; 
	Vec old_ug=NULL; 
	Vec old_uf=NULL; 
	DataSet* loads=NULL;

	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pf=NULL;
	int converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int numberofnodes;
	int min_mechanical_constraints;
	int max_nonlinear_iterations;

	/*parameters:*/
	int kflag,pflag;
	char* solver_string=NULL;
	int verbose=0;
	int dim;

	/*Recover parameters: */
	kflag=1; pflag=1;
	fem->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	fem->parameters->FindParam(&solver_string,SolverStringEnum);
	fem->parameters->FindParam(&dim,DimEnum);
	fem->parameters->FindParam(&verbose,VerboseEnum);
	fem->parameters->FindParam(&min_mechanical_constraints,MinMechanicalConstraintsEnum);
	fem->parameters->FindParam(&max_nonlinear_iterations,MaxNonlinearIterationsEnum);
	
	/*Were loads requested as output? : */
	if(conserve_loads){
		loads=fem->loads->Copy(); //protect loads from being modified by the solution
	}
	else{
		loads=fem->loads; //modify loads  in this solution
	}

	count=1;
	converged=0;

	/*Start non-linear iteration using input velocity: */
	GetSolutionFromInputsx(&ug, fem->elements, fem->nodes, fem->vertices, fem->loads, fem->materials, fem->parameters);
	Reducevectorgtofx(&uf, ug, fem->nodesets);

	for(;;){

		//save pointer to old velocity
		VecFree(&old_ug);old_ug=ug;
		VecFree(&old_uf);old_uf=uf;

		if (verbose) _printf_("   Generating matrices\n");
		//*Generate system matrices
		SystemMatricesx(&Kgg, &pg,fem->elements,fem->nodes,fem->vertices,loads,fem->materials,fem->parameters,kflag,pflag);

		if (verbose) _printf_("   Generating penalty matrices\n");
		//*Generate penalty system matrices
		PenaltySystemMatricesx(Kgg, pg,NULL,fem->elements,fem->nodes,fem->vertices,loads,fem->materials,fem->parameters,kflag,pflag);

		if (verbose) _printf_("   reducing matrix from g to f set\n");
		/*!Reduce matrix from g to f size:*/
		Reducematrixfromgtofx(&Kff,&Kfs,Kgg,fem->Gmn,fem->nodesets);

		/*Free ressources: */
		MatFree(&Kgg);
	
		if (verbose) _printf_("   reducing load from g to f set\n");
		/*!Reduce load from g to f size: */
		Reduceloadfromgtofx(&pf, pg, fem->Gmn, Kfs, fem->ys, fem->nodesets);

		//no need for pg and Kfs anymore 
		VecFree(&pg); 
		MatFree(&Kfs);

		/*Solve: */
		if (verbose) _printf_("   solving\n");
		Solverx(&uf, Kff, pf, old_uf, solver_string);

		//Merge back to g set
		if (verbose) _printf_("   merging solution from f to g set\n");
		Mergesolutionfromftogx(&ug, uf,fem->Gmn,fem->ys,fem->nodesets);

		//Update inputs using new solution:
		UpdateInputsFromSolutionx( fem->elements,fem->nodes, fem->vertices, fem->loads, fem->materials, fem->parameters,ug);

		//Deal with penalty loads
		if (verbose) _printf_("   penalty constraints\n");
		PenaltyConstraintsx(&constraints_converged, &num_unstable_constraints, fem->elements,fem->nodes,fem->vertices,loads,fem->materials,fem->parameters);

		if(verbose)_printf_("   number of unstable constraints: %i\n",num_unstable_constraints);

		/*Figure out if convergence is reached.*/
		convergence(&converged,Kff,pf,uf,old_uf,fem->parameters);
		MatFree(&Kff);VecFree(&pf);
		
		/*add converged to inputs: */
		UpdateInputsFromConstantx( fem->elements,fem->nodes, fem->vertices, fem->loads, fem->materials, fem->parameters,converged,ConvergedEnum);

		//rift convergence
		if (!constraints_converged) {
			if (converged){
				if (num_unstable_constraints <= min_mechanical_constraints) converged=1;
				else converged=0;
			}
		}

		/*Increase count: */
		count++;
		if(converged==1)break;
		if(count>=max_nonlinear_iterations){
			_printf_("   maximum number of iterations (%i) exceeded\n",max_nonlinear_iterations); 
			break;
		}
	}
	
	/*extrude if we are in 3D: */
	if (dim==3){
		if(verbose)_printf_("%s\n","extruding velocity and pressure in 3d...");
		ExtrudeInputx( fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters,VxEnum);
		ExtrudeInputx( fem->elements,fem->nodes,fem->vertices,fem->loads,fem->materials,fem->parameters,VyEnum);
	}

	//more output might be needed, when running in control
	if(pKff0){

		kflag=1; pflag=0; //stiffness generation only
	
		SystemMatricesx(&Kgg, &pg,fem->elements,fem->nodes,fem->vertices,loads,fem->materials,fem->parameters,kflag,pflag);
		Reducematrixfromgtofx(&Kff,&Kfs,Kgg,fem->Gmn,fem->nodesets);
		MatFree(&Kgg);VecFree(&pg);

	}

	/*Delete loads only if no ouput was requested: */
	if(!conserve_loads)delete loads;

	/*clean up*/
	VecFree(&uf);
	VecFree(&old_uf);
	VecFree(&old_ug);
	xfree((void**)&solver_string);
	
	/*Assign output pointers: */
	if(pug)*pug=ug;
	else VecFree(&ug);

	if(pKff0)*pKff0=Kff;
	if(pKfs0)*pKfs0=Kfs;

}
