/*!\file FemModel.c
 * \brief: implementation of the FemModel object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "../DataSet/DataSet.h"
#include "../modules/ModelProcessorx/ModelProcessorx.h"
#include "./objects.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

/*Object constructors and destructor*/
/*FUNCTION FemModel::constructor {{{1*/
FemModel::FemModel(ConstDataHandle IOMODEL,int in_solution_type,int* analyses, int nummodels){

	/*intermediary*/
	int i;
	IoModel* iomodel=NULL;
	int analysis_type;

	/*Initialize internal data: */
	this->nummodels=nummodels;
	this->solution_type=in_solution_type;
	analysis_counter=nummodels-1; //point to last analysis_type carried out.
	
	/*Dynamically allocate whatever is a list of length nummodels: */
	analysis_type_list=(int*)xmalloc(nummodels*sizeof(int));
	Rmg=(Mat*)xmalloc(nummodels*sizeof(Mat));
	Gmn=(Mat*)xmalloc(nummodels*sizeof(Mat));
	nodesets=(NodeSets**)xmalloc(nummodels*sizeof(NodeSets*));
	yg=(Vec*)xmalloc(nummodels*sizeof(Vec));
	ys=(Vec*)xmalloc(nummodels*sizeof(Vec));

	/*Initialize: */
	for(i=0;i<nummodels;i++)analysis_type_list[i]=analyses[i];
	for(i=0;i<nummodels;i++)Rmg[i]=NULL;
	for(i=0;i<nummodels;i++)Gmn[i]=NULL;
	for(i=0;i<nummodels;i++)nodesets[i]=NULL;
	for(i=0;i<nummodels;i++)yg[i]=NULL;
	for(i=0;i<nummodels;i++)ys[i]=NULL;

	_printf_("   fill model with matlab workspace data\n");
	iomodel = new IoModel(IOMODEL);

	for(i=0;i<nummodels;i++){

		analysis_type=analysis_type_list[i];
	
		_printf_("   create datasets:\n");
		CreateDataSets(&elements,&nodes,&vertices,&materials,&constraints,&loads,&parameters,iomodel,IOMODEL,solution_type,analysis_type,nummodels,i);

		_printf_("   create degrees of freedom: \n");
		VerticesDofx( &partition,&tpartition,vertices,parameters);
		NodesDofx(nodes,parameters);

		_printf_("   create single point constraints: \n");
		SpcNodesx( &yg[i], nodes,constraints,analysis_type); 

		_printf_("   create rigid body constraints:\n");
		MpcNodesx( &Rmg[i], nodes,constraints,analysis_type); 

		_printf_("   create node sets:\n");
		BuildNodeSetsx(&nodesets[i], nodes,analysis_type);

		_printf_("   reducing single point constraints vector:\n");
		Reducevectorgtosx(&ys[i], yg[i],nodesets[i]);

		_printf_("   normalizing rigid body constraints matrix:\n");
		NormalizeConstraintsx(&Gmn[i], Rmg[i],nodesets[i]);

		_printf_("   configuring element and loads:\n");
		ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters,i,analysis_type);
	}

	_printf_("   free ressources:\n");
	delete iomodel;
}
/*}}}1*/
/*FUNCTION FemModel::destructor {{{1*/
FemModel::~FemModel(){

	/*Intermediary*/
	int i;

	/*Delete all the datasets: */
	xfree((void**)&analysis_type_list);
	delete elements;
	delete nodes;
	delete vertices;
	delete constraints;
	delete loads;
	delete materials;
	delete parameters;
	delete partition;
	delete tpartition;

	for(i=0;i<nummodels;i++){
		Mat temp_Rmg=Rmg[i];
		MatFree(&temp_Rmg);
		Mat temp_Gmn=Gmn[i];
		MatFree(&temp_Gmn);
		NodeSets* temp_nodesets=nodesets[i];
		delete nodesets;
		Vec temp_yg=yg[i];
		VecFree(&temp_yg);
		Vec temp_ys=ys[i];
		VecFree(&temp_ys);
	}

	/*Delete dynamically allocated arrays: */
	delete Rmg;
	delete Gmn;
	delete nodesets;
	delete yg;
	delete ys;

}
/*}}}1*/

/*Object management*/
/*FUNCTION FemModel::Echo {{{1*/
void FemModel::Echo(void){

	printf("FemModel echo: \n");
	printf("   number of fem models: %i\n",nummodels);
	printf("   analysis_type_list: \n");
	for(int i=0;i<nummodels;i++)printf("     %i: %s\n",i,EnumAsString(analysis_type_list[i]));
	printf("   current analysis_type: \n");
	printf("     %i: %s\n",analysis_counter,EnumAsString(analysis_type_list[analysis_counter]));


}
/*}}}*/

/*Numerics: */
/*FUNCTION FemModel::GetCurrentAnalysis {{{1*/
int FemModel::GetCurrentAnalysis(){
	return analysis_type_list[analysis_counter];
}
/*}}}1*/
/*FUNCTION FemModel::SetCurrentAnalysis {{{1*/
void FemModel::SetCurrentAnalysis(int analysis_type){
	int found=-1;
	for(int i=0;i<nummodels;i++){
		if (analysis_type_list[i]==analysis_type){
			found=i;
			break;
		}
	}
	if(found!=-1) analysis_counter=found;
	else ISSMERROR("Could not find analysis_type %s in list of FemModel analyses",EnumAsString(analysis_type));
}
/*}}}1*/

