/*!\file: diagnostic_core.cpp
 * \brief: core of the diagnostic solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"

void diagnostic_core(FemModel* femmodel){

	/*parameters: */
	int  verbose=0;
	bool qmu_analysis=false;
	int  dim=-1;
	bool ishutter=false;
	bool ismacayealpattyn=false;
	bool isstokes=false;
	double stokesreconditioning;

	//first recover parameters needed to drive the solution
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&ishutter,IsHutterEnum);
	femmodel->parameters->FindParam(&ismacayealpattyn,IsMacAyealPattynEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	femmodel->parameters->FindParam(&qmu_analysis,QmuAnalysisEnum);

	/*for qmu analysis, reinitialize velocity so that fake sensitivities do not show up as a result of a different restart of the convergence at each trial.*/
	if(qmu_analysis){
		ReinitializeInputx(femmodel,VxEnum,QmuVxEnum);
		ReinitializeInputx(femmodel,VyEnum,QmuVyEnum);
		ReinitializeInputx(femmodel,VzEnum,QmuVzEnum);
		ReinitializeInputx(femmodel,PressureEnum,QmuPressureEnum);
	}

	/*Compute slopes: */
	if(ishutter){
		slope_core(femmodel,SurfaceXAnalysisEnum);
		slope_core(femmodel,SurfaceYAnalysisEnum);
	}
	if(isstokes){
		slope_core(femmodel,BedSlopeXAnalysisEnum);
		slope_core(femmodel,BedSlopeYAnalysisEnum);
	}
		
	if(ishutter){
			
		if(verbose)_printf_("%s\n"," computing hutter velocities...");
		solve_linear(NULL,femmodel,DiagnosticAnalysisEnum,HutterAnalysisEnum);
		
		if (ismacayealpattyn) ResetBoundaryConditions(femmodel,DiagnosticAnalysisEnum,HorizAnalysisEnum);
	}

	if (ismacayealpattyn){
		
		if(verbose)_printf_("%s\n"," computing horizontal velocities...");
		diagnostic_core_nonlinear(NULL,NULL,NULL,femmodel,false,DiagnosticAnalysisEnum,HorizAnalysisEnum); //false means we expose loads to changes inside the solution
	}
	
	
	if (dim==3){

		if(verbose)_printf_("%s\n"," computing vertical velocities...");
		solve_linear(NULL,femmodel,DiagnosticAnalysisEnum,VertAnalysisEnum);

		if (isstokes){

			//"recondition" pressure computed previously:
			DuplicateInputx(femmodel,PressureEnum,PressureStokesEnum); ScaleInputx(femmmodel,PressureStokesEnum,1.0/stokesreconditioning);

			if(verbose)_printf_("%s\n"," update boundary conditions for stokes using velocities previously computed...");
			GetSolutionFromInputsx( &ug, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,  femmodel->parameters, DiagnosticAnalysisEnum,StokesAnalysisEnum);
			ResetBoundaryConditions(femmodel,DiagnosticAnalysisEnum,StokesAnalysisEnum);

			if(verbose)_printf_("%s\n"," computing stokes velocities and pressure ...");
			diagnostic_core_nonlinear(NULL,NULL,NULL,NULL,fem_ds,DiagnosticAnalysisEnum,StokesAnalysisEnum);
		}
	}
}
