/*!\file Beam.c
 * \brief: implementation of the Beam object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
#include "../../include/include.h"

/*Object constructors and destructor*/
/*FUNCTION Beam::Beam(){{{1*/
Beam::Beam(){
	this->inputs=NULL;
	this->parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION Beam::~Beam(){{{1*/
Beam::~Beam(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object management*/
/*FUNCTION Beam::copy{{{1*/
Object* Beam::copy() {

	int i;
	Beam* beam=NULL;

	beam=new Beam();

	/*copy fields: */
	beam->id=this->id;
	if(this->inputs){
		beam->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		beam->inputs=new Inputs();
	}
	/*point parameters: */
	beam->parameters=this->parameters;

	/*pointers: */
	beam->nodes=(Node**)xmalloc(2*sizeof(Node*)); 
	for(i=0;i<2;i++)beam->nodes[i]=this->nodes[i];
	beam->matice=this->matice;
	beam->matpar=this->matpar;

	return beam;
}
/*}}}*/
/*FUNCTION Beam::DeepEcho{{{1*/
void Beam::DeepEcho(void){

	int i;

	printf("Beam:\n");
	printf("   id: %i\n",id);
	for(i=0;i<2;i++){
		nodes[i]->DeepEcho();
	}
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Beam::Demarshall{{{1*/
void  Beam::Demarshall(char** pmarshalled_dataset){
	ISSMERROR("not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Echo {{{1*/
void Beam::Echo(void){

	int i;

	printf("Beam:\n");
	printf("   id: %i\n",id);
	for(i=0;i<2;i++){
		nodes[i]->DeepEcho();
	}
	matice->DeepEcho();
	matpar->DeepEcho();
	printf("   parameters\n");
	parameters->Echo();
	printf("   inputs\n");
	inputs->Echo();

	return;
}
/*}}}*/
/*FUNCTION Beam::IsInput{{{1*/
bool Beam::IsInput(int name){
	if (name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum){
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Beam::Marshall{{{1*/
void  Beam::Marshall(char** pmarshalled_dataset){
	ISSMERROR("not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MarshallSize{{{1*/
int   Beam::MarshallSize(){
	ISSMERROR("not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::UpdateInputsFromSolution {{{1*/
void  Beam::UpdateInputsFromSolution(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetSolutionFromInputs(Vec solution,  int analysis_type,int sub_analysis_type);{{{1*/
void  Beam::GetSolutionFromInputs(Vec solution,  int analysis_type,int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Object functions*/
/*FUNCTION Beam::ComputeBasalStress{{{1*/
void  Beam::ComputeBasalStress(Vec eps,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Beam::ComputePressure{{{1*/
void  Beam::ComputePressure(Vec p_g,int analysis_type,int sub_analysis_type){

	int i;
	const int numgrids=2;
	int doflist[numgrids];
	double pressure[numgrids];
	double surface[numgrids];
	double rho_ice,g;
	double xyz_list[numgrids][3];
	double gauss[numgrids][numgrids]={{1,0},{0,1}};

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
        
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*recover value of surface at gauss points (0,1) and (1,0): */
	inputs->GetParameterValues(&surface[0],&gauss[0][0],2,SurfaceEnum);
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(p_g,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Beam::ComputeStrainRate{{{1*/
void  Beam::ComputeStrainRate(Vec eps,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Beam::CostFunction{{{1*/
double Beam::CostFunction(int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::CreateKMatrix{{{1*/
void  Beam::CreateKMatrix(Mat Kgg,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum) {
	
		if (sub_analysis_type==HutterAnalysisEnum) {

			CreateKMatrixDiagnosticHutter( Kgg,analysis_type,sub_analysis_type);
		}
		else 
			ISSMERROR("%s%i%s\n","sub_analysis_type: ",sub_analysis_type," not supported yet");
	}
	else{
		ISSMERROR("%s%i%s\n","analysis: ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Beam::CreateKMatrixDiagnosticHutter{{{1*/

void  Beam::CreateKMatrixDiagnosticHutter(Mat Kgg,int analysis_type,int sub_analysis_type){
	
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    Ke_gg[numdofs][numdofs]={{0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0}};
	int       numberofdofspernode;
	bool onbed;
	bool onsurface;
	int connectivity[2];
	double one0,one1;
	
	connectivity[0]=nodes[0]->GetConnectivity();
	connectivity[1]=nodes[1]->GetConnectivity();
	
	one0=1/(double)connectivity[0];
	one1=1/(double)connectivity[1];
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	GetDofList(&doflist[0],&numberofdofspernode);

	if (onbed){
		Ke_gg[0][0]=one0;
		Ke_gg[1][1]=one0;
		Ke_gg[2][0]=-2*one1;
		Ke_gg[2][2]=2*one1;
		Ke_gg[3][1]=-2*one1;
		Ke_gg[3][3]=2*one1;
	}
	else if (onsurface){
		Ke_gg[2][0]=-one1;
		Ke_gg[2][2]=one1;
		Ke_gg[3][1]=-one1;
		Ke_gg[3][3]=one1;
	}
	else{ //node is on two horizontal layers and beams include the values only once, so the have to use half of the connectivity
		Ke_gg[2][0]=-2*one1;
		Ke_gg[2][2]=2*one1;
		Ke_gg[3][1]=-2*one1;
		Ke_gg[3][3]=2*one1;
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Beam::CreatePVector{{{1*/
void  Beam::CreatePVector(Vec pg,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum) {
		if (sub_analysis_type==HutterAnalysisEnum) {
			CreatePVectorDiagnosticHutter( pg,analysis_type,sub_analysis_type);
		}
		else
			ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}
	else{
		ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Beam::CreatePVectorDiagnosticHutter{{{1*/

void Beam::CreatePVectorDiagnosticHutter( Vec pg,  int analysis_type,int sub_analysis_type){

	int i,j,k;
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    pe_g[4]={0,0,0,0};
	double    pe_g_gaussian[4]={0,0,0,0};
	int       found=0;
	int       dofs[1]={0};
	double    xyz_list[numgrids][3];
	double    z_list[numgrids];
	double    constant_part;
	int       numberofdofspernode;

	/*gaussian points: */
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double   gauss_coord;
	double*  gauss_weights=NULL;
	double   gauss_weight;
	double   gauss1[2]={1,0};
	int      ig;
	double   l1l2[2];
	double   slope[2];
	double   slope2;

	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;
	double   surface,thickness;
	
	bool onbed;
	bool onsurface;
	int  connectivity[2];

	/*recover doflist: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	inputs->GetParameterValue(&onsurface,ElementOnSurfaceEnum);

	/*recover parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();

	//recover extra inputs
	inputs->GetParameterValue(&slope[0],&gauss1[0],SurfaceSlopeXEnum);
	inputs->GetParameterValue(&slope[1],&gauss1[0],SurfaceSlopeYEnum);
	inputs->GetParameterValue(&surface,&gauss1[0],SurfaceEnum);
	inputs->GetParameterValue(&thickness,&gauss1[0],ThicknessEnum);

	connectivity[0]=nodes[0]->GetConnectivity();
	connectivity[1]=nodes[1]->GetConnectivity();

	//Get all element grid data:
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	for(i=0;i<numgrids;i++)z_list[i]=xyz_list[i][2];
	
	//compute slope2 slopex and slopy
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//%compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	GaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	//Start  looping on the number of gaussian points:
	for(ig=0;ig<num_gauss;ig++){

		//Pick up the gaussian point and its weight:
		gauss_weight=gauss_weights[ig];
		gauss_coord=segment_gauss_coord[ig];

		//compute constant_part
		GetParameterValue(&z_g, &z_list[0],gauss_coord);

		//Get Jacobian determinant:
		GetJacobianDeterminant(&Jdet, &z_list[0],gauss_coord);
		
		/*Add contribution*/
		if (onsurface){
			for(j=0;j<NDOF2;j++){
				pe_g_gaussian[NDOF2+j]=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight/(double)connectivity[1];
			}
		}
		else{//connectivity is too large, should take only half on it
			for(j=0;j<NDOF2;j++){
				pe_g_gaussian[NDOF2+j]=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight*2/(double)connectivity[1];
			}
		}
		
		//add pe_gaussian vector to pe:
		for(j=0;j<numdofs;j++){
			pe_g[j]+=pe_g_gaussian[j];
		}
	} //for(ig=0;ig<num_gauss;ig++)

	//Deal with lower surface
	if (onbed){

		//compute ub
		constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*thickness;
		ub=constant_part*slope[0];
		vb=constant_part*slope[1];

		//Add to pe: 
		pe_g[0]+=ub/(double)connectivity[0];
		pe_g[1]+=vb/(double)connectivity[0];
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&segment_gauss_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Beam::Du{{{1*/
void  Beam::Du(Vec,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Enum{{{1*/
int Beam::Enum(void){

	return BeamEnum;

}
/*}}}*/
/*FUNCTION Beam::GetBedList{{{1*/
void  Beam::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetDofList{{{1*/
void  Beam::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	for(i=0;i<2;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Beam::GetDofList1{{{1*/
void  Beam::GetDofList1(int* doflist){

	int i;
	for(i=0;i<2;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Beam::Id{{{1*/
int    Beam::Id(void){ return id; }
/*}}}*/
/*FUNCTION Beam::GetJacobianDeterminant{{{1*/
void Beam::GetJacobianDeterminant(double* pJdet,double* z_list, double gauss_coord){


	double Jdet;

	Jdet=1.0/2.0*(z_list[1]-z_list[0]);

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}
	
	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Beam::GetMatPar{{{1*/
void* Beam::GetMatPar(){

	return matpar;
}
/*}}}*/
/*FUNCTION Beam::GetNodalFunctions{{{1*/
void Beam::GetNodalFunctions(double* l1l2, double gauss_coord){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2[0]=-0.5*gauss_coord+0.5;

	/*Second nodal function: */
	l1l2[1]=+0.5*gauss_coord+0.5;
}
/*}}}*/
/*FUNCTION Beam::GetNodes{{{1*/
void  Beam::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=NULL;

	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Beam::GetOnBed{{{1*/
bool   Beam::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetParameterValue{{{1*/
void Beam::GetParameterValue(double* pvalue, double* value_list,double gauss_coord){

	double l1l2[2];
	
	GetNodalFunctions(&l1l2[0],gauss_coord);

	*pvalue=l1l2[0]*value_list[0]+l1l2[1]*value_list[1];
}
/*}}}*/
/*FUNCTION Beam::GetShelf{{{1*/
bool   Beam::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetThicknessList{{{1*/
void  Beam::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Gradj{{{1*/
void  Beam::Gradj(Vec, int, int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GradjB{{{1*/
void  Beam::GradjB(Vec, int, int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GradjDrag{{{1*/
void  Beam::GradjDrag(Vec, int,int ){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MassFlux{{{1*/
double Beam::MassFlux( double* segment,double* ug){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Misfit{{{1*/
double Beam::Misfit(int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MyRank{{{1*/
int    Beam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Beam::SurfaceArea{{{1*/
double Beam::SurfaceArea(int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::SetClone {{{1*/
void  Beam::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Beam::UpdateInputsFromVector(double* vector, int name, int type);{{{1*/
void  Beam::UpdateInputsFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;
	switch(type){

		case VertexEnum:

			/*New PentaVertexInpu*/
			double values[2];

			/*Get values on the 6 vertices*/
			for (int i=0;i<2;i++){
				values[i]=vector[nodes[i]->GetVertexDof()];
			}

			/*update input*/
			this->inputs->AddInput(new BeamVertexInput(name,values));
			return;

		default:

			ISSMERROR("type %i (%s) not implemented yet",type,EnumAsString(type));
	}
}
/*}}}*/
/*FUNCTION Beam::UpdateInputsFromVector(int* vector, int name, int type);{{{1*/
void  Beam::UpdateInputsFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::UpdateInputsFromVector(bool* vector, int name, int type);{{{1*/
void  Beam::UpdateInputsFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes, int enum_type){{{1*/
void  Beam::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes, int enum_type){

	int    i;
	Input *input       = NULL;
	bool   found       = false;
	int    numrows;
	int    numvertices;
	int    numnodes;

	/*Recover counter: */
	numrows=*pnumrows;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	for (i=0;i<this->inputs->Size();i++){
		input=(Input*)this->inputs->GetObjectByOffset(i);
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*Ok, the input looked for does not exist. No problem. Just be sure numvertices and numnodes
		 * are 0, so that they do not increase the size of the patches array. Also, do not increase 
		 * the counter, as this element has nothing to do with results: */
		numvertices=0;
		numnodes=0;
	}
	else{
		/*Ok, we found an input with the correct enum_type. Ask it to tell us how many nodal values it 
		 * holds. : */
		numnodes=input->PatchSize();
		/*We know the number of vertices from this element: */
		numvertices=2;
		/*Increase counter, because this element does hold a result in its inputs: */
		numrows++;
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=numvertices;
	*pnumnodes=numnodes;
	
}
/*}}}*/
/*FUNCTION Beam::PatchFill(int* pcount, double* patches,int numcols,int max_vertices,int enum_type){{{1*/
void  Beam::PatchFill(int* pcount, double* patches,int numcols,int max_vertices,int enum_type){

	/*A patch is made of the following information: 
	 * element_id  interpolation_type  vertex_ids    values. 
	 * For example: 

	 1 P0  1 2       4.5 NaN 
	 2 P1  1 3       4.5 3.2
	 3 P0  1 5       5.5 NaN
	 4 P1  2 4       4.5 3.2

	 Here, we will provide the nodal values, after having processed them, can provide: id, and vertices ids. 
	 Then go find an input with enum_type. If we find it, fill in the values at the nodal points. 
	 If we don't find an input, get out of here doing nothing, as this element is not involved in 
	 outputting results. 
	 */

	int      i;
	Input   *input      = NULL;
	bool     found      = false;
	int      count      = 0;
	double  *this_patch = NULL;


	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	for (i=0;i<this->inputs->Size();i++){
		input=(Input*)this->inputs->GetObjectByOffset(i);
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*Ok, the input looked for does not exist. No problem. Just return :*/
	}
	else{
		
		/*First, recover the patches row where we will plug in the information: */
		count=*pcount;

		/*set this_patch to point at the beginning of the patches row, where we will plug our information :*/
		this_patch=patches+numcols*count;

		/*Fill in id: */
		this_patch[0]=this->id;

		/*Fill in vertices ids: */
		for(i=0;i<2;i++) this_patch[2+i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

		/*We found the input, get it to fill the interpolation type, and the nodal values:*/
		input->PatchFill(this_patch,max_vertices,this->parameters);
		
		/*Increase counter, so that next time, we don't point to the same patches row: */
		count++;

		/*Assign output pointers:*/
		*pcount=count;
	}
}
/*}}}*/
