/*!\file Sing.c
 * \brief: implementation of the Sing object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
#include "../../include/include.h"

/*Object constructors and destructor*/
/*FUNCTION Sing::Sing(){{{1*/
Sing::Sing(){
	this->inputs=NULL;
	this->parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION Sing::~Sing(){{{1*/
Sing::~Sing(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object management*/
/*FUNCTION Sing::Configure {{{1*/
void  Sing::Configure(DataSet* elementsin,DataSet* loadsin, DataSet* nodesin, DataSet* materialsin, Parameters* parametersin,int analysis_counter){

	int i;
	
	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	hmatice.configure(materialsin);
	hmatpar.configure(materialsin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Sing::copy {{{1*/
Object* Sing::copy() {
	
	return new Sing(this->id,&this->hnodes,&this->hmatice,&this->hmatpar,this->parameters,this->inputs);

}
/*}}}*/
/*FUNCTION Sing::DeepEcho {{{1*/
void Sing::DeepEcho(void){

	printf("Sing:\n");
	printf("   id: %i\n",id);
	hnodes.DeepEcho();
	hmatice.DeepEcho();
	hmatpar.DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Sing::Demarshall {{{1*/
void  Sing::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	hmatice.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);

	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Sing::Echo{{{1*/

void Sing::Echo(void){

	printf("Sing:\n");
	printf("   id: %i\n",id);
	hnodes.Echo();
	hmatice.Echo();
	hmatpar.Echo();
	printf("   parameters\n");
	parameters->Echo();
	printf("   inputs\n");
	inputs->Echo();
}
/*}}}*/
/*FUNCTION Sing::IsInput{{{1*/
bool Sing::IsInput(int name){
	if (name==SurfaceSlopexEnum ||
				name==SurfaceSlopeyEnum){
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Sing::Marshall {{{1*/
void  Sing::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Sing: */
	enum_type=SingEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Sing data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	
	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	hmatice.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	
	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Sing::MashallSize {{{1*/
int   Sing::MarshallSize(){

	return sizeof(id)
		+hnodes.MarshallSize()
		+hmatice.MarshallSize()
		+hmatpar.MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Sing::UpdateInputsFromSolution {{{1*/
void  Sing::UpdateInputsFromSolution(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetSolutionFromInputs(Vec solution,  int analysis_type,int sub_analysis_type);{{{1*/
void  Sing::GetSolutionFromInputs(Vec solution,  int analysis_type,int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
		
/*Object functions*/
/*FUNCTION Sing::ComputeBasalStress {{{1*/
void  Sing::ComputeBasalStress(Vec p_g,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Sing::ComputePressure {{{1*/
void  Sing::ComputePressure(Vec p_g,int analysis_type,int sub_analysis_type){

	int    dof;
	double pressure;
	double thickness;
	double rho_ice,g;

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&dof);

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValue(&thickness,ThicknessEnum);
	pressure=rho_ice*g*thickness;
	
	/*plug local pressure values into global pressure vector: */
	VecSetValue(p_g,dof,pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Sing::ComputeStrainRate {{{1*/
void  Sing::ComputeStrainRate(Vec p_g,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Sing::CostFunction {{{1*/
double Sing::CostFunction( int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::CreateKMatrix {{{1*/

void  Sing::CreateKMatrix(Mat Kgg,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticAnalysisEnum) && (sub_analysis_type==HutterAnalysisEnum)){

		CreateKMatrixDiagnosticHutter( Kgg,analysis_type,sub_analysis_type);

	}
	else{
		ISSMERROR("%s%i%s%i%s\n","analysis: ",analysis_type," and sub_analysis_type: ",sub_analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Sing::CreateKMatrixDiagnosticHutter {{{1*/
void  Sing::CreateKMatrixDiagnosticHutter(Mat Kgg,int analysis_type,int sub_analysis_type){
	
	const int numgrids=1;
	const int NDOF2=2;
	const int numdofs=numgrids*NDOF2;
	double Ke_gg[numdofs][numdofs]={{1,0},{0,1}};
	int    doflist[numdofs];
	int    numberofdofspernode;
	int    connectivity;

	/*dynamic objects pointed to by hooks: */
	Node**  node=NULL;

	/*recover objects from hooks: */
	node=(Node**)hnodes.deliverp();

	/*Find connectivity of the node and divide Ke_gg by this connectivity*/
	connectivity=node[0]->GetConnectivity();
	Ke_gg[0][0]=1/(double)connectivity;
	Ke_gg[1][1]=1/(double)connectivity;

	GetDofList(&doflist[0],&numberofdofspernode);

	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Sing::CreatePVector {{{1*/
void  Sing::CreatePVector(Vec pg,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticAnalysisEnum) && (sub_analysis_type==HutterAnalysisEnum)){
	
			CreatePVectorDiagnosticHutter( pg,analysis_type,sub_analysis_type);

	}
	else{
		ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Sing::CreatePVectorDiagnosticHutter {{{1*/
void Sing::CreatePVectorDiagnosticHutter( Vec pg,  int analysis_type,int sub_analysis_type){
	
	const int numgrids=1;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	int       dofs[1]={0};
	int       found=0;
	double    slope[2];
	double    slope2;
	double    pe_g[numdofs]={0,0};
	double    ub,vb;
	double    constant_part;
	int       numberofdofspernode;
	double    rho_ice,gravity,n,B;
	double    thickness;
	int       connectivity;

	/*dynamic objects pointed to by hooks: */
	Node**  node=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	node=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	inputs->GetParameterValue(&slope[0],SurfaceSlopexEnum);
	inputs->GetParameterValue(&slope[1],SurfaceSlopeyEnum);

	GetDofList(&doflist[0],&numberofdofspernode);

	//Get connectivity of the node
	connectivity=node[0]->GetConnectivity();

	//compute slope2 
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//compute ub
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();
	inputs->GetParameterValue(&thickness,ThicknessEnum);

	ub=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[0];
	vb=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[1];

	//compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	pe_g[0]=(ub-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[0])/(double)connectivity;
	pe_g[1]=(vb-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[1])/(double)connectivity;

	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Sing::Du {{{1*/
void  Sing::Du(Vec,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Enum {{{1*/
int Sing::Enum(void){

	return SingEnum;

}
/*}}}*/
/*FUNCTION Sing::GetBedList {{{1*/
void  Sing::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetDofList {{{1*/
void  Sing::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	nodes[0]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
	for(i=0;i<numberofdofspernode;i++){
		doflist[i]=doflist_per_node[i];
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Sing::GetDofList1 {{{1*/
void  Sing::GetDofList1(int* doflist){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	int i;
	doflist[0]=nodes[0]->GetDofList1();

}
/*}}}*/
/*FUNCTION Sing::Id {{{1*/
int    Sing::Id(void){ return id; }
/*}}}*/
/*FUNCTION Sing::GetMatPar {{{1*/
void* Sing::GetMatPar(){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	return matpar;
}
/*}}}*/
/*FUNCTION Sing::GetNodes {{{1*/
void  Sing::GetNodes(void** vpnodes){
	
	Node** pnodes=NULL;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	
	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	pnodes[0]=nodes[0];
}
/*}}}*/
/*FUNCTION Sing::GetOnBed {{{1*/
bool   Sing::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetShelf {{{1*/
bool   Sing::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetThicknessList {{{1*/
void  Sing::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Gradj {{{1*/
void  Sing::Gradj(Vec,  int, int ,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GradB {{{1*/
void  Sing::GradjB(Vec,  int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GradjDrag {{{1*/
void  Sing::GradjDrag(Vec,  int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MassFlux {{{1*/
double Sing::MassFlux( double* segment,double* ug){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Misfit {{{1*/
double Sing::Misfit( int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MyRank {{{1*/
int    Sing::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Sing::SurfaceArea {{{1*/
double Sing::SurfaceArea( int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::SetClone {{{1*/
void  Sing::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Sing::UpdateInputsFromVector(double* vector, int name, int type);{{{1*/
void  Sing::UpdateInputsFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;
	Node*  node=NULL;

	switch(type){

		case VertexEnum:

			/*New SingVertexInpu*/
			double value;

			/*Get values on the 6 vertices*/
			node=(Node*)this->hnodes.delivers();
			value=vector[node->GetVertexDof()];

			/*update input*/
			this->inputs->AddInput(new SingVertexInput(name,value));
			return;

		default:
			ISSMERROR("type %i (%s) not implemented yet",type,EnumAsString(type));
	}
}
/*}}}*/
/*FUNCTION Sing::UpdateInputsFromVector(int* vector, int name, int type);{{{1*/
void  Sing::UpdateInputsFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::UpdateInputsFromVector(bool* vector, int name, int type);{{{1*/
void  Sing::UpdateInputsFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes, int enum_type){{{1*/
void  Sing::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes, int enum_type){

	int    i;
	Input *input       = NULL;
	bool   found       = false;
	int    numrows;
	int    numvertices;
	int    numnodes;

	/*Recover counter: */
	numrows=*pnumrows;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	for (i=0;i<this->inputs->Size();i++){
		input=(Input*)this->inputs->GetObjectByOffset(i);
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*Ok, the input looked for does not exist. No problem. Just be sure numvertices and numnodes
		 * are 0, so that they do not increase the size of the patches array. Also, do not increase 
		 * the counter, as this element has nothing to do with results: */
		numvertices=0;
		numnodes=0;
	}
	else{
		/*Ok, we found an input with the correct enum_type. Ask it to tell us how many nodal values it 
		 * holds. : */
		numnodes=input->PatchSize();
		/*We know the number of vertices from this element: */
		numvertices=1;
		/*Increase counter, because this element does hold a result in its inputs: */
		numrows++;
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=numvertices;
	*pnumnodes=numnodes;
}
	
/*FUNCTION Sing::PatchFill(int* pcount, double* patches,int numcols,int max_vertices,int enum_type){{{1*/
void  Sing::PatchFill(int* pcount, double* patches,int numcols,int max_vertices,int enum_type){

	/*A patch is made of the following information: 
	 * element_id  interpolation_type  vertex_ids    values. 
	 * For example: 

	 1 P0  1       4.5 
	 2 P1  3       4.5
	 3 P0  5       5.5
	 4 P1  4       4.5

	 Here, we will provide the nodal values, after having processed them, can provide: id, and vertices ids. 
	 Then go find an input with enum_type. If we find it, fill in the values at the nodal points. 
	 If we don't find an input, get out of here doing nothing, as this element is not involved in 
	 outputting results. 
	 */

	int      i;
	Input   *input      = NULL;
	bool     found      = false;
	int      count      = 0;
	Node   **nodes      = NULL;
	double  *this_patch = NULL;


	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	for (i=0;i<this->inputs->Size();i++){
		input=(Input*)this->inputs->GetObjectByOffset(i);
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*Ok, the input looked for does not exist. No problem. Just return :*/
	}
	else{
		
		/*First, recover the patches row where we will plug in the information: */
		count=*pcount;

		/*Recover nodes: */
		nodes=(Node**)hnodes.deliverp();

		/*set this_patch to point at the beginning of the patches row, where we will plug our information :*/
		this_patch=patches+numcols*count;

		/*Fill in id: */
		this_patch[0]=this->id;

		/*Fill in vertices ids: */
		for(i=0;i<1;i++) this_patch[2+i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

		/*We found the input, get it to fill the interpolation type, and the nodal values:*/
		input->PatchFill(this_patch,max_vertices,this->parameters);
		
		/*Increase counter, so that next time, we don't point to the same patches row: */
		count++;

		/*Assign output pointers:*/
		*pcount=count;
	}
}
/*}}}*/
