/*!\file: diagnostic_core.cpp
 * \brief: core of the diagnostic solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"

Results* diagnostic_core(FemModel* femmodel){

	extern int my_rank;
	int        dummy;

	/*output: */
	Results* results=NULL;
	Result*  result=NULL;

	/*solutions: */
	Vec ug=NULL;
	Vec ug_horiz=NULL;
	Vec ug_vert=NULL;
	Vec ug_stokes=NULL;
	Vec pg=NULL;
	Vec riftproperties=NULL;
	double* u_g_initial=NULL;
	double* vx=NULL;
	double* vy=NULL;
	double* vz=NULL;

	/*flags: */
	int verbose=0;
	bool qmu_analysis=false;
	int dim=-1;
	bool ishutter=false;
	bool ismacayealpattyn=false;
	bool isstokes=false;
	int numberofdofspernode_sl;
	int numberofdofspernode_dh;
	int numberofdofspernode_ds;
	int numberofnodes;
	int numrifts=0;

	/*slopes: */
	Vec surfaceslopex=NULL;
	Vec surfaceslopey=NULL;
	Vec bedslopex=NULL;
	Vec bedslopey=NULL;

	double stokesreconditioning;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};
	double* dofset=NULL;

	//initialize results
	results=new Results();

	//first recover parameters needed to drive the solution
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&ishutter,IsHutterEnum);
	femmodel->parameters->FindParam(&ismacayealpattyn,IsMacAyealPattynEnum);
	femmodel->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&stokesreconditioning,StokesReconditioningEnum);
	femmodel->parameters->FindParam(&numrifts,NumRiftsEnum);
	femmodel->parameters->FindParam(&qmu_analysis,QmuAnalysisEnum);

	//specific parameters for specific models
	femmodel->parameters->FindParam(&numberofdofspernode_dh,NumberOfDofsPerNodeEnum);
	femmodel->parameters->FindParam(&numberofdofspernode_sl,NumberOfDofsPerNodeEnum);
	femmodel->parameters->FindParam(&numberofdofspernode_ds,NumberOfDofsPerNodeEnum);

	/*for qmu analysis, reinitialize velocity so that fake sensitivities do not show up as a result of a different restart of the convergence at each trial.*/
	if(qmu_analysis){
		ReinitializeInputx(femmodel,VxEnum,QmuVxEnum);
		ReinitializeInputx(femmodel,VyEnum,QmuVyEnum);
		ReinitializeInputx(femmodel,VzEnum,QmuVzEnum);
	}

	/*Compute slopes: */
	slope_core(&surfaceslopex,&surfaceslopey,femmodel,SurfaceAnalysisEnum);
	slope_core(&bedslopex,&bedslopey,femmodel,BedAnalysisEnum);
		
	/*Update: */
	model->UpdateInputsFromVector(surfaceslopex,SurfaceSlopexEnum,VertexEnum);
	model->UpdateInputsFromVector(surfaceslopey,SurfaceSlopeyEnum,VertexEnum);
	model->UpdateInputsFromVector(bedslopex,BedSlopexEnum,VertexEnum);
	model->UpdateInputsFromVector(bedslopey,BedSlopeyEnum,VertexEnum);

	
	if(ishutter){
			
		if(verbose)_printf_("%s\n"," computing hutter velocities...");
		diagnostic_core_linear(&ug,fem_dhu,DiagnosticAnalysisEnum,HutterAnalysisEnum);

		if(verbose)_printf_("%s\n"," computing pressure according to MacAyeal...");
		ComputePressurex(&pg,fem_dhu->elements,fem_dhu->nodes, fem_dhu->vertices,fem_dhu->loads,fem_dhu->materials,fem_dhu->parameters,DiagnosticAnalysisEnum,HutterAnalysisEnum);

		if(verbose)_printf_("%s\n"," update boundary conditions for macyeal pattyn using hutter results...");
		if (ismacayealpattyn){
			VecFree(&fem_dh->yg->vector); VecFree(&fem_dh->ys);
			VecDuplicatePatch(&fem_dh->yg->vector,ug);
			Reducevectorgtosx(&fem_dh->ys,fem_dh->yg->vector,fem_dh->nodesets);
		}

	}

	if (ismacayealpattyn){
		
		if(verbose)_printf_("%s\n"," computing horizontal velocities...");
		diagnostic_core_nonlinear(&ug,NULL,NULL,fem_dh->loads,fem_dh,DiagnosticAnalysisEnum,HorizAnalysisEnum);

		if(dim==2){
			if(verbose)_printf_("%s\n"," computing pressure according to MacAyeal...");
			ComputePressurex(&pg,fem_dh->elements,fem_dh->nodes, fem_dh->vertices,fem_dh->loads,fem_dh->materials,fem_dh->parameters,DiagnosticAnalysisEnum,HorizAnalysisEnum);
		}

	}
	
	
	if (dim==3){

		if(verbose)_printf_("%s\n"," extruding horizontal velocities...");
		FieldExtrudex(ug,fem_dh->elements,fem_dh->nodes, fem_dh->vertices,fem_dh->loads,fem_dh->materials,fem_dh->parameters,"velocity",1);
		SplitSolutionVectorx(ug,numberofnodes,numberofdofspernode_dh,&vx,&vy);
		model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
		model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
		
		if(verbose)_printf_("%s\n"," computing vertical velocities...");
		diagnostic_core_linear(&ug_vert,fem_dv,DiagnosticAnalysisEnum,VertAnalysisEnum);
		model->UpdateInputsFromVector(ug_vert,VzEnum,VertexEnum);

		if(verbose)_printf_("%s\n"," computing pressure according to Pattyn...");
		ComputePressurex(&pg,fem_dh->elements, fem_dh->nodes, fem_dh->vertices,fem_dh->loads,  fem_dh->materials,fem_dh->parameters,DiagnosticAnalysisEnum,HorizAnalysisEnum);
		model->UpdateInputsFromVector(pg,PressureEnum,VertexEnum);
		
		if (isstokes){

			//"recondition" pressure 
			VecScale(pg,1.0/stokesreconditioning);

			if(verbose)_printf_("%s\n"," update boundary conditions for stokes using velocities previously computed...");
			xfree((void**)&dofset);dofset=dofsetgen(2,dof01,4,numberofnodes*4); VecMerge(fem_ds->yg->vector,ug,dofset,2*numberofnodes);
			xfree((void**)&dofset);dofset=dofsetgen(1,dof2,4,numberofnodes*4); VecMerge(fem_ds->yg->vector,ug_vert,dofset,1*numberofnodes);
			VecFree(&fem_ds->ys);
			Reducevectorgtosx(&fem_ds->ys,fem_ds->yg->vector,fem_ds->nodesets);

			if(verbose)_printf_("%s\n"," computing stokes velocities and pressure ...");
			VecFree(&ug);
			diagnostic_core_nonlinear(&ug,NULL,NULL,NULL,fem_ds,DiagnosticAnalysisEnum,StokesAnalysisEnum);
		
			//decondition" pressure
			VecFree(&pg);	
			xfree((void**)&dofset);dofset=dofsetgen(1,dof3,4,numberofnodes*4); VecPartition(&pg, ug, dofset, numberofnodes*1);
			VecScale(pg,stokesreconditioning);
		}
	}
	
	/*Plug results into output dataset: */
	if(dim==2){
		if(ismacayealpattyn){
			InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,VxEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,VyEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,PressureEnum,results->Size()+1,0,1); results->AddObject(result);
		}
		else{
			InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,VxEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,VyEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,PressureEnum,results->Size()+1,0,1); results->AddObject(result);
		}
	}
	else{
		if(isstokes){
			InputToResultx(&result,fem_ds->elements,fem_ds->nodes,fem_ds->vertices, fem_ds->loads, fem_ds->materials,fem_ds->parameters,VxEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_ds->elements,fem_ds->nodes,fem_ds->vertices, fem_ds->loads, fem_ds->materials,fem_ds->parameters,VyEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_ds->elements,fem_ds->nodes,fem_ds->vertices, fem_ds->loads, fem_ds->materials,fem_ds->parameters,VzEnum,results->Size()+1,0,1); results->AddObject(result);
			InputToResultx(&result,fem_ds->elements,fem_ds->nodes,fem_ds->vertices, fem_ds->loads, fem_ds->materials,fem_ds->parameters,PressureEnum,results->Size()+1,0,1); results->AddObject(result);
		}
		else{
			if(ismacayealpattyn){
				InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,VxEnum,results->Size()+1,0,1); results->AddObject(result);
				InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,VyEnum,results->Size()+1,0,1); results->AddObject(result);
				InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,VzEnum,results->Size()+1,0,1); results->AddObject(result);
				InputToResultx(&result,fem_dh->elements,fem_dh->nodes,fem_dh->vertices, fem_dh->loads, fem_dh->materials,fem_dh->parameters,PressureEnum,results->Size()+1,0,1); results->AddObject(result);
			}
			else{
				InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,VxEnum,results->Size()+1,0,1); results->AddObject(result);
				InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,VyEnum,results->Size()+1,0,1); results->AddObject(result);
				InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,VzEnum,results->Size()+1,0,1); results->AddObject(result);
				InputToResultx(&result,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices, fem_dhu->loads, fem_dhu->materials,fem_dhu->parameters,PressureEnum,results->Size()+1,0,1); results->AddObject(result);
			}

		}
	}
	
	results->AddObject(new StringResult(results->Size()+1,AnalysisTypeEnum,0,1,EnumAsString(DiagnosticAnalysisEnum)));

	/*output if we have rifts: */
	if(numrifts){
		OutputRiftsx( &riftproperties,fem_dh->loads,numrifts);
		results->AddObject(new Result(results->Size()+1,0,1,"riftproperties",riftproperties));
	}

	/*Free ressources: */
	VecFree(&ug);
	VecFree(&pg);
	xfree((void**)&dofset);
	xfree((void**)&u_g_initial);
	
	VecFree(&surfaceslopex);
	VecFree(&surfaceslopey);
	VecFree(&bedslopex);
	VecFree(&bedslopey);

	/*return: */
	return results;
}
