/*
  CHACO : Hendrickson/Leland's graph partitioner.

  [part1,part2,chaco_time] = chaco(A) returns a 50/50 vertex partition of the mesh
  whose symmetric adjacency matrix is A.  

  Optional arguments:
  map = chaco(A, xy, method, nparts, goal);
  map = chaco(A, xy, method, inmap, goal);

  A:          Depending on "method", A may contain vertex and edge weights.

  xy:         Each row of xy is the coordinates of a vertex.
              If xy is non-null and there is no output, draw a picture.

  method:     Scalar or vector describing the desired method.  
              Default is multilevel Kernighan-Lin; other possibilities below.

  nparts      Number of parts to divide into.  Default is 2.  If nparts is 
    or        present, the output is a "map vector", see below.  (If method(5) 
  inmap:      is specified, nparts is interpreted differently; see below.  In
              any case, the default is to divide into two parts.)
              If method(1) = 7 (see below), this argument is a map vector
              specifying an initial 2-way partition, and Chaco refines it.

  goal:       Optionally, a vector of desired sizes (or total vertex weights)
              for each of the nparts parts.  Default is all sizes equal.

  map:        If nparts and inmap are not present, the output is a vector of 
              the n/2 vertex numbers in one part of the 2-way partition, for
              compatibility with geopart and specpart.
              If nparts or imap is present, the output is a vector of the
              n part numbers, from 0 to nparts-1, assigned to the vertices.

  This is a Matlab interface to the graph partitioning software described
  in B. Hendrickson and R. Leland, "The Chaco User's Guide (Version 2.0)",
  Sandia National Laboratories report SAND94-2692, October 1994.
  This interface was written by John Gilbert, Xerox PARC, and is
  Copyright (c) 1994-1996 by Xerox Corporation.  All rights reserved.
  HELP COPYRIGHT for complete copyright and licensing notice.

  Modified by Tim Davis, for Matlab 5.1.  July 6, 1998.
    05/26/10    jes    Reorganization into Matlab-layer and x-layer.

  See also GEOPART, SPECPART.

  "method" is a vector of flags as follows.  Not all combinations are supported.
  See Section 6.10 of the Chaco manual for more details on all the arguments.
  If "method" is shorter than 10, we use the defaults for unspecified entries.

  method[0]:  Global partitioning method  ("global_method" in the Chaco manual).
              1 Multilevel Kernighan-Lin (default)
              2 Spectral
              3 Inertial
              4 Linear
              5 Random
              6 Scattered
              7 Use "inmap" as the global (2-way) partition

  method[1]:  Local refinement method  ("local_method" in the Chaco manual).
              1 Kernighan-Lin (default)
              2 None

  method[2]:  Vertex weighting.
              0 No weights (default)
              1 Use diag(A) as (positive integer) vertex weights

  method[3]:  Edge weighting.
              0 No weights (default)
              1 Use off-diagonals of A as (positive integer) edge weights

  method[4]:  Target architecture  ("architecture" in the Chaco manual).
              If method[4] = 0, the target is a hypercube, "nparts" is the 
              number of dimensions, and the partition is into 2^nparts parts.  
              If method[4] = 1, 2, or 3, the target is a 1-, 2-, or 3-D grid,
              "nparts" is a vector of the sizes of the grid in each dimension,
              and the partition is into prod(nparts) parts.
              Default is method[4] = 1, so nparts is the number of parts.

  method[5]:  Partitioning dimension  ("ndims" in the Chaco manual).
              1 Bisection (default)
              2 Quadrisection
              3 Octasection

  method[6]:  Number of vertices to coarsen to  ("vmax" in the Chaco manual).
              Default is 50.

  method[7]:  Eigensolver  ("rqi_flag" in the Chaco manual).
              0 RQI/Symmlq (default)
              1 Lanczos 

  method[8]:  Eigensolver convergence tolerance  ("eigtol" in the Chaco manual).
              Default is .001

  method[9]:  Seed for random number generator  ("seed" in the Chaco manual).
              Default is 7654321.

  Many esoteric details of Chaco's behavior can be changed by placing a file
  called "User_Params" in the same directory as the executable mlchaco.mex.
  As always, see the manual for details.
*/

#include <stdio.h>
#include "mex.h"

#define THISFUNCTION "Chacox"

int Chacox(
	int       nvtxs,          /* number of vertices in full graph */
	int      *start,          /* start of edge list for each vertex */
	int      *adjacency,      /* edge list data */
	int      *vwgts,          /* weights for all vertices */
	float    *ewgts,          /* weights for all edges */
	float    *x,
	float    *y,
	float    *z,              /* coordinates for inertial method */
	short    *assignment,     /* set number of each vtx (length n) */
	double    method[10],     /* vector describing the desired method */
	int      *nparts,         /* number of parts to divide into */
	double   *goal            /* desired set sizes for each set */
)
{
	/* Arguments to Chaco "interface" routine: */

	char     *outassignname;  /* name of assignment output file */
	char     *outfilename;    /* output file name */
	int       architecture;   /* 0 => hypercube, d => d-dimensional mesh */
	int       ndims_tot=0;    /* total number of cube dimensions to divide */
	int       mesh_dims[3]={0,0,0};
							  /* dimensions of mesh of processors */
	int       global_method;  /* global partitioning algorithm */
	int       local_method;   /* local partitioning algorithm */
	int       rqi_flag;       /* should I use RQI/Symmlq eigensolver? */
	int       vmax;           /* if so, how many vertices to coarsen down to? */
	int       ndims;          /* number of eigenvectors (2^d sets) */
	double    eigtol;         /* tolerance on eigenvectors */
	long      seed;           /* for random graph mutations */
 
	int    i, nsets, failure;
	double totalvwgt, totalgoal;

/*	Chaco numbers vertices from 1 and the Matlab sparse data structure 
	numbers rows from 0, so we add an empty first row to make things line up. */

	for (i=0; i<start[nvtxs]; i++)
		adjacency[i]++;

	outassignname = NULL;
	outfilename = NULL;

/*	Decode "method" to get the actual args to Chaco.
	Note that "nparts" may correspond to any of several Chaco
	parameters, depending on the method.  */

	global_method = (int)method[0];
	local_method = (int)method[1];

	if ((int)method[2] && vwgts) {
		printf("%s -- Applying weights for %d vertices.\n",
			   THISFUNCTION,nvtxs);
		totalvwgt = 0.;
		for (i=0; i<nvtxs; i++)
			totalvwgt += vwgts[i];
	}
	else {
		totalvwgt = (double)nvtxs;
		if      ( (int)method[2] && !vwgts)
			printf("%s -- method[2]=%d, but no vertex weights specified.\n",
				   THISFUNCTION,method[2]);
		else if (!(int)method[2] &&  vwgts)
			printf("%s -- method[2]=%d, so specified vertex weights ignored.\n",
				   THISFUNCTION,method[2]);
	}

	if ((int)method[3] && ewgts) {
		printf("%s -- Applying weights for %d edges.\n",
			   THISFUNCTION,start[nvtxs]/2);
	}
	else {
		if      ( (int)method[3] && !ewgts)
			printf("%s -- method[3]=%d, but no edge weights specified.\n",
				   THISFUNCTION,method[3]);
		else if (!(int)method[3] &&  ewgts)
			printf("%s -- method[3]=%d, so specified edge weights ignored.\n",
				   THISFUNCTION,method[3]);
	}

	architecture = (int)method[4];
	if (global_method == 7) {
/*		Refine an input partition: "nparts" is the input partition.
		This seems to work only for hypercube architecture, 
		so we force a 1-D hypercube with 2-way partitioning.  */
		architecture = 0;
		for (i=0; i<nvtxs; i++)
			assignment[i] = nparts[i];
		ndims_tot = 1;
		ndims = 1;
		nsets = 2;
	}
	else if (architecture == 0) {
/*		Partition for hypercube: "nparts" is # of dimensions (default 1).  */
		if (!nparts)
			ndims_tot = 1;
		else
			ndims_tot = nparts[0];
		ndims = (int)method[5];
		nsets = 2^ndims_tot;
	}
	else {
/*		Partition for mesh: "nparts" is vector of mesh sizes in each
		dimension, default [2 1 ... 1] with "architecture" dimensions.  */
		if (!nparts) {
			mesh_dims[0] = 2;
			for (i=1; i<architecture; i++)
				mesh_dims[i] = 1;
		}
		else
			for (i=0; i<architecture; i++)
				mesh_dims[i] = nparts[i];
		ndims = (int)method[5];
		nsets = 1;
		for (i=0; i<architecture; i++)
			nsets *= mesh_dims[i];
	}

/*	Scale goal to total vertex weight so relative goals can be used.  */
	if (goal) {
		printf("%s -- Applying goals for %d sets.\n",
			   THISFUNCTION,nsets);
		totalgoal = 0.;
		for (i=0; i<nsets; i++)
			totalgoal += goal[i];
		for (i=0; i<nsets; i++)
			goal[i] *= totalvwgt/totalgoal;
	}

	vmax = (int)method[6];
	rqi_flag = (int)method[7];
	eigtol = method[8];
	seed = (long)method[9];

	/* Finally, call Chaco */

	printf("\n%s -- Calling Chaco interface:\n\n",
		   THISFUNCTION);
	failure = interface(nvtxs, start, adjacency,
		((int)method[2] && vwgts ? vwgts : NULL),
		((int)method[3] && ewgts ? ewgts : NULL), x, y, z,
		outassignname, outfilename, assignment, architecture, ndims_tot, 
		mesh_dims, goal, global_method, local_method, rqi_flag, vmax, 
		ndims, eigtol, seed);
	printf("\n%s -- Chaco interface returning failure=%d.\n",
		   THISFUNCTION,failure);

/*	Reset adjacency matrix in case calling function needs it.  */

	for (i=0; i<start[nvtxs]; i++)
		adjacency[i]--;

	return(failure);
}

