/*
  CHACO : Hendrickson/Leland's graph partitioner.

  [part1,part2,chaco_time] = chaco(A) returns a 50/50 vertex partition of the mesh
  whose symmetric adjacency matrix is A.  

  Optional arguments:
  map = chaco(A, xy, method, nparts, goal);
  map = chaco(A, xy, method, inmap, goal);

  A:          Depending on "method", A may contain vertex and edge weights.

  xy:         Each row of xy is the coordinates of a vertex.
              If xy is non-null and there is no output, draw a picture.

  method:     Scalar or vector describing the desired method.  
              Default is multilevel Kernighan-Lin; other possibilities below.

  nparts      Number of parts to divide into.  Default is 2.  If nparts is 
    or        present, the output is a "map vector", see below.  (If method(5) 
  inmap:      is specified, nparts is interpreted differently; see below.  In
              any case, the default is to divide into two parts.)
              If method(1) = 7 (see below), this argument is a map vector
              specifying an initial 2-way partition, and Chaco refines it.

  goal:       Optionally, a vector of desired sizes (or total vertex weights)
              for each of the nparts parts.  Default is all sizes equal.

  map:        If nparts and inmap are not present, the output is a vector of 
              the n/2 vertex numbers in one part of the 2-way partition, for
              compatibility with geopart and specpart.
              If nparts or imap is present, the output is a vector of the
              n part numbers, from 0 to nparts-1, assigned to the vertices.

  This is a Matlab interface to the graph partitioning software described
  in B. Hendrickson and R. Leland, "The Chaco User's Guide (Version 2.0)",
  Sandia National Laboratories report SAND94-2692, October 1994.
  This interface was written by John Gilbert, Xerox PARC, and is
  Copyright (c) 1994-1996 by Xerox Corporation.  All rights reserved.
  HELP COPYRIGHT for complete copyright and licensing notice.

  Modified by Tim Davis, for Matlab 5.1.  July 6, 1998.
  Modified by Tim Davis, July 1999.  Added 2nd output argument,
  which is the cpu time taken by chaco itself.
    05/25/10    jes    Reorganization into Matlab-layer and x-layer.

  See also GEOPART, SPECPART.

  "method" is a vector of flags as follows.  Not all combinations are supported.
  See Section 6.10 of the Chaco manual for more details on all the arguments.
  If "method" is shorter than 10, we use the defaults for unspecified entries.

  method[0]:  Global partitioning method  ("global_method" in the Chaco manual).
              1 Multilevel Kernighan-Lin (default)
              2 Spectral
              3 Inertial
              4 Linear
              5 Random
              6 Scattered
              7 Use "inmap" as the global (2-way) partition

  method[1]:  Local refinement method  ("local_method" in the Chaco manual).
              1 Kernighan-Lin (default)
              2 None

  method[2]:  Vertex weighting.
              0 No weights (default)
              1 Use diag(A) as (positive integer) vertex weights

  method[3]:  Edge weighting.
              0 No weights (default)
              1 Use off-diagonals of A as (positive integer) edge weights

  method[4]:  Target architecture  ("architecture" in the Chaco manual).
              If method[4] = 0, the target is a hypercube, "nparts" is the 
              number of dimensions, and the partition is into 2^nparts parts.  
              If method[4] = 1, 2, or 3, the target is a 1-, 2-, or 3-D grid,
              "nparts" is a vector of the sizes of the grid in each dimension,
              and the partition is into prod(nparts) parts.
              Default is method[4] = 1, so nparts is the number of parts.

  method[5]:  Partitioning dimension  ("ndims" in the Chaco manual).
              1 Bisection (default)
              2 Quadrisection
              3 Octasection

  method[6]:  Number of vertices to coarsen to  ("vmax" in the Chaco manual).
              Default is 50.

  method[7]:  Eigensolver  ("rqi_flag" in the Chaco manual).
              0 RQI/Symmlq (default)
              1 Lanczos 

  method[8]:  Eigensolver convergence tolerance  ("eigtol" in the Chaco manual).
              Default is .001

  method[9]:  Seed for random number generator  ("seed" in the Chaco manual).
              Default is 7654321.

  Many esoteric details of Chaco's behavior can be changed by placing a file
  called "User_Params" in the same directory as the executable mlchaco.mex.
  As always, see the manual for details.
*/

#include <stdio.h>
#include "mex.h"
#include <sys/times.h>
#include <time.h>      /*  CLOCKS_PER_SEC  */

/* Aliases for input and output arguments */
#define A_IN         prhs[0]
#define VWGTS_IN     prhs[1]
#define EWGTS_IN     prhs[2]
#define XYZ_IN       prhs[3]
#define METHOD_IN    prhs[4]
#define NPARTS_IN    prhs[5]
#define GOAL_IN      prhs[6]
#define MAP_OUT      plhs[0]
#define TIME_OUT     plhs[1]

#define THISFUNCTION "Chaco"

void ChacoUsage( void );

int Chacox(
	int       nvtxs,          /* number of vertices in full graph */
	int      *start,          /* start of edge list for each vertex */
	int      *adjacency,      /* edge list data */
	int      *vwgts,          /* weights for all vertices */
	float    *ewgts,          /* weights for all edges */
	float    *x,
	float    *y,
	float    *z,              /* coordinates for inertial method */
	short    *assignment,     /* set number of each vtx (length n) */
	double    method[10],     /* vector describing the desired method */
	int      *nparts,         /* number of parts to divide into */
	double   *goal            /* desired set sizes for each set */
);


void mexFunction(
	int           nlhs,           /* number of expected outputs */
	mxArray       *plhs[],        /* array of pointers to output arguments */
	int           nrhs,           /* number of inputs */
	const mxArray *prhs[]         /* array of pointers to input arguments */
)
{
	/* Arguments to Chaco "interface" routine: */

	int       nvtxs;          /* number of vertices in full graph */
	int      *start;          /* start of edge list for each vertex */
	int      *adjacency;      /* edge list data */
	int      *vwgts;          /* weights for all vertices */
	float    *ewgts;          /* weights for all edges */
	float    *x;
	float    *y;
	float    *z;              /* coordinates for inertial method */
	short    *assignment;     /* set number of each vtx (length n) */
	double    method[10]={1,1,0,0,1,1,50,0,.001,7654321};
							  /* vector describing the desired method */
	int      *nparts;         /* number of parts to divide into */
	double   *goal;           /* desired set sizes for each set */

	int    i, nedges, nterms, geodims, failure;
	double *p;
	mwIndex *mwstart, *mwadjacency;

	double *Tout ;			/* time: added by Tim Davis */
	struct tms t1 ;
	struct tms t2 ;

	/* Check for correct number of arguments passed from Matlab. */

	if      (nrhs == 0 && nlhs == 0) {
		ChacoUsage();
		return;
	} else if (nrhs < 1) {
		ChacoUsage();
		mexErrMsgTxt("Chaco mex-layer requires at least one input argument.");
	} else if (nlhs > 2) {
		ChacoUsage();
		mexErrMsgTxt("Chaco mex-layer requires one or two output arguments.");
	}

	/* Slog through the arguments, converting from Matlab matrices
	   to the various types Chaco wants.                            */

	if (!mxIsEmpty(A_IN) && mxIsNumeric(A_IN) && mxIsSparse(A_IN)) {
		nvtxs = mxGetN(A_IN);
		mwstart = mxGetJc(A_IN);
		start = mxMalloc((mxGetN(A_IN)+1)*sizeof(int));
		for (i=0; i<(mxGetN(A_IN)+1); i++)
			start[i]= (int)mwstart[i];
		nedges = start[nvtxs];
		mwadjacency = mxGetIr(A_IN);
		adjacency = mxMalloc(mxGetNzmax(A_IN)*sizeof(int));
		for (i=0; i<mxGetNzmax(A_IN); i++)
			adjacency[i]= (int)mwadjacency[i];
	}
	else {
		mexPrintf("%s -- Adjacency matrix must be numeric and sparse.\n",
				  THISFUNCTION);
		mexErrMsgTxt(" ");
	}

	if (nrhs >= 2 && !mxIsEmpty(VWGTS_IN)) {
		if (mxIsNumeric(VWGTS_IN) && !mxIsSparse(VWGTS_IN) &&
			(nterms=mxGetM(VWGTS_IN)*mxGetN(VWGTS_IN)) == nvtxs) {
			vwgts = (int *) mxCalloc(nvtxs, sizeof(int));
			p = mxGetPr(VWGTS_IN);
			for (i = 0; i < nvtxs; vwgts[i++] = (int) *p++);
		}
		else {
			mexPrintf("%s -- Vertex weight vector must be numeric, full, and length=%d.\n",
					  THISFUNCTION,nvtxs);
			mexErrMsgTxt(" ");
		}
	} else vwgts = NULL;

	if (nrhs >= 3 && !mxIsEmpty(EWGTS_IN)) {
		if (mxIsNumeric(EWGTS_IN) && mxIsSparse(EWGTS_IN) &&
			(nterms=mxGetNzmax(EWGTS_IN)) == nedges) {
			ewgts = (float *) mxCalloc(nedges, sizeof(float));
			p = mxGetPr(A_IN);
			for (i = 0; i < nedges; ewgts[i++] = (float) *p++);
		}
		else {
			mexPrintf("%s -- Edge weight matrix must be numeric, sparse, and nonzeroes=%d.\n",
					  THISFUNCTION,nedges);
			mexErrMsgTxt(" ");
		}
	} else ewgts = NULL;

	if (nrhs >= 4 && (geodims = mxGetN(XYZ_IN)) >= 1) {
		if (mxIsNumeric(XYZ_IN) && !mxIsSparse(XYZ_IN) &&
			mxGetM(XYZ_IN) == nvtxs) {
			x = (float *) mxCalloc(nvtxs, sizeof(float));
			p = mxGetPr(XYZ_IN);
			for (i = 0; i < nvtxs; x[i++] = (float) *p++);
		}
		else {
			mexPrintf("%s -- XYZ coordinate vectors must be numeric, full, and length=%d.\n",
					  THISFUNCTION,nvtxs);
			mexErrMsgTxt(" ");
		}
	} else x = NULL;
	if (nrhs >= 4 && (geodims                 ) >= 2) {
		y = (float *) mxCalloc(nvtxs, sizeof(float));
		for (i = 0; i < nvtxs; y[i++] = (float) *p++);
	} else y = NULL;
	if (nrhs >= 4 && (geodims                 ) >= 3) {
		z = (float *) mxCalloc(nvtxs, sizeof(float));
		for (i = 0; i < nvtxs; z[i++] = (float) *p++);
	} else z = NULL;

	if (nrhs >= 5 && !mxIsEmpty(METHOD_IN)) {
		if (mxIsNumeric(METHOD_IN) && !mxIsSparse(METHOD_IN) &&
			(nterms=mxGetM(METHOD_IN)*mxGetN(METHOD_IN))) {
			p = mxGetPr(METHOD_IN);
			for (i = 0; i < (nterms<10 ? nterms : 10); method[i++] = *p++);
		}
		else {
			mexPrintf("%s -- Method vector must be numeric and full.\n",
					  THISFUNCTION);
			mexErrMsgTxt(" ");
		}
	}

	if (nrhs >= 6 && !mxIsEmpty(NPARTS_IN)) {
		if (mxIsNumeric(NPARTS_IN) && !mxIsSparse(NPARTS_IN) &&
			(nterms=mxGetM(NPARTS_IN)*mxGetN(NPARTS_IN))) {
			nparts = (int *) mxCalloc(nterms, sizeof(int));
			p = mxGetPr(NPARTS_IN);
			for (i = 0; i < nterms; nparts[i++] = (int) *p++);
		}
		else {
			mexPrintf("%s -- Parts vector must be numeric and full.\n",
					  THISFUNCTION);
			mexErrMsgTxt(" ");
		}
	} else nparts = NULL;

	if (nrhs >= 7 && !mxIsEmpty(GOAL_IN)) {
		if (mxIsNumeric(GOAL_IN) && !mxIsSparse(GOAL_IN) &&
			(nterms=mxGetM(GOAL_IN)*mxGetN(GOAL_IN))) {
			goal = (double *) mxCalloc(nterms, sizeof(double));
			p = mxGetPr(GOAL_IN);
			for (i = 0; i < nterms; goal[i++] = *p++);
		}
		else {
			mexPrintf("%s -- Goal vector must be numeric and full.\n",
					  THISFUNCTION);
			mexErrMsgTxt(" ");
		}
	} else goal = NULL;

	assignment = (short *) mxCalloc(nvtxs, sizeof(short));

	/* Finally, call Chaco */

	(void) times (&t1) ;

	failure = Chacox(nvtxs, start, adjacency, vwgts, ewgts, x, y, z,
		assignment, method, nparts, goal);

	(void) times (&t2) ;

	/* Copy the mapping vector into a Matlab matrix. */

	if (!failure) {
		MAP_OUT = mxCreateDoubleMatrix(1,nvtxs,mxREAL);
		p = mxGetPr(MAP_OUT);
		for (i = 0; i < nvtxs; *p++ = (double) assignment[i++]);
	}

	if (nlhs > 1)
	{
		TIME_OUT = mxCreateDoubleMatrix (1, 1, mxREAL) ;
		Tout = mxGetPr (TIME_OUT) ;
		Tout [0] =
			((double) (t2.tms_utime + t2.tms_stime - t1.tms_utime - t1.tms_stime)) /
			((double) CLOCKS_PER_SEC) ;
	}

	/* Free what we allocated */
   
	if (start != NULL) mxFree((void *) start);
	if (adjacency != NULL) mxFree((void *) adjacency);
	if (vwgts != NULL) mxFree((void *) vwgts);
	if (ewgts != NULL) mxFree((void *) ewgts);
	if (x != NULL) mxFree((void *) x);
	if (y != NULL) mxFree((void *) y);
	if (z != NULL) mxFree((void *) z);
	if (nparts != NULL) mxFree((void *) nparts);
	if (goal != NULL) mxFree((void *) goal);
	if (assignment != NULL) mxFree((void *) assignment);

	if (failure)
		mexErrMsgTxt("The call to C Chaco returned failure.");

	return;
}

void ChacoUsage( void )
{
	mexPrintf("\n");
	mexPrintf("Usage: [map,chaco_time] = Chaco(A,vwgts,ewgts,xyz,method,nparts,goal);\n");
	mexPrintf("\n");

	return;
}

