function md=SetIceShelfBC(md,icefrontfile)
%SETICESHELFBC - Create the boundary conditions for diagnostic and thermal models for a  Ice Shelf with Ice Front
%
%   Neumann BC are used on the ice front (an ANRGUS contour around the ice front
%   must be given in input)
%   Dirichlet BC are used elsewhere for diagnostic
%
%   Usage:
%      md=SetIceShelfBC(md,icefrontfile)
%
%   Example:
%      md=SetIceShelfBC(md,'Front.exp');
%
%   See also: SETICESHEETBC, SETMARINEICESHEETBC

%grid on Dirichlet (boundary and ~icefront)
if ~exist(icefrontfile)
	error(['SetIceShelfBC error message: ice front file ' icefrontfile ' not found']);
end
gridinsideicefront=ContourToMesh(md.elements,md.x,md.y,expread(icefrontfile,1),'node',2);
gridonicefront=double(md.gridonboundary & gridinsideicefront);
pos=find(md.gridonboundary & ~gridonicefront);
md.spcvelocity=zeros(md.numberofgrids,6);
md.spcvelocity(pos,1:2)=1;

%Dirichlet Values
if (length(md.vx_obs)==md.numberofgrids & length(md.vy_obs)==md.numberofgrids)
	disp('      boundary conditions for diagnostic model: spc set as observed velocities');
	md.spcvelocity(pos,4:5)=[md.vx_obs(pos) md.vy_obs(pos)]; %zeros for vz
else
	disp('      boundary conditions for diagnostic model: spc set as zero');
end

%segment on Ice Front
%segment on Neumann (Ice Front)
pos=find(gridonicefront(md.segments(:,1)) | gridonicefront(md.segments(:,2)));
if (md.dim==2)
	pressureload=md.segments(pos,:);
elseif md.dim==3
	pressureload_layer1=[md.segments(pos,1:2)  md.segments(pos,2)+md.numberofgrids2d  md.segments(pos,1)+md.numberofgrids2d  md.segments(pos,3)];
	pressureload=[];
	for i=1:md.numlayers-1,
		pressureload=[pressureload ;pressureload_layer1(:,1:4)+(i-1)*md.numberofgrids2d pressureload_layer1(:,5)+(i-1)*md.numberofelements2d ];
	end
end

%Add water or air enum depending on the element
pressureload=[pressureload WaterEnum*md.elementoniceshelf(pressureload(:,end))];

%plug onto model
md.pressureload=pressureload;

%Create zeros melting_rate and accumulation_rate if not specified
if isnan(md.accumulation_rate),
	md.accumulation_rate=zeros(md.numberofgrids,1);
	disp('      no accumulation_rate specified: values set as zero');
end
if isnan(md.melting_rate),
	md.melting_rate=zeros(md.numberofgrids,1);
	disp('      no melting_rate specified: values set as zero');
end
if isnan(md.dhdt),
	md.dhdt=zeros(md.numberofgrids,1);
	disp('      no dhdt specified: values set as zero');
end

displaystring(md.verbose,'%s',['      boundary conditions for prognostic model initialization']);
md.spcthickness=zeros(md.numberofgrids,2);

if (length(md.observed_temperature)==md.numberofgrids),
	displaystring(md.verbose,'%s',['      boundary conditions for thermal model']);
	md.spctemperature=[md.gridonsurface md.observed_temperature]; %impose observed temperature on surface
	if (length(md.geothermalflux)~=md.numberofgrids),
		md.geothermalflux=zeros(md.numberofgrids,1);
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
