/*!\file FemModel.c
 * \brief: implementation of the FemModel object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "../DataSet/DataSet.h"
#include "../modules/FemModelProcessorx/FemModelProcessorx.h"
#include "./objects.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"


/*Object constructors and destructor*/
/*FUNCTION FemModel::constructor {{{1*/
FemModel::FemModel(int in_nummodels){

	nummodels=in_nummodels;
	analysis_counter=-1;
	
	/*Dynamically allocate whatever is a list of length nummodels: */
	analysis_type_list=(int*)xmalloc(nummodels*sizeof(int));
	for(i=0;i<nummodels;i++)analysis_type_list[i]=NoneAnalysisEnum;

	Rmg=(Mat*)xmalloc(nummodels*sizeof(Mat));
	for(i=0;i<nummodels;i++)Rmg[i]=NULL;

	Gmn=(Mat*)xmalloc(nummodels*sizeof(Mat));
	for(i=0;i<nummodels;i++)Gmn[i]=NULL;

	nodesets=(NodeSets**)xmalloc(nummodels*sizeof(NodeSet*));
	for(i=0;i<nummodels;i++)nodesets[i]=NULL;

	yg=(Vec*)xmalloc(nummodels*sizeof(Vec));
	for(i=0;i<nummodels;i++)yg[i]=NULL;

	ys=(Vec*)xmalloc(nummodels*sizeof(Vec));
	for(i=0;i<nummodels;i++)ys[i]=NULL;

}
/*}}}1*/
/*FUNCTION FemModel::destructor {{{1*/
FemModel::~FemModel(){

	int i;

	/*Delete all the datasets: */
	xfree((void**)&analysis_type_list);
	delete elements;
	delete nodes;
	delete vertices;
	delete constraints;
	delete loads;
	delete materials;
	delete parameters;
	delete partition;
	delete tpartition;

	for(i=0;i<nummodels;i++){
		Mat temp_Rmg=Rmg[i];
		MatFree(&temp_Rmg);
		Mat temp_Gmn=Gmn[i];
		MatFree(&temp_Gmn);
		NodeSet* temp_nodesets=nodesets[i];
		delete nodesets;
		Vec temp_yg=yg[i];
		VecFree(&temp_yg);
		Vec temp_ys=ys[i];
		VecFree(&temp_ys);
	}

	/*Delete dynamically allocated arrays: */
	delete Rmg;
	delete Gmn;
	delete nodesets;
	delete yg;
	delete ys;

}
/*}}}1*/

/*Object management*/
/*FUNCTION FemModel::Echo {{{1*/

void FemModel::Echo(void){

	printf("FemModel echo: \n");
	printf("   number of fem models: %i\n",nummodels);
	printf("   analysis_type_list: \n");
	for(i=0;i<nummodels;i++)printf("     %i: %s\n",i,EnumAsString(analysis_type_list[i]));
	printf("   current analysis_type: \n");
	printf("     %i: %s\n",i,EnumAsString(analysis_type_list[analysis_counter]));


}
/*}}}1*/

/*Numerics: */
/*FUNCTION FemModel::AddAnalysis(ConstDataHandle FEMMODEL, int analysis_type) {{{1*/
void  FemModel::AddAnalysis(ConstDataHandle IOMODEL, int analysis_type){

	/*Set counter: */
	if (analysis_counter==-1)analysis_counter=0;
	else analysis_counter++;

	/*intermediary: */
	IoFemModel* iomodel=NULL;
	
	_printf_("   fill model with matlab workspace data\n");
	iomodel=new IoFemModel(IOMODEL);

	_printf_("   create datasets:\n");
	CreateDataSets(&elements,&nodes,&vertices,&materials,&constraints,&loads,&parameters,iomodel,IOMODEL,analysis_type,nummodels,analysis_counter);

	_printf_("   create degrees of freedom: \n");
	Dofx( &partition,&tpartition,elements,nodes, vertices,parameters);
	
	_printf_("   create single point constraints: \n");
	SpcNodesx( &yg[analysis_counter], nodes,constraints); 
	
	_printf_("   create rigid body constraints:\n");
	MpcNodesx( &Rmg[analysis_counter], nodes,constraints); 
	
	_printf_("   create node sets:\n");
	BuildNodeSetsx(&nodesets[analysis_counter], nodes);

	_printf_("   reducing single point constraints vector:\n");
	Reducevectorgtosx(&ys[analysis_counter], yg[analysis_counter]->vector,nodesets[analysis_counter]);
	
	_printf_("   normalizing rigid body constraints matrix:\n");
	NormalizeConstraintsx(&Gmn[analysis_counter], Rmg[analysis_counter],nodesets[analysis_counter]);

	_printf_("   configuring element and loads:\n");
	ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters);

	_printf_("   process parameters:\n");
	ProcessParamsx( parameters, partition->vector);

	_printf_("   free ressources:\n");
	delete iomodel;
}
/*}}}1*/
/*FUNCTION FemModel::GetCurrentAnalysis {{{1*/
FemFemModel* FemModel::GetCurrentAnalysis(){
	return analysis_type_list[analysis_counter];
}
/*}}}1*/
/*FUNCTION FemModel::SetCurrentAnalysis {{{1*/
void FemModel::SetCurrentAnalysis(int analysis_type){
	int found=-1;
	for(i=0;i<nummodels;i++){
		if (analysis_type_list[i]==analysis_type){
			found=i;
			break;
		}
	}
	if(found)analysis_counter=i;
	else ISSMERRR("%s%s%s"," could not find analysis_type ",EnumAsString(analysis_type) " in list of FemModel analyses");
}
/*}}}1*/

