/*!\file:  ElementsAndVerticesPartitioning.cpp
 * \brief: partition elements and nodes and vertices
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <string.h>
#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../io/io.h"
#include "../../include/include.h"
#include "../MeshPartitionx/MeshPartitionx.h"
#include "../ModelProcessorx/ModelProcessorx.h"

void  ElementsAndVerticesPartitioning(bool** pmy_elements, bool** pmy_vertices, bool** pmy_bordervertices, IoModel* iomodel, ConstDataHandle iomodel_handle){

	int i;

	extern int my_rank;
	extern int num_procs;

	/*output: */
	bool* my_elements=NULL;
	bool* my_vertices=NULL;
	bool* my_bordervertices=NULL;

	/*intermediary: */
	int* epart=NULL; //element partitioning.
	int* npart=NULL; //node partitioning.
	int  elements_width; //number of columns in elements (2d->3, 3d->6)
	Vec  bordervertices=NULL;
	double* serial_bordervertices=NULL;
	int  el1,el2;

	/*First, check that partitioning has not yet been carryed out. Just check whether my_elements pointers is not already assigned a value: */
	if (*pmy_elements)return;

	/*Number of vertices per elements, needed to correctly retrieve data: */
	if(iomodel->dim==2) elements_width=3; //tria elements
	else elements_width=6; //penta elements

	#ifdef _PARALLEL_
	/*Determine parallel partitioning of elements: we use Metis for now. First load the data, then partition*/
	if(iomodel->dim==2){
		/*load elements: */
		IoModelFetchData(&iomodel->elements,NULL,NULL,iomodel_handle,"elements");
	}
	else{
		/*load elements2d: */
		IoModelFetchData(&iomodel->elements2d,NULL,NULL,iomodel_handle,"elements2d");
	}

	MeshPartitionx(&epart, &npart,iomodel->numberofelements,iomodel->numberofvertices,iomodel->elements, iomodel->numberofelements2d,iomodel->numberofvertices2d,iomodel->elements2d,iomodel->numlayers,elements_width, iomodel->dim,num_procs);

	/*Free elements and elements2d: */
	xfree((void**)&iomodel->elements);
	xfree((void**)&iomodel->elements2d);

	#else
	/*In serial mode, epart is full of 0: all elements belong to cpu 0: */
	epart=(int*)xcalloc(iomodel->numberofelements,sizeof(int));
	#endif

	/*Deal with rifts, they have to be included into one partition only, not several: */
	IoModelFetchData(&iomodel->riftinfo,&iomodel->numrifts,NULL,iomodel_handle,"riftinfo");

	for(i=0;i<iomodel->numrifts;i++){
		el1=(int)*(iomodel->riftinfo+RIFTINFOSIZE*i+2)-1; //matlab indexing to c indexing
		el2=(int)*(iomodel->riftinfo+RIFTINFOSIZE*i+3)-1; //matlab indexing to c indexing
		epart[el2]=epart[el1]; //ensures that this pair of elements will be in the same partition, as well as the corresponding grids;
	}

	/*Free rifts: */
	xfree((void**)&iomodel->riftinfo); 

	/*Used later on: */
	my_vertices=(bool*)xcalloc(iomodel->numberofvertices,sizeof(bool));
	my_elements=(bool*)xcalloc(iomodel->numberofelements,sizeof(bool));

	/*Start figuring out, out of the partition, which elements belong to this cpu: */
	IoModelFetchData(&iomodel->elements,NULL,NULL,iomodel_handle,"elements");
	for (i=0;i<iomodel->numberofelements;i++){

		/*!All elements have been partitioned above, only deal with elements for this cpu: */
		if(my_rank==epart[i]){ 

			my_elements[i]=1;
			
			/*Now that we are here, we can also start building the list of vertices belonging to this cpu partition: we use 
			 *the  element index to do this. For each element n, we know index[n][0:2] holds the indices (matlab indexing) 
			 into the vertices coordinates. If we start plugging 1 into my_vertices for each index[n][i] (i=0:2), then my_vertices 
			 will hold which vertices belong to this partition*/
			my_vertices[(int)*(iomodel->elements+elements_width*i+0)-1]=1;
			my_vertices[(int)*(iomodel->elements+elements_width*i+1)-1]=1;
			my_vertices[(int)*(iomodel->elements+elements_width*i+2)-1]=1;
			
			if(elements_width==6){
				my_vertices[(int)*(iomodel->elements+elements_width*i+3)-1]=1;
				my_vertices[(int)*(iomodel->elements+elements_width*i+4)-1]=1;
				my_vertices[(int)*(iomodel->elements+elements_width*i+5)-1]=1;
			}
		}
	}//for (i=0;i<numberofelements;i++)
	/*Free data : */
	xfree((void**)&iomodel->elements);

	#ifdef _PARALLEL_
		/*From the element partitioning, we can determine which vertices are on the inside of this cpu's 
		 *element partition, and which are on its border with other vertices:*/
		bordervertices=NewVec(iomodel->numberofvertices);

		for (i=0;i<iomodel->numberofvertices;i++){
			if(my_vertices[i])VecSetValue(bordervertices,i,1,ADD_VALUES);
		}
		VecAssemblyBegin(bordervertices);
		VecAssemblyEnd(bordervertices);

		VecToMPISerial(&serial_bordervertices,bordervertices);

		/*now go through serial_bordervertices, and booleanize it: */
		my_bordervertices=(bool*)xcalloc(iomodel->numberofvertices,sizeof(bool));
		for(i=0;i<iomodel->numberofvertices;i++){
			if(serial_bordervertices[i]>1)my_bordervertices[i]=1;
		}
	#else
		/*No border vertices: */
		my_bordervertices=(bool*)xcalloc(iomodel->numberofvertices,sizeof(bool));
	#endif

	/*Free ressources:*/
	xfree((void**)&npart);
	xfree((void**)&epart);
	xfree((void**)&serial_bordervertices);
	VecFree(&bordervertices);

	/*Assign output pointers:*/
	*pmy_elements=my_elements;
	*pmy_vertices=my_vertices;
	*pmy_bordervertices=my_bordervertices;
}


