/*!\file: slope_core.cpp
 * \brief: core of the slope solution 
 */ 

#include "./solutions.h"
#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void slope_core(Vec* pslopex,Vec* pslopey,FemModel* femmodel, int AnalysisEnum){

	/*parameters: */
	int verbose;
	int dim;
	bool isstokes;
	bool ishutter;
	int xanalysis;
	int yanalysis;

	/*output: */
	Vec slopex=NULL;
	Vec slopey=NULL; 

	/*Recover some parameters: */
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&ishutter,IsHutterEnum);

	if(verbose)_printf_("%s\n","computing slope (x and y derivatives)...");

	/*Specify type of computations: */
	if(AnalysisEnum==SurfaceAnalysisEnum){
		xanalysis=SurfaceXAnalysisEnum;
		yanalysis=SurfaceYAnalysisEnum;
	}
	else if(AnalysisEnum==BedAnalysisEnum){
		xanalysis=BedXAnalysisEnum;
		yanalysis=BedYAnalysisEnum;
	}
	else ISSMERROR("%s%s%s"," analysis ",EnumAsString(AnalysisEnum)," not supported yet!");


	/*Early return possible? */
	if(!ishutter && AnalysisEnum==SurfaceAnalysisEnum){
		/*no need to compute Surface Slope except for Hutter: */
		*pslopex=NULL;
		*pslopey=NULL;
		return;
	}
	if(!isstokes && AnalysisEnum==BedAnalysisEnum){
		/*no need to compute Bed Slope except for full Stokes: */
		*pslopex=NULL;
		*pslopey=NULL;
		return;
	}
	
	
	/*Call on core computations: */
	diagnostic_core_linear(&slopex,femmodel,SlopecomputeAnalysisEnum,xanalysis);
	diagnostic_core_linear(&slopey,femmodel,SlopecomputeAnalysisEnum,yanalysis);

	/*extrude if we are in 3D: */
	if (dim==3){
		if(verbose)_printf_("%s\n","extruding slopes in 3d...");
		FieldExtrudex( slopex, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,"slopex",0);
		FieldExtrudex( slopey, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,"slopey",0);
	}

	/*Assign output pointers:*/
	*pslopex=slopex;
	*pslopey=slopey;
}
