/*!\file PetscVecResult.c
 * \brief: implementation of the PetscVecResult object
 */

/*header files: */
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
#include "../../include/include.h"
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION PetscVecResult::PetscVecResult(){{{1*/
PetscVecResult::PetscVecResult(){
	return;
}
/*}}}*/
/*FUNCTION PetscVecResult::PetscVecResult(int id, int enum_type,double time, int step, double* value,int M){{{1*/
PetscVecResult::PetscVecResult(int in_id, int in_enum_type,double in_time, int in_step, Vec in_value){

	id=in_id;
	enum_type=in_enum_type;
	time=in_time;
	step=in_step;
	
	if(in_value){
		VecDuplicate(in_value,&value);
		VecCopy(in_value,value);
	}
}
/*}}}*/
/*FUNCTION PetscVecResult::~PetscVecResult(){{{1*/
PetscVecResult::~PetscVecResult(){
	VecFree(&value);
}
/*}}}*/

/*Object methods*/
/*FUNCTION PetscVecResult::copy{{{1*/
Object* PetscVecResult::copy() {
	
	return new PetscVecResult(this->id,this->enum_type,this->time,this->step,this->value);

}
/*}}}*/
/*FUNCTION PetscVecResult::DeepEcho{{{1*/
void PetscVecResult::DeepEcho(void){
	
	int i;

	printf("PetscVecResult echo:\n");
	printf("   id:  %i \n",this->id);
	printf("   enum:  %i (%s)\n",this->enum_type,EnumAsString(this->enum_type));
	printf("   time:  %g \n",this->time);
	printf("   step:  %i \n",this->step);
	VecView(value,PETSC_VIEWER_STDOUT_WORLD);
}
/*}}}*/
/*FUNCTION PetscVecResult::Demarshall{{{1*/
void  PetscVecResult::Demarshall(char** pmarshalled_dataset){

	int   i;
	char* marshalled_dataset=NULL;
	int   M;
	double* serial_vec=NULL;
	int*    idxm=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum value, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(&time,marshalled_dataset,sizeof(time));marshalled_dataset+=sizeof(time);
	memcpy(&step,marshalled_dataset,sizeof(step));marshalled_dataset+=sizeof(step);
	
	memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
	if(M){
		serial_vec=(double*)xmalloc(M*sizeof(double));
		memcpy(serial_vec,marshalled_dataset,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));

		value=NewVec(M);
		idxm=(int*)xmalloc(M*sizeof(int));
		for(i=0;i<M;i++)idxm[i]=i;
		VecSetValues(value,M,idxm,serial_vec,INSERT_VALUES);

		VecAssemblyBegin(value);
		VecAssemblyEnd(value);
	}
	else{
		value=NULL;
	}

	/*Free ressources:*/
	xfree((void**)&serial_vec);
	xfree((void**)&idxm);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;

}
/*}}}*/
/*FUNCTION PetscVecResult::Echo {{{1*/
void PetscVecResult::Echo(void){

	int M;

	VecGetSize(value,&size);
	printf("PetscVecResult echo:\n");
	printf("   id:  %i \n",this->id);
	printf("   enum:  %i (%s)\n",this->enum_type,EnumAsString(this->enum_type));
	printf("   time:  %g \n",this->time);
	printf("   step:  %i \n",this->step);
	printf("   vector size: %i\n",M);

}
/*}}}*/
/*FUNCTION PetscVecResult::Enum{{{1*/
int PetscVecResult::Enum(void){

	return PetscVecResultEnum;

}
/*}}}*/
/*FUNCTION PetscVecResult::Id{{{1*/
int    PetscVecResult::Id(void){ return this->id; }
/*}}}*/
/*FUNCTION PetscVecResult::Marshall{{{1*/
void  PetscVecResult::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;
	int   M;
	double* serial_value=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of PetscVecResult: */
	enum_value=PetscVecResultEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall PetscVecResult data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(marshalled_dataset,&time,sizeof(time));marshalled_dataset+=sizeof(time);
	memcpy(marshalled_dataset,&step,sizeof(step));marshalled_dataset+=sizeof(step);

	if(value){
		VecGetSize(value,&M);
		VecToMPISerial(&serial_value,value);
		memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
		memcpy(marshalled_dataset,serial_value,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));
	}
	else{
		M=0;
		memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
	}

	/*Free ressources:*/
	xfree((void**)&serial_value);

	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION PetscVecResult::MarshallSize{{{1*/
int   PetscVecResult::MarshallSize(){

	int M=0;
	if(value)VecGetSize(value,&M);
	
	return sizeof(id)+
		sizeof(enum_type)+
		sizeof(time)+
		sizeof(step)+
		sizeof(M)+
		M*sizeof(double)+
		sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION PetscVecResult::MyRank{{{1*/
int    PetscVecResult::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION PetscVecResult::UpdateInputsFromVector(double* vector, int name, int type) {{{1*/
void  PetscVecResult::UpdateInputsFromVector(double* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION PetscVecResult::UpdateInputsFromVector(int* vector, int name, int type) {{{1*/
void  PetscVecResult::UpdateInputsFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION PetscVecResult::UpdateInputsFromVector(bool* vector, int name, int type) {{{1*/
void  PetscVecResult::UpdateInputsFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION PetscVecResult::UpdateInputsFromConstant(double constant, int name) {{{1*/
void  PetscVecResult::UpdateInputsFromConstant(double constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION PetscVecResult::UpdateInputsFromConstant(int constant, int name) {{{1*/
void  PetscVecResult::UpdateInputsFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION PetscVecResult::UpdateInputsFromConstant(bool constant, int name) {{{1*/
void  PetscVecResult::UpdateInputsFromConstant(bool constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
