/*!\file: transient_2d_core.cpp
 * \brief: core of the transient_2d solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"

Results* transient_core_2d(Model* model){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;
	FemModel* fem_p=NULL;

	/*output: */
	Results* results=NULL;
	Result * result =NULL;

	/*solutions: */
	int step,i;
	double time;

	Vec u_g=NULL;
	Vec p_g=NULL;
	Vec h_g=NULL;
	Vec new_h_g=NULL;
	Vec h_g_intermediary=NULL;
	Vec s_g=NULL;
	Vec new_s_g=NULL;
	Vec b_g=NULL;
	Vec new_b_g=NULL;
	double* vx=NULL;
	double* vy=NULL;

	/*intermediary: */
	double finaltime;
	double dt;
	Results* diagnostic_results=NULL;
	Results* prognostic_results=NULL;


	/*flags: */
	int verbose=0;
	int numberofnodes;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};

	//initialize results
	results=new Results();

	/*recover fem models: */
	fem_dh=model->GetFormulation(DiagnosticAnalysisEnum,HorizAnalysisEnum);
	fem_dv=model->GetFormulation(DiagnosticAnalysisEnum,VertAnalysisEnum);
	fem_ds=model->GetFormulation(DiagnosticAnalysisEnum,StokesAnalysisEnum);
	fem_dhu=model->GetFormulation(DiagnosticAnalysisEnum,HutterAnalysisEnum);
	fem_sl=model->GetFormulation(SlopecomputeAnalysisEnum);
	fem_p=model->GetFormulation(PrognosticAnalysisEnum);


	//first recover parameters common to all solutions
	model->FindParam(&verbose,VerboseEnum);
	model->FindParam(&finaltime,NdtEnum);
	model->FindParam(&dt,DtEnum);
	model->FindParam(&numberofnodes,NumberOfNodesEnum);

	/*initialize: */
	step=1;
	time=0;

	while(time<finaltime){ //make sure we run up to finaltime.
	
		_printf_("%s%g%s%i%s%g\n","time [yr]: ",time,"    iteration number: ",step,"/",floor(finaltime/dt));

		step+=1;
		time+=dt;

		//Deal with velocities.
		_printf_("%s\n","computing new velocity");
		diagnostic_results= diagnostic_core(model);

		//extract u_g and p_g from diagnostic_results, and erase diagnostic_results;
		VecFree(&u_g); diagnostic_results->FindResult(&u_g,"u_g");
		VecFree(&p_g); diagnostic_results->FindResult(&p_g,"p_g");
		delete diagnostic_results;

		//compute new thickness
		_printf_("%s\n","computing new thickness");
		
		SplitSolutionVectorx(u_g,numberofnodes,2,&vx,&vy);
		model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
		model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
		xfree((void**)&vx); xfree((void**)&vy);
		
		prognostic_results= prognostic_core(model);
	
		//extract h_g prognostic_results, and erase prognostic_results;
		prognostic_results->FindResult(&h_g_intermediary,"h_g");
		delete prognostic_results;

		//update surface and bed using the new thickness
		_printf_("   updating geometry\n");
		UpdateGeometryx(&new_h_g,&new_b_g,&new_s_g, fem_p->elements, fem_p->nodes,fem_p->vertices,fem_p->loads, fem_p->materials, fem_p->parameters,h_g_intermediary,b_g,s_g); 

		VecFree(&h_g);h_g=new_h_g; 
		VecFree(&s_g);s_g=new_s_g; 
		VecFree(&b_g);b_g=new_b_g; 
		VecFree(&h_g_intermediary);

		//plug into results.
		results->AddObject(new Result(results->Size()+1,time,step,"u_g",u_g)); 
		results->AddObject(new Result(results->Size()+1,time,step,"p_g",p_g)); 
		results->AddObject(new Result(results->Size()+1,time,step,"h_g",h_g)); 
		results->AddObject(new Result(results->Size()+1,time,step,"s_g",s_g)); 
		results->AddObject(new Result(results->Size()+1,time,step,"b_g",b_g)); 

		//update inputs
		model->UpdateInputsFromVector(h_g,ThicknessEnum,VertexEnum);
		model->UpdateInputsFromVector(s_g,SurfaceEnum,VertexEnum);
		model->UpdateInputsFromVector(b_g,BedEnum,VertexEnum);
	
		SplitSolutionVectorx(u_g,numberofnodes,2,&vx,&vy);
		model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
		model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);

	}

	/*Add analysis_type to results: */
	results->AddObject(new StringResult(results->Size()+1,AnalysisTypeEnum,0,1,EnumAsString(TransientAnalysisEnum)));

	/*Free ressources:*/
	VecFree(&u_g);
	VecFree(&p_g);
	VecFree(&h_g);
	VecFree(&s_g);
	VecFree(&b_g);
	xfree((void**)&vx);
	xfree((void**)&vy);
	
	//return: 
	return results;

}
