/*!\file: thermal_core.cpp
 * \brief: core of the thermal solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"

Results* thermal_core(Model* model){

	extern int my_rank;
	int i;

	/*fem models: */
	FemModel* fem_t=NULL;
	FemModel* fem_m=NULL;

	/*output: */
	Results* results=NULL;
	Result*  result=NULL;

	/*solutions vectors: */
	Vec*    t_g=NULL;
	Vec*    m_g=NULL;
	double* time=NULL;
	double* t_g_serial=NULL;
	double* m_g_serial=NULL;

	/*intermediary*/
	Vec Vect=NULL;
	Vec Vecm=NULL;

	/*flags: */
	int    verbose=0;
	int    numberofdofspernode;
	int    numberofnodes;
	int    nsteps;
	double ndt;
	double dt;

	int    sub_analysis_type;
	double melting_offset;

	//initialize results
	results=new Results();

	/*recover fem models: */
	fem_t=model->GetFormulation(ThermalAnalysisEnum);
	fem_m=model->GetFormulation(MeltingAnalysisEnum);

	//first recover parameters common to all solutions
	fem_t->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	fem_t->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);
	fem_t->parameters->FindParam(&verbose,VerboseEnum);
	fem_t->parameters->FindParam(&ndt,NdtEnum);
	fem_t->parameters->FindParam(&dt,DtEnum);

	if(dt==0){

		time=(double*)xmalloc(sizeof(double));
		time[0]=0;

		/*allocate t_g and m_g arrays: */
		t_g=(Vec*)xmalloc(sizeof(Vec));
		m_g=(Vec*)xmalloc(sizeof(Vec));

		if(verbose)_printf_("computing temperatures:\n");
		thermal_core_nonlinear(&t_g[0],&melting_offset,fem_t,ThermalAnalysisEnum,NoneAnalysisEnum);
		model->UpdateInputsFromVector(t_g[0],TemperatureEnum,VertexEnum);
		model->UpdateInputsFromConstant(melting_offset,MeltingOffsetEnum);
		
		if(verbose)_printf_("computing melting:\n");
		diagnostic_core_linear(&m_g[0],fem_m,MeltingAnalysisEnum,NoneAnalysisEnum);
	}
	else{
		
		nsteps=(int)(ndt/dt);
		time=(double*)xmalloc((nsteps)*sizeof(double));

		/*allocate t_g and m_g arrays: */
		t_g=(Vec*)xmalloc((nsteps)*sizeof(Vec));
		m_g=(Vec*)xmalloc((nsteps)*sizeof(Vec));

		for(i=0;i<nsteps;i++){
			if(verbose)_printf_("time step: %i/%i\n",i+1,nsteps);
			time[i]=(i+1)*dt;
			
			if(verbose)_printf_("computing temperatures:\n");
			thermal_core_nonlinear(&t_g[i],&melting_offset,fem_t,ThermalAnalysisEnum,NoneAnalysisEnum);
			
			if(verbose)_printf_("computing melting:\n");
			model->UpdateInputsFromVector(t_g[i],TemperatureEnum,VertexEnum);
			model->UpdateInputsFromConstant(melting_offset,MeltingOffsetEnum);
			diagnostic_core_linear(&m_g[i],fem_m,MeltingAnalysisEnum,NoneAnalysisEnum);
		}
	}
	
	/*Plug results into output dataset: */
	if(dt==0){
		InputToResultx(&result,fem_t->elements,fem_t->nodes,fem_t->vertices, fem_t->loads, fem_t->materials,fem_t->parameters,TemperatureEnum,results->Size()+1,0,1); results->AddObject(result);
		InputToResultx(&result,fem_m->elements,fem_m->nodes,fem_m->vertices, fem_m->loads, fem_m->materials,fem_m->parameters,MeltingRateEnum,results->Size()+1,0,1); results->AddObject(result);

		/*free ressource*/
		Vect=t_g[0];    Vecm=m_g[0];
		VecFree(&Vect); VecFree(&Vecm);
	}
	else{
		for(i=0;i<nsteps;i++){
			/*To be moved inside the loop!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
			InputToResultx(&result,fem_t->elements,fem_t->nodes,fem_t->vertices, fem_t->loads, fem_t->materials,fem_t->parameters,TemperatureEnum,results->Size()+1,time[i],i+1); results->AddObject(result);
			InputToResultx(&result,fem_m->elements,fem_m->nodes,fem_m->vertices, fem_m->loads, fem_m->materials,fem_m->parameters,MeltingRateEnum,results->Size()+1,time[i],i+1); results->AddObject(result);

			/*free ressource*/
			Vect=t_g[i];    Vecm=m_g[i];
			VecFree(&Vect); VecFree(&Vecm);
		}
	}
	/*Add analysis_type to results: */
	results->AddObject(new StringResult(results->Size()+1,AnalysisTypeEnum,0,1,EnumAsString(ThermalAnalysisEnum)));

	/*free ressource*/
	xfree((void**)&t_g);
	xfree((void**)&m_g);
	xfree((void**)&time);
	
	//return: 
	return results;

}
