/*!\file:  gradjcompute_core.cpp
 * \brief compute inverse method gradient direction.
 */ 

#include "../modules/modules.h"
#include "./solutions.h"
#include "../EnumDefinitions/EnumDefinitions.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

Results* gradjcompute_core(Model* model){
	
	
	/*intermediary: */
	FemModel* femmodel=NULL;
	Results* diagnostic_results=NULL;
	Result*  result=NULL;
	int analysis_type;
	int sub_analysis_type;
	int numberofnodes;
	int control_steady;
	int numberofdofspernode;
	char* solverstring=NULL;
	int  control_type;
	
	Vec u_g=NULL;

	Vec du_g=NULL;
	Vec du_f=NULL;
	Vec grad_g=NULL;

	Vec lambda_f=NULL;
	Vec lambda_g=NULL;
	double* lambdax=NULL;
	double* lambday=NULL;
	double* vx=NULL;
	double* vy=NULL;
	double* vz=NULL;

	Mat K_ff0=NULL;
	Mat K_fs0=NULL;
	
	/*output: */
	Results* results=NULL;

	/*flags: */
	int verbose=0;
	int dim=-1;
	int extrude_param=0;

	//initialize results
	results=new Results();
	
	/*some parameters:*/
	femmodel=model->GetActiveFormulation();
	femmodel->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	femmodel->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);
	femmodel->parameters->FindParam(&control_steady,ControlSteadyEnum);
	femmodel->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	femmodel->parameters->FindParam(&numberofdofspernode,NumberOfDofsPerNodeEnum);
	femmodel->parameters->FindParam(&solverstring,SolverStringEnum);
	femmodel->parameters->FindParam(&control_type,ControlTypeEnum);
	femmodel->parameters->FindParam(&extrude_param,ExtrudeParamEnum);
	femmodel->parameters->FindParam(&verbose,VerboseEnum);
	femmodel->parameters->FindParam(&dim,DimEnum);

	_printf_("%s\n","      recover solution for this stiffness and right hand side:");
	diagnostic_core_nonlinear(&u_g,&K_ff0,&K_fs0,NULL, femmodel,DiagnosticAnalysisEnum,sub_analysis_type);

	_printf_("%s\n","      buid Du, difference between observed velocity and model velocity:");
	Dux( &du_g, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,analysis_type,sub_analysis_type);

	_printf_("%s\n","      reduce adjoint load from g-set to f-set:");
	Reduceloadfromgtofx(&du_f, du_g, femmodel->Gmn, K_fs0, femmodel->ys0, femmodel->nodesets);
	VecFree(&du_g);MatFree(&K_fs0);

	_printf_("%s\n","      solve for adjoint vector:");
	Solverx(&lambda_f, K_ff0, du_f, NULL, solverstring);
	VecFree(&du_f); MatFree(&K_ff0);
	
	_printf_("%s\n","      merge back to g set:");
	Mergesolutionfromftogx(&lambda_g, lambda_f,femmodel->Gmn,femmodel->ys0,femmodel->nodesets);
	VecFree(&lambda_f);

	/*add to inputs: */
	SplitSolutionVectorx(lambda_g,numberofnodes,numberofdofspernode,&lambdax,&lambday);
	model->UpdateInputsFromVector(lambdax,AdjointxEnum,VertexEnum);
	model->UpdateInputsFromVector(lambday,AdjointyEnum,VertexEnum);
	VecFree(&lambda_g);
	
	_printf_("%s\n","      compute gradJ:");
	Gradjx( &grad_g, numberofnodes,femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters, 
				analysis_type,sub_analysis_type,control_type);

	if (dim==3 && extrude_param){

		_printf_("%s\n","      extruding gradient...");
		FieldExtrudex( grad_g, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,"gradj",0);
	}

	if(control_steady){
		diagnostic_results= diagnostic_core(model);

		//extract u_g and add it to input (3d velocity needed by thermal_core)
		diagnostic_results->FindResult(&u_g,"u_g");
		
		SplitSolutionVectorx(u_g,numberofnodes,3,&vx,&vy,&vz);
		model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
		model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
		model->UpdateInputsFromVector(vz,VzEnum,VertexEnum);

		delete diagnostic_results;
	}

	/*Plug results into output dataset: */
	InputToResultx(&result,femmodel->elements,femmodel->nodes,femmodel->vertices, femmodel->loads, femmodel->materials,femmodel->parameters,GradientAnalysisEnum,results->Size()+1,0,1); results->AddObject(result);
	results->AddObject(new StringResult(results->Size()+1,AnalysisTypeEnum,0,1,EnumAsString(GradientAnalysisEnum)));
	
	/*Free ressources:*/
	VecFree(&u_g);
	VecFree(&grad_g);
	xfree((void**)&solverstring);
	xfree((void**)&lambdax);
	xfree((void**)&lambday);
	xfree((void**)&vx);
	xfree((void**)&vy);
	xfree((void**)&vz);

}
