function md=transient2d(md);
%TRANSIENT2D - 2d transient solution
%
%   Usage:
%      md=transient(md)
%
%   See also: TRANSIENT3D, TRANSIENT

%Build all models requested
models.analysis_type='transient'; %needed for processresults

displaystring(md.verbose,'%s',['reading diagnostic horiz model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=HorizAnalysisEnum(); models.dh=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading diagnostic vert model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=VertAnalysisEnum(); models.dv=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading diagnostic stokes model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=StokesAnalysisEnum(); models.ds=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading diagnostic hutter model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=HutterAnalysisEnum(); models.dhu=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading surface and bed slope computation model data']);
md.analysis_type=SlopecomputeAnalysisEnum(); md.sub_analysis_type=NoneAnalysisEnum(); models.sl=CreateFemModel(md);

displaystring(md.verbose,'%s',['reading prognostic model data']);
md.analysis_type=PrognosticAnalysisEnum(); models.p=CreateFemModel(md);

%initialize solution
solution=struct('step',[],'time',[],'u_g',[],'p_g',[],'h_g',[],'s_g',[],'b_g',[]);
solution.step=1;
solution.time=0;
solution.u_g=models.dh.parameters.u_g(dofsetgen([1,2],3,length(models.dh.parameters.u_g)));
solution.p_g=[];
solution.h_g=models.p.parameters.h_g;
solution.s_g=models.p.parameters.s_g;
solution.b_g=models.p.parameters.b_g;

%initialize inputs
displaystring(md.verbose,'\n%s',['setup inputs...']);
inputs=inputlist;
inputs=add(inputs,'velocity',solution.u_g,'doublevec',2,models.p.parameters.NumberOfNodes);
inputs=add(inputs,'melting',models.p.parameters.m_g,'doublevec',1,models.p.parameters.NumberOfNodes);
inputs=add(inputs,'accumulation',models.p.parameters.a_g,'doublevec',1,models.p.parameters.NumberOfNodes);
inputs=add(inputs,'dt',models.p.parameters.Dt*models.p.parameters.Yts,'double'); %change time from yrs to sec

% figure out number of dof: just for information purposes.
md.dof=modelsize(models);

%first time step is given by model. 
dt=models.p.parameters.Dt;
finaltime=models.p.parameters.Ndt;
time=dt;
yts=models.p.parameters.Yts;
n=1; %counter

while  time<finaltime+dt, %make sure we run up to finaltime.

	disp(sprintf('\n%s%g%s%g%s%g\n','time [yr]: ',time,'    iteration number: ',n,'/',floor(finaltime/dt)));

	solution(n+1).step=n+1;
	solution(n+1).time=time;

	%update inputs
	inputs=add(inputs,'thickness',solution(n).h_g,'doublevec',1,models.p.parameters.NumberOfNodes);
	inputs=add(inputs,'surface',solution(n).s_g,'doublevec',1,models.p.parameters.NumberOfNodes);
	inputs=add(inputs,'bed',solution(n).b_g,'doublevec',1,models.p.parameters.NumberOfNodes);

	%Deal with velocities.

	%Get horizontal solution. 
	rawresults=diagnostic_core(models);
	solution(n+1).u_g=rawresults.u_g; solution(n+1).p_g=rawresults.p_g;

	%compute new thickness
	disp(sprintf('%s','   computing new thickness...'));
	inputs=add(inputs,'velocity',solution(n+1).u_g,'doublevec',2,models.p.parameters.NumberOfNodes);
	rawresults=prognostic_core(models,PrognosticAnalysisEnum(),NoneAnalysisEnum());
	new_thickness=rawresults.h_g;

	%update surface and bed using the new thickness
	disp(sprintf('%s','   updating geometry...'));
	[new_thickness,new_bed,new_surface]=UpdateGeometry(models.p.elements,models.p.nodes,models.p.vertices,models.p.loads,models.p.materials,models.p.parameters,new_thickness,solution(n).b_g,solution(n).s_g);

	%Record bed surface and thickness in the solution
	solution(n+1).h_g=new_thickness;
	solution(n+1).b_g=new_bed;
	solution(n+1).s_g=new_surface;

	%figure out if time stepping is good
	%disp(sprintf('%s','   checking time stepping...'));
	%[back,dt,time]=TimeStepping(md,solution,dt,time);
	%if back,
	%	continue;
	%end

	%update time and counter
	time=time+dt;
	n=n+1;
end

%Load results onto model
results=struct('step',[],'time',[],'vx',[],'vy',[],'vel',[],'pressure',[],'thickness',[],'surface',[],'bed',[]);
for i=1:length(solution),
	results(i).step=solution(i).step;
	results(i).time=solution(i).time/yts;
	results(i).vx=solution(i).u_g(1:2:end)*yts;
	results(i).vy=solution(i).u_g(2:2:end)*yts;
	results(i).vel=sqrt(solution(i).u_g(1:2:end).^2+solution(i).u_g(2:2:end).^2)*yts;
	results(i).bed=solution(i).b_g;
	results(i).surface=solution(i).s_g;
	results(i).thickness=solution(i).h_g;
end
if ~isstruct(md.results), md.results=struct(); end
md.results.TransientAnalysis=results;
