/*!\file Bamgx
 * \brief: use Bamg capabilities.
 */
#include "./Bamgx.h"
#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include ../../objects/BamgObjects.h"

using namespace bamg;
using namespace std;

int Bamgx(BamgMesh* bamgmesh_out,BamgGeom* bamggeom_out,BamgMesh* bamgmesh_in,BamgGeom* bamggeom_in,BamgOpts* bamgopts){

	/*Bamg options*/
	int    maxnbv;
	double coef;
	int    verbosity;
	int    NbSmooth;

	/*intermediary*/
	int noerr=1;
	int i,j,num;
	double costheta=2;
	double hminaniso=1e-100; 

	Triangles* Thr=NULL;
	Triangles* Thb=NULL;

	/*Bamg options*/
	NbSmooth=bamgopts->NbSmooth;
	coef=bamgopts->coef;
	maxnbv=bamgopts->maxnbv;
	verbosity=bamgopts->verbose;

	// no metric -> no smoothing 
	if (bamgopts->metric==NULL){
		NbSmooth=0; 
	}

	/*If no mesh in input, generate one*/

	if(bamgmesh_in->TrianglesSize[0]==0){
		/*Mesh generation {{{1*/

		//Step1: generate geometry Gh
		if (verbosity>0) printf("Construction of a mesh from a given geometry\n");
		if (verbosity>1) printf("   Processing geometry...\n");
		Geometry Gh(bamggeom_in,bamgopts);

		//get hmin and hmax from geometry to generate the metric
		bamgopts->hmin = Max(bamgopts->hmin,Gh.MinimalHmin());
		bamgopts->hmax = Min(bamgopts->hmax,Gh.MaximalHmax());

		//build metric using geometry
		if (verbosity>1) printf("   Generating Metric...\n");
		for(i=0;i<Gh.nbv;i++){
			Metric M=Gh[i];
			MatVVP2x2 Vp(M/coef);
			Vp.Maxh(bamgopts->hmax);
			Vp.Minh(bamgopts->hmin);
			Gh.vertices[i].m = Vp;
		}

		//generate mesh
		if (verbosity>1) printf("   Generating Mesh...\n");
		Triangles Th(maxnbv,Gh,bamgopts);

		//Split corners if requested
		if(bamgopts->splitcorners) Th.SplitInternalEdgeWithBorderVertices();

		//Renumbering
		Th.ReNumberingTheTriangleBySubDomain();

		//Crack mesh if requested
		if(bamgopts->Crack) Th.CrackMesh(bamgopts);

		//Build output
		if (verbosity>1) printf("   Write Mesh...\n");
		Th.WriteMesh(bamgmesh_out,bamgopts);
		if (verbosity>1) printf("   Write Geometry...\n");
		Gh.WriteGeometry(bamggeom_out,bamgopts);

		//clean up
	//	delete &Th;
	//	delete &Gh;
		/*}}}*/
	}

	else{
		/*Anisotropic mesh adaptation {{{1*/

		// read background mesh 
		if (verbosity>0) printf("Anisotropic mesh adaptation\n");
		if (verbosity>1) printf("   Processing initial mesh and geometry...\n");
		Triangles BTh(bamggeom_in,bamgmesh_in,bamgopts); 

		//Make Quadtree from background mesh
		BTh.MakeQuadTree();

		//Bound hmin and hmax
		bamgopts->hmin=Max(bamgopts->hmin,BTh.MinimalHmin());
		bamgopts->hmax=Min(bamgopts->hmax,BTh.MaximalHmax());

		//Generate initial metric
		if (bamgopts->metric){
			if (verbosity>1) printf("   Processing Metric...\n");
			BTh.ReadMetric(bamgopts);
		}
		else { 
			if (verbosity>1) printf("   Generating initial Metric...\n");
			Metric Mhmax(bamgopts->hmax);
			for (int iv=0;iv<BTh.nbv;iv++) BTh[iv].m = Mhmax;
		}

		//use present fields to generate metric if present
		if (bamgopts->field){
			if (verbosity>1) printf("   Merge metric with field provided...\n");
			BTh.AddMetric(bamgopts);
		}

		// change using hVertices if provided
		if (bamgmesh_in->hVertices){
			if (verbosity>1) printf("   Merging Metric with hVertices...\n");
			for (i=0;i<BTh.nbv;i++){
				if (!isnan(bamgmesh_in->hVertices[i])){
					BTh[i].m=Metric((float)bamgmesh_in->hVertices[i]);
				}
			}
		}

		// change using hminVertices if provided
		if (bamgopts->hminVertices){
			if (verbosity>1) printf("   Merging Metric with hminVertices...\n");
			for (i=0;i<BTh.nbv;i++){
				if (!isnan(bamgopts->hminVertices[i])){
					Metric M=BTh.vertices[i].m;
					MatVVP2x2 Vp(M/coef);
					Vp.Minh(bamgopts->hminVertices[i]);
					BTh.vertices[i].m=Vp;
				}
			}
		}

		// change using hmaxVertices if provided
		if (bamgopts->hmaxVertices){
			if (verbosity>1) printf("   Merging Metric with hmaxVertices...\n");
			for (i=0;i<BTh.nbv;i++){
				if (!isnan(bamgopts->hmaxVertices[i])){
					Metric M=BTh.vertices[i].m;
					MatVVP2x2 Vp(M/coef);
					Vp.Minh(bamgopts->hmaxVertices[i]);
					BTh.vertices[i].m=Vp;
				}
			}
		}

		//Add geometry metric if provided
		if(bamgopts->geometricalmetric) BTh.AddGeometryMetric(bamgopts);

		//Smoothe metric
		BTh.SmoothMetric(bamgopts->gradation);

		//Control element subdivision
		BTh.MaxSubDivision(bamgopts->maxsubdiv);

		//Bound anisotropy
		BTh.BoundAnisotropy(bamgopts->anisomax,hminaniso);

		//Build new mesh
		if (verbosity>1) printf("   Generating Mesh...\n");
		Thr=&BTh,Thb=0;
		Triangles & Th( *(0 ?  new Triangles(*Thr,&Thr->Gh,Thb,maxnbv) :  new Triangles(maxnbv,BTh,bamgopts,bamgopts->KeepVertices)));
		if (Thr != &BTh) delete Thr;

		//Make quadrangle if requested
		if(costheta<=1.0) Th.MakeQuadrangles(costheta);
		//if () Th.SplitElement(2);

		//Split corners if requested
		if(bamgopts->splitcorners) Th.SplitInternalEdgeWithBorderVertices();

		//Renumber by subdomain
		Th.ReNumberingTheTriangleBySubDomain();

		//Smooth vertices
		if(NbSmooth>0) Th.SmoothingVertex(NbSmooth,bamgopts->omega);

		//display info
		if(verbosity>0) {
			if (Th.nbt-Th.NbOutT-Th.NbOfQuad*2){
				printf("   new number of triangles = %i\n",(Th.nbt-Th.NbOutT-Th.NbOfQuad*2));
			}
			if (Th.NbOfQuad ){
				printf("   new number of quads = %i\n",Th.NbOfQuad);
			}
		}

		//Build output
		if (verbosity>1) printf("   Write Mesh...\n");
		Th.WriteMesh(bamgmesh_out,bamgopts);
		if (verbosity>1) printf("   Write Geometry...\n");
		Th.Gh.WriteGeometry(bamggeom_out,bamgopts);
		if (verbosity>1) printf("   Write Metric...\n");
		BTh.WriteMetric(bamgopts);

		/*clean up*/
		delete &Th;
		//delete &BTh;
		/*}}}*/
	}

	/*No error return*/
	if (verbosity>1) printf("   Exiting Bamg.\n");
	return noerr;

}
