/*!\file: transient_3d_core.cpp
 * \brief: core of the transient_3d solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../modules.h"

DataSet* transient_core_3d(Model* model){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;
	FemModel* fem_p=NULL;
	FemModel* fem_t=NULL;
	FemModel* fem_m=NULL;

	/*output: */
	DataSet* results=NULL;
	Result* result=NULL;

	/*solutions: */
	int step;
	double time;
	
	double* vx=NULL;
	double* vy=NULL;
	double* vz=NULL;
	
	Vec u_g=NULL;
	Vec p_g=NULL;
	Vec h_g=NULL;
	Vec h_g_intermediary=NULL;
	Vec s_g=NULL;
	Vec b_g=NULL;
	Vec t_g=NULL;
	Vec t_g_average=NULL;
	double melting_offset;
	Vec m_g=NULL;

	/*intermediary: */
	double finaltime;
	double dt;
	DataSet* diagnostic_results=NULL;
	DataSet* prognostic_results=NULL;


	/*flags: */
	int verbose=0;
	int numberofnodes;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};

	//initialize results
	results=new DataSet(ResultsEnum);

	/*recover fem models: */
	fem_dh=model->GetFormulation(DiagnosticAnalysisEnum,HorizAnalysisEnum);
	fem_dv=model->GetFormulation(DiagnosticAnalysisEnum,VertAnalysisEnum);
	fem_ds=model->GetFormulation(DiagnosticAnalysisEnum,StokesAnalysisEnum);
	fem_dhu=model->GetFormulation(DiagnosticAnalysisEnum,HutterAnalysisEnum);
	fem_sl=model->GetFormulation(SlopecomputeAnalysisEnum);
	fem_p=model->GetFormulation(PrognosticAnalysisEnum);
	fem_t=model->GetFormulation(ThermalAnalysisEnum);
	fem_m=model->GetFormulation(MeltingAnalysisEnum);


	//first recover parameters common to all solutions
	model->FindParam(&verbose,VerboseEnum);
	model->FindParam(&finaltime,NdtEnum);
	model->FindParam(&dt,DtEnum);
	model->FindParam(&numberofnodes,NumberOfDofsPerNodeEnum);

	/*initialize: */
	step=1;
	time=0;

	while(time<finaltime){ //make sure we run up to finaltime.
	
		if(verbose)_printf_("%s%g%s%i%s%g\n","time [yr]: ",time,"    iteration number: ",step,"/",floor(finaltime/dt));

		step+=1;
		time+=dt;

		//Deal with temperature first 
		if(verbose)_printf_("%s\n","computing temperature");
		thermal_core_nonlinear(&t_g,&melting_offset,fem_t,ThermalAnalysisEnum,TransientAnalysisEnum);
		if(verbose)_printf_("%s\n","computing melting");
		model->UpdateInputsFromVector(t_g,TemperatureEnum,VertexEnum);
		model->UpdateInputsFromConstant(melting_offset,MeltingOffsetEnum);
		diagnostic_core_linear(&m_g,fem_m,MeltingAnalysisEnum,TransientAnalysisEnum);

		//Compute depth averaged temperature and add to inputs
		if(verbose)_printf_("%s\n","computing depth average temperature");
		VecDuplicatePatch(&t_g_average,t_g); 
		FieldDepthAveragex( t_g_average, fem_t->elements,fem_t->nodes, fem_t->vertices,fem_t->loads, fem_t->materials,fem_t->parameters,"temperature");
		model->UpdateInputsFromVector(t_g_average,TemperatureAverageEnum,VertexEnum);
		VecFree(&t_g_average); //not needed anymore

		//Deal with velocities.
		diagnostic_results=diagnostic_core(model);

		//extract u_g and p_g from diagnostic_results, and erase diagnostic_results;
		diagnostic_results->FindResult(&u_g,"u_g");
		diagnostic_results->FindResult(&p_g,"p_g");
		delete diagnostic_results;

		//compute new thickness
		if(verbose)_printf_("%s\n","computing new thickness");
	
		SplitSolutionVectorx(u_g,numberofnodes,3,&vx,&vy,&vz);
		model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
		model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
		model->UpdateInputsFromVector(vz,VzEnum,VertexEnum);
		xfree((void**)&vx); xfree((void**)&vy); xfree((void**)&vz);
	
		prognostic_results=prognostic_core(model);
	
		//extract h_g prognostic_results, and erase prognostic_results;
		prognostic_results->FindResult(&h_g_intermediary,"h_g");
		delete prognostic_results;

		//update surface and bed using the new thickness
		if(verbose)_printf_("   updating geometry\n");
		UpdateGeometryx(&h_g,&b_g,&s_g, fem_p->elements, fem_p->nodes, fem_p->vertices,fem_p->loads, fem_p->materials, fem_p->parameters,h_g_intermediary,b_g,s_g); 
		VecFree(&h_g_intermediary);
		
		if(verbose)_printf_("%s\n","updating node positions");
		UpdateVertexPositionsx( fem_dh ->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_dv ->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_dhu->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_ds ->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_sl ->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_p  ->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_t  ->vertices,h_g,b_g);
		UpdateVertexPositionsx( fem_m  ->vertices,h_g,b_g);

		//plug into results.
		result=new Result(results->Size()+1,time,step,"u_g",u_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"p_g",p_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"h_g",h_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"s_g",s_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"b_g",b_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"t_g",t_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"m_g",m_g); results->AddObject(result);

		//update inputs
		model->UpdateInputsFromVector(h_g,ThicknessEnum,VertexEnum);
		model->UpdateInputsFromVector(s_g,SurfaceEnum,VertexEnum);
		model->UpdateInputsFromVector(b_g,BedEnum,VertexEnum);
	
		SplitSolutionVectorx(u_g,numberofnodes,3,&vx,&vy,&vz);
		model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
		model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
		model->UpdateInputsFromVector(vz,VzEnum,VertexEnum);
		xfree((void**)&vx); xfree((void**)&vy); xfree((void**)&vz);
		
		model->UpdateInputsFromVector(p_g,PressureEnum,VertexEnum);
		model->UpdateInputsFromVector(t_g,TemperatureEnum,VertexEnum);

		//Save temporary results every 5 steps
		if (step%5==0){
			_printf_("%s","      saving temporary results...");

			DataSet* processed_results=NULL;
			char*    outputfilename=NULL;

			model->FindParam(&outputfilename,OutputFileNameEnum);
			ProcessResults(&processed_results,results,model,TransientAnalysisEnum);
			OutputResults(processed_results,outputfilename);

			delete processed_results;
			xfree((void**)&outputfilename);

			_printf_("%s\n"," done.");
		}
	}

	/*Add analysis_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type",EnumAsString(TransientAnalysisEnum));
	results->AddObject(result);


	/*Free ressources:*/
	VecFree(&u_g);
	VecFree(&p_g);
	VecFree(&h_g);
	VecFree(&s_g);
	VecFree(&b_g);
	VecFree(&t_g);
	VecFree(&m_g);
	xfree((void**)&vx); 
	xfree((void**)&vy); 
	xfree((void**)&vz);
	
	//return: 
	return results;

}
