function converged=convergence(K_ff,p_f,u_f,u_f_old,parameters)

%Get convergence options
verbose=parameters.Verbose;
yts=parameters.Yts;
eps_res=parameters.EpsRes;
eps_rel=parameters.EpsRel;
eps_abs=parameters.EpsAbs;

%initialization
converged=0;
displaystring(verbose,' ');

%Display solver caracteristics
if (verbose>1),
	solver_res=norm(K_ff*u_f-p_f,2)/norm(p_f,2);
	displaystring(verbose>1,'%s%g','      condition number of stiffness matrix: ',condest(K_ff));
	displaystring(verbose>1,'%s%g','      solver residue: norm(KU-F)/norm(F)=',solver_res);
end

%Force equilibrium (Mandatory)
res=norm(K_ff*u_f_old-p_f,2)/norm(p_f,2);
if isnan(res),
	error('convergence error message: mechanical equilibrium convergence criterion is NaN!'); 
end
if (res<=eps_res),
	displaystring(verbose,'%-60s%g%s%g%s','      mechanical equilibrium convergence criterion',res*100,' < ',eps_res*100,' %');
	converged=1;
else
	displaystring(verbose,'%-60s%g%s%g%s','      mechanical equilibrium convergence criterion',res*100,' > ',eps_res*100,' %');
	converged=0;
end

%Relative criterion (optional)
if ((~isnan(eps_rel)) | (verbose>1)),

	%compute ndu/nu
	duf=u_f-u_f_old;
	ndu=norm(duf,2); 
	nu=norm(u_f_old,2); 
	if isnan(ndu),
		error('convergence error message: relative convergence criterion is NaN!'); 
	end

	%print criterion
	if ~isnan(eps_rel),
		if nu, %avoid "dividing by zero" warning
			if (ndu/nu<=eps_rel),
				displaystring(verbose,'%-60s%g%s%g%s','      relative convergence criterion: norm(du)/norm(u)',ndu/nu*100,' < ',eps_rel*100,' %');
			else
				displaystring(verbose,'%-60s%g%s%g%s','      relative convergence criterion: norm(du)/norm(u)',ndu/nu*100,' > ',eps_rel*100,' %');
				converged=0;
			end
		else
			converged=0;
		end
	else
		if nu, %avoid "dividing by zero" warning
			displaystring(verbose,'%-60s%g%s','      relative convergence criterion: norm(du)/norm(u)',ndu/nu*100,' %');
		end
	end

end

%Absolute criterion (optional)
if ((~isnan(eps_abs)) | (verbose>1)),

	%compute max(du)
	duf=u_f-u_f_old;
	nduinf=norm(duf,inf)*yts; 
	if isnan(nduinf),
		error('convergence error message: absolute convergence criterion is NaN!'); 
	end

	%print criterion
	if ~isnan(eps_abs),
		if (nduinf<=eps_abs),
			displaystring(verbose,'%-60s%g%s%g%s','      absolute convergence criterion: max(du)',nduinf,' < ',eps_abs,' m/yr');
		else
			displaystring(verbose,'%-60s%g%s%g%s','      absolute convergence criterion: max(du)',nduinf,' > ',eps_abs,' m/yr');
			converged=0;
		end
	else
		displaystring(verbose,'%-60s%g%s','      absolute convergence criterion: max(du)',nduinf,' m/yr');
	end

end
