/*!\file TriaVertexInput.c
 * \brief: implementation of the TriaVertexInput object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
#include "../../include/include.h"

/*Object constructors and destructor*/
/*FUNCTION TriaVertexInput::TriaVertexInput(){{{1*/
TriaVertexInput::TriaVertexInput(){
	return;
}
/*}}}*/
/*FUNCTION TriaVertexInput::TriaVertexInput(double* values){{{1*/
TriaVertexInput::TriaVertexInput(int in_enum_type,double* in_values){

	enum_type=in_enum_type;
	values[0]=in_values[0];
	values[1]=in_values[1];
	values[2]=in_values[2];
}
/*}}}*/
/*FUNCTION TriaVertexInput::~TriaVertexInput(){{{1*/
TriaVertexInput::~TriaVertexInput(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION TriaVertexInput::copy{{{1*/
Object* TriaVertexInput::copy() {
	
	return new TriaVertexInput(this->enum_type,this->values);

}
/*}}}*/
/*FUNCTION TriaVertexInput::DeepEcho{{{1*/
void TriaVertexInput::DeepEcho(void){

	printf("TriaVertexInput:\n");
	printf("   enum: %i\n",this->enum_type);
	printf("   %g|%g|%g\n",this->values[0],this->values[1],this->values[2]);
}
/*}}}*/
/*FUNCTION TriaVertexInput::Demarshall{{{1*/
void  TriaVertexInput::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(&values,marshalled_dataset,sizeof(values));marshalled_dataset+=sizeof(values);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Echo {{{1*/
void TriaVertexInput::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION TriaVertexInput::Enum{{{1*/
int TriaVertexInput::Enum(void){

	return TriaVertexInputEnum;

}
/*}}}*/
/*FUNCTION TriaVertexInput::EnumType{{{1*/
int TriaVertexInput::EnumType(void){

	return this->enum_type;

}
/*}}}*/
/*FUNCTION TriaVertexInput::Id{{{1*/
int    TriaVertexInput::Id(void){ return -1; }
/*}}}*/
/*FUNCTION TriaVertexInput::Marshall{{{1*/
void  TriaVertexInput::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of TriaVertexInput: */
	enum_value=TriaVertexInputEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall TriaVertexInput data: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(marshalled_dataset,&values,sizeof(values));marshalled_dataset+=sizeof(values);

	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION TriaVertexInput::MarshallSize{{{1*/
int   TriaVertexInput::MarshallSize(){
	
	return sizeof(values)+
		+sizeof(enum_type)+
		+sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION TriaVertexInput::MyRank{{{1*/
int    TriaVertexInput::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Object functions*/
/*FUNCTION TriaVertexInput::GetParameterValue(double* pvalue,double* gauss){{{1*/
void TriaVertexInput::GetParameterValue(double* pvalue,double* gauss){
	/*P1 interpolation on Gauss point*/

	/*intermediary*/
	double l1l2l3[3];

	/*nodal functions: */
	GetNodalFunctions(l1l2l3,gauss);

	/*Assign output pointers:*/
	*pvalue=l1l2l3[0]*values[0]+l1l2l3[1]*values[1]+l1l2l3[2]*values[2];

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterValues(double* values,double* gauss_pointers, int numgauss){{{1*/
void TriaVertexInput::GetParameterValues(double* values,double* gauss_pointers, int numgauss){
	/*It is assume that output values has been correctly allocated*/

	int i,j;
	double gauss[3];

	for (i=0;i<numgauss;i++){

		/*Get current Gauss point coordinates*/
		for (j=0;j<3;j++) gauss[j]=gauss_pointers[i*3+j];

		/*Assign parameter value*/
		GetParameterValue(&values[i],&gauss[0]);
	}

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss){{{1*/
void TriaVertexInput::GetParameterDerivativeValue(double* p, double* xyz_list, double* gauss){
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter derivative value at gaussian 
	 * point specified by gauss_l1l2l3:
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *
	 * p is a vector of size 2x1 already allocated.
	 */

	const int NDOF2=2;
	const int numgrids=3;
	double dh1dh3[NDOF2][numgrids]; //nodal derivative functions in actual coordinate system.

	/*Get nodal funnctions derivatives in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss);

	p[0]=this->values[0]*dh1dh3[0][0]+this->values[1]*dh1dh3[0][1]+this->values[2]*dh1dh3[0][2];
	p[1]=this->values[0]*dh1dh3[1][0]+this->values[1]*dh1dh3[1][1]+this->values[2]*dh1dh3[1][2];

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVxStrainRate2d(double* epsilonvx,double* xyz_list, double* gauss,int formulation_enum) {{{1*/
void TriaVertexInput::GetVxStrainRate2d(double* epsilonvx,double* xyz_list, double* gauss,int formulation_enum){

	/*Intermediary*/
	int       i;
	const int numgrids=3;
	const int NDOF2=2;
	double B[3][NDOF2*numgrids];
	double velocity[3][NDOF2];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss);

	/*Here, we are computing the strain rate of (vx,0)*/
	for(i=0;i<3;i++){
		velocity[i][0]=this->values[i];
		velocity[i][1]=0.0;
	}
	/*Get epsilon(vx) = B*velocity*/
	MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
				&velocity[0][0],NDOF2*numgrids,1,0,
				epsilonvx,0);
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVyStrainRate2d(double* epsilonvy,double* xyz_list, double* gauss,int formulation_enum) {{{1*/
void TriaVertexInput::GetVyStrainRate2d(double* epsilonvy,double* xyz_list, double* gauss,int formulation_enum){

	/*Intermediary*/
	int       i;
	const int numgrids=3;
	const int NDOF2=2;
	double B[3][NDOF2*numgrids];
	double velocity[3][NDOF2];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss);

	/*Here, we are computing the strain rate of (0,vy)*/
	for(i=0;i<3;i++){
		velocity[i][0]=0.0;
		velocity[i][1]=this->values[i];
	}
	/*Get epsilon(vy) = B*velocity*/
	MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
				&velocity[0][0],NDOF2*numgrids,1,0,
				epsilonvy,0);
}
/*}}}*/
/*FUNCTION TriaVertexInput::ChangeEnum(int newenumtype){{{1*/
void TriaVertexInput::ChangeEnum(int newenumtype){
	this->enum_type=newenumtype;
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterAverage(double* pvalue){{{1*/
void TriaVertexInput::GetParameterAverage(double* pvalue){
	*pvalue=1./3.*(values[0]+values[1]+values[2]);
}
/*}}}*/

/*Intermediary*/
/*FUNCTION TriaVertexInput::GetNodalFunctions {{{1*/
void TriaVertexInput::GetNodalFunctions(double* l1l2l3, double* gauss){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2l3[0]=gauss[0];

	/*Second nodal function: */
	l1l2l3[1]=gauss[1];

	/*Third nodal function: */
	l1l2l3[2]=gauss[2];

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetB {{{1*/
void TriaVertexInput::GetB(double* B, double* xyz_list, double* gauss){
	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dh/dx           0    ]
	 *          [   0           dh/dy  ]
	 *          [ 1/2*dh/dy  1/2*dh/dx ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double dh1dh3[NDOF2][numgrids];


	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh3[0][i]; //B[0][NDOF2*i]=dh1dh3[0][i];
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh3[1][i];
		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh3[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh3[0][i]; 
	}
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetNodalFunctionsDerivatives {{{1*/
void TriaVertexInput::GetNodalFunctionsDerivatives(double* dh1dh3,double* xyz_list, double* gauss){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system: */
	int i;
	const int NDOF2=2;
	const int numgrids=3;
	double dh1dh3_ref[NDOF2][numgrids];
	double Jinv[NDOF2][NDOF2];

	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReference(&dh1dh3_ref[0][0], gauss); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss);

	/*Build dh1dh3: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 */
	for (i=0;i<numgrids;i++){
		dh1dh3[numgrids*0+i]=Jinv[0][0]*dh1dh3_ref[0][i]+Jinv[0][1]*dh1dh3_ref[1][i];
		dh1dh3[numgrids*1+i]=Jinv[1][0]*dh1dh3_ref[0][i]+Jinv[1][1]*dh1dh3_ref[1][i];
	}

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetNodalFunctionsDerivativesReference {{{1*/
void TriaVertexInput::GetNodalFunctionsDerivativesReference(double* dl1dl3,double* gauss_l1l2l3){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	const int NDOF2=2;
	const int numgrids=3;

	/*First nodal function: */
	*(dl1dl3+numgrids*0+0)=-0.5; 
	*(dl1dl3+numgrids*1+0)=-1.0/(2.0*SQRT3);

	/*Second nodal function: */
	*(dl1dl3+numgrids*0+1)=0.5;
	*(dl1dl3+numgrids*1+1)=-1.0/(2.0*SQRT3);

	/*Third nodal function: */
	*(dl1dl3+numgrids*0+2)=0;
	*(dl1dl3+numgrids*1+2)=1.0/SQRT3;

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetJacobian {{{1*/
void TriaVertexInput::GetJacobian(double* J, double* xyz_list,double* gauss){

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	const int NDOF2=2;
	const int numgrids=3;
	double x1,y1,x2,y2,x3,y3;

	x1=*(xyz_list+numgrids*0+0);
	y1=*(xyz_list+numgrids*0+1);
	x2=*(xyz_list+numgrids*1+0);
	y2=*(xyz_list+numgrids*1+1);
	x3=*(xyz_list+numgrids*2+0);
	y3=*(xyz_list+numgrids*2+1);


	*(J+NDOF2*0+0)=0.5*(x2-x1);
	*(J+NDOF2*1+0)=SQRT3/6.0*(2*x3-x1-x2);
	*(J+NDOF2*0+1)=0.5*(y2-y1);
	*(J+NDOF2*1+1)=SQRT3/6.0*(2*y3-y1-y2);
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetJacobianInvert {{{1*/
void TriaVertexInput::GetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss){

	double Jdet;
	const int NDOF2=2;
	const int numgrids=3;

	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(Jinv, xyz_list, gauss);

	/*Invert Jacobian matrix: */
	MatrixInverse(Jinv,NDOF2,NDOF2,NULL,0,&Jdet);

}
/*}}}*/
