/*
 * \file Inputs.c
 * \brief: implementation of the Inputs class, derived from DataSet class
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Inputs::Inputs(){{{1*/
Inputs::Inputs(){
	return;
}
/*}}}*/
/*FUNCTION Inputs::Inputs(int in_enum){{{1*/
Inputs::Inputs(int in_enum): DataSet(in_enum) {
	//do nothing;
	return;
}
/*}}}*/
/*FUNCTION Inputs::~Inputs(){{{1*/
Inputs::~Inputs(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,gauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type,double defaultvalue){{{1*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type,double defaultvalue){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. Return the default value: */
		*pvalue=defaultvalue;
	}
	else{
		input->GetParameterValue(pvalue,gauss);
	}
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){{{1*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValues(values,gauss_pointers,numgauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue, Node* node, int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,Node* node,int enum_type){

	/*given a node, instead of a gauss point, we want to recover a value: probably in an element!: */

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,node);
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue, Node* node1, Node* node2,int enum_type){{{1*/
void Inputs::GetParameterValue(double* pvalue,Node* node1, Node* node2,double gauss_coord,int enum_type){

	/*given a node, instead of a gauss point, we want to recover a value: probably in an element!: */

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,node1,node2,gauss_coord);
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){{{1*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){

	int i;
	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. Return the default values: */
		for(i=0;i<numgauss;i++)values[i]=defaultvalues[i];
	}
	else{
		input->GetParameterValues(values,gauss_pointers,numgauss);
	}

}
/*}}}*/
/*FUNCTION Inputs::GetParameterAverage(double* pvalue,int enum-type){{{1*/
void Inputs::GetParameterAverage(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterAverage(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(bool* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(bool* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(int* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(int* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){{{1*/
void Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterDerivativeValue(derivativevalues,xyz_list,gauss);
}
/*}}}*/
/*FUNCTION Inputs::GetStrainRate2d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum, int formulation_enum){{{1*/
void Inputs::GetStrainRate2d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum,int formulation_enum){

	vector<Object*>::iterator object;
	int i;
	Input* vxinput=NULL;
	Input* vyinput=NULL;
	double epsilonvx[3];
	double epsilonvy[3];

	/*Go through inputs and find data for vxenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vxinput=(Input*)(*object); 
		if (vxinput->EnumType()==vxenum)break;
	}
	/*Go through inputs and find data for vyenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vyinput=(Input*)(*object); 
		if (vyinput->EnumType()==vyenum)break;
	}

	/*Check that both inputs have been found*/
	if (!vxinput || !vyinput){
		ISSMERROR("Could not find input with enum %i or enum %i",vxenum,vyenum);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vxinput->GetVxStrainRate2d(epsilonvx,xyz_list,gauss,formulation_enum);
	vyinput->GetVyStrainRate2d(epsilonvy,xyz_list,gauss,formulation_enum);
	for(i=0;i<3;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];

}
/*}}}*/
/*FUNCTION Inputs::GetStrainRate3d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum, int formulation_enum){{{1*/
void Inputs::GetStrainRate3d(double* epsilon,double* xyz_list, double* gauss, int vxenum, int vyenum,int vzenum,int formulation_enum){

	vector<Object*>::iterator object;
	Input* vxinput=NULL;
	Input* vyinput=NULL;
	Input* vzinput=NULL;

	/*Go through inputs and find data for vxenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vxinput=(Input*)(*object); 
		if (vxinput->EnumType()==vxenum)break;
	}
	/*Go through inputs and find data for vyenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vyinput=(Input*)(*object); 
		if (vyinput->EnumType()==vyenum)break;
	}
	/*Go through inputs and find data for vzenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		vzinput=(Input*)(*object); 
		if (vzinput->EnumType()==vzenum)break;
	}

	if (!vxinput || !vyinput || !vzinput){
		/*we could not find one input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("Could not find input with enum %i or enum %i or enum %i",vxenum,vyenum,vzenum);
	}

	ISSMERROR("STOP");

}
/*}}}*/
/*FUNCTION Inputs::AddInput{{{1*/
int  Inputs::AddInput(Input* in_input){

	/*First, go through dataset of inputs and check whether any input 
	 * with the same name is already in. If so, erase the corresponding 
	 * object before adding this new one: */
	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==in_input->EnumType()){
			this->DeleteObject(input);
			break;
		}
	}
	this->AddObject(in_input);
}
/*}}}*/
/*FUNCTION Inputs::ChangeEnum{{{1*/
void  Inputs::ChangeEnum(int enumtype,int newenumtype){

	/*Go through dataset of inputs and look for input with 
	 * same enum as input enum, once found, just change its name */
	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==enumtype){
			input->ChangeEnum(newenumtype);
			break;
		}
	}
}
/*}}}*/
