function results=diagnostic_core(model);
%DIAGNOSTIC_CORE - compute the core velocity field 
%
%   Usage:
%      results=diagnostic_core(model);
%

%recover fem models
m_dh=model.dh;
m_dv=model.dv;
m_ds=model.ds;
m_dhu=model.dhu;
m_sl=model.sl;

%recover parameters common to all solutions
verbose=m_dhu.parameters.Verbose;
dim=m_dh.parameters.Dim;
ishutter=m_dhu.parameters.IsHutter;
ismacayealpattyn=m_dh.parameters.IsMacAyealPattyn;
isstokes=m_ds.parameters.IsStokes;
numrifts=m_dhu.parameters.NumRifts;
qmu_analysis=m_dh.parameters.QmuAnalysis;

	
%for qmu analysis, be sure the velocity input we are starting from  is the one in the parameters: 
if qmu_analysis,
	ModelUpdateInputsFromVector(model,m_dh.vx,VxEnum,VertexEnum);
	ModelUpdateInputsFromVector(model,m_dh.vy,VyEnum,VertexEnum);
	ModelUpdateInputsFromVector(model,m_dh.vz,VzEnum,VertexEnum);
end

%Compute slopes: 
[surfaceslopex,surfaceslopey]=slope_core(m_sl,SurfaceAnalysisEnum);
[bedslopex,bedslopey]=slope_core(m_sl,BedAnalysisEnum);

%Update:
ModelUpdateInputsFromVector(model,surfaceslopex,SurfaceSlopexEnum,VertexEnum);
ModelUpdateInputsFromVector(model,surfaceslopey,SurfaceSlopeyEnum,VertexEnum);
ModelUpdateInputsFromVector(model,bedslopex,BedSlopexEnum,VertexEnum);
ModelUpdateInputsFromVector(model,bedslopey,BedSlopeyEnum,VertexEnum);

if ishutter,

	displaystring(verbose,'\n%s',['computing hutter velocities...']);
	u_g=diagnostic_core_linear(m_dhu,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

	displaystring(verbose,'\n%s',['computing pressure according to MacAyeal...']);
	p_g=ComputePressure(m_dhu.elements,m_dhu.nodes,m_dhu.vertices,m_dhu.loads,m_dhu.materials,m_dhu.parameters,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

	displaystring(verbose,'\n%s',['update boundary conditions for macyeal pattyn using hutter results...']);
	if ismacayealpattyn,
		m_dh.y_g=u_g;
		[m_dh.ys m_dh.ys0]=Reducevectorgtos(m_dh.y_g,m_dh.nodesets);
	end

end
		
if ismacayealpattyn,

	displaystring(verbose,'\n%s',['computing horizontal velocities...']);
	[u_g m_dh.loads]=diagnostic_core_nonlinear(m_dh,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

	if dim==2,
		displaystring(verbose,'\n%s',['computing pressure according to MacAyeal...']);
		p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.vertices,m_dh.loads,m_dh.materials,m_dh.parameters,DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	end
end
	
if dim==3,

	displaystring(verbose,'\n%s',['extruding horizontal velocities...']);
	u_g_horiz=FieldExtrude(m_dh.elements,m_dh.nodes,m_dh.vertices,m_dh.loads,m_dh.materials,m_dh.parameters,u_g,'velocity',1);

	[vx,vy]=SplitSolutionVector(ug,numberofnodes,numberofdofspernode_dh);
	ModelUpdateInputsFromVector(model,vx,VxEnum,VertexEnum);
	ModelUpdateInputsFromVector(model,vy,VyEnum,VertexEnum);
		
	displaystring(verbose,'\n%s',['computing vertical velocities...']);
	u_g_vert=diagnostic_core_linear(m_dv,DiagnosticAnalysisEnum(),VertAnalysisEnum());
	ModelUpdateInputsFromVector(model,u_g_vert,VzEnum,VertexEnum);

	displaystring(verbose,'\n%s',['computing pressure according to Pattyn...']);
	p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.vertices,m_dh.loads,m_dh.materials,m_dh.parameters,DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	ModelUpdateInputsFromVector(model,p_g,PressureEnum,VertexEnum);
	
	if isstokes,

		%"recondition" pressure 
		p_g=p_g/m_ds.parameters.stokesreconditioning;

		displaystring(verbose,'\n%s',['update boundary conditions for stokes using velocities previously computed...']);
		m_ds.y_g=zeros(m_ds.nodesets.gsize,1);
		m_ds.y_g(dofsetgen([1,2],4,m_ds.nodesets.gsize))=u_g;
		m_ds.y_g(dofsetgen([3],4,m_ds.nodesets.gsize))=u_g_vert;
		[m_ds.ys m_ds.ys0]=Reducevectorgtos(m_ds.y_g,m_ds.nodesets);

		displaystring(verbose,'\n%s',['computing stokes velocities and pressure ...']);
		u_g=diagnostic_core_nonlinear(m_ds,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	
		%"decondition" pressure
		p_g=u_g(4:4:end)*m_dh.parameters.stokesreconditioning;
	end
end
%load onto results
results.step=1;
results.time=0;
results.u_g=u_g;
results.p_g=p_g;

if numrifts,
	results.riftproperties=OutputRifts(m_dh.loads,m_dh.parameters);
end
