/*!\file Model.c
 * \brief: implementation of the Model object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "../DataSet/DataSet.h"
#include "../ModelProcessorx/ModelProcessorx.h"
#include "./objects.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules.h"


/*Object constructors and destructor*/
/*FUNCTION Model::constructor {{{1*/
Model::Model(){

	femmodels=new DataSet(0);
	active=NULL;

}
/*}}}1*/
/*FUNCTION Model::destructor {{{1*/
Model::~Model(){

	delete femmodels;
	/*do not delete active, it just points to one of the femmodels!*/
}
/*}}}1*/

/*Object functions*/
/*FUNCTION Model::AddFormulation(ConstDataHandle MODEL, int analysis_type,int sub_analysis_type){{{1*/
void  Model::AddFormulation(ConstDataHandle MODEL, int analysis_type,int sub_analysis_type){

	/*FemModel: */
	FemModel*           femmodel=NULL;
	DataSet*            elements=NULL;
	DataSet*            nodes=NULL;
	DataSet*            vertices=NULL;
	DataSet*            constraints=NULL;
	DataSet*            loads=NULL;
	DataSet*            materials=NULL;
	Parameters*            parameters=NULL;
	DofVec*                 partition=NULL;
	DofVec*                 tpartition=NULL;
	DofVec*                 yg=NULL;
	Mat                 Rmg=NULL;
	Mat                 Gmn=NULL;
	NodeSets*           nodesets=NULL;
	Vec                 ys=NULL;
	Vec                 ys0=NULL;

	/*intermediary: */
	IoModel* iomodel=NULL;
	
	_printf_("   fill model with matlab workspace data\n");
	iomodel=new IoModel(MODEL);

	_printf_("   specifying analysis\n");
	if (analysis_type!=0){
		iomodel->analysis_type=analysis_type;
	}
	if (sub_analysis_type!=0){
		iomodel->sub_analysis_type=sub_analysis_type;
	}

	_printf_("   create datasets:\n");
	CreateDataSets(&elements,&nodes,&vertices,&materials,&constraints,&loads,&parameters,iomodel,MODEL);

	_printf_("   create degrees of freedom: \n");
	Dofx( &partition,&tpartition,elements,nodes, vertices,parameters);
	
	_printf_("   create single point constraints: \n");
	SpcNodesx( &yg, nodes,constraints); 
	
	_printf_("   create rigid body constraints:\n");
	MpcNodesx( &Rmg, nodes,constraints); 
	
	_printf_("   create node sets:\n");
	BuildNodeSetsx(&nodesets, nodes);

	_printf_("   reducing single point constraints vector:\n");
	Reducevectorgtosx(&ys,&ys0, yg->vector,nodesets);
	
	_printf_("   normalizing rigid body constraints matrix:\n");
	NormalizeConstraintsx(&Gmn, Rmg,nodesets);

	_printf_("   configuring element and loads:\n");
	ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters);

	_printf_("   process parameters:\n");
	ProcessParamsx( parameters, partition->vector);

	_printf_("   free ressources:\n");
	delete iomodel;

	/*Use all the data created to create an femmodel: */
	femmodel=new FemModel(elements, nodes, vertices, constraints,loads,materials,parameters,
			              partition,tpartition,yg,Rmg,Gmn,nodesets,ys,ys0);

	/*Add to femmodels in model: */
	femmodels->AddObject((Object*)femmodel);

}
/*}}}1*/
/*FUNCTION Model::AddFormulation(ConstDataHandle MODEL, int analysis_type) {{{1*/
void  Model::AddFormulation(ConstDataHandle MODEL, int analysis_type){

	/*FemModel: */
	FemModel*           femmodel=NULL;
	DataSet*            elements=NULL;
	DataSet*            nodes=NULL;
	DataSet*            vertices=NULL;
	DataSet*            constraints=NULL;
	DataSet*            loads=NULL;
	DataSet*            materials=NULL;
	Parameters*            parameters=NULL;
	DofVec*                 partition=NULL;
	DofVec*                 tpartition=NULL;
	DofVec*                 yg=NULL;
	Mat                 Rmg=NULL;
	Mat                 Gmn=NULL;
	NodeSets*           nodesets=NULL;
	Vec                 ys=NULL;
	Vec                 ys0=NULL;

	/*intermediary: */
	IoModel* iomodel=NULL;
	int sub_analysis_type;
	
	_printf_("   fill model with matlab workspace data\n");
	iomodel=new IoModel(MODEL);

	_printf_("   specifying analysis\n");
	if (analysis_type!=0){
		iomodel->analysis_type=analysis_type;
	}
	/*sub_analysis_type should default to none, as it was not specified: */
	sub_analysis_type=NoneAnalysisEnum;
	
	if (sub_analysis_type!=0){
		iomodel->sub_analysis_type=sub_analysis_type;
	}

	_printf_("   create datasets:\n");
	CreateDataSets(&elements,&nodes,&vertices,&materials,&constraints,&loads,&parameters,iomodel,MODEL);

	_printf_("   create degrees of freedom: \n");
	Dofx( &partition,&tpartition,elements,nodes, vertices,parameters);
	
	_printf_("   create single point constraints: \n");
	SpcNodesx( &yg, nodes,constraints); 
	
	_printf_("   create rigid body constraints:\n");
	MpcNodesx( &Rmg, nodes,constraints); 
	
	_printf_("   create node sets:\n");
	BuildNodeSetsx(&nodesets, nodes);

	_printf_("   reducing single point constraints vector:\n");
	Reducevectorgtosx(&ys,&ys0, yg->vector,nodesets);
	
	_printf_("   normalizing rigid body constraints matrix:\n");
	NormalizeConstraintsx(&Gmn, Rmg,nodesets);

	_printf_("   configuring element and loads:\n");
	ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters);

	_printf_("   process parameters:\n");
	ProcessParamsx( parameters, partition->vector);

	_printf_("   free ressources:\n");
	delete iomodel;

	/*Use all the data created to create an femmodel: */
	femmodel=new FemModel(elements,nodes,vertices, constraints,loads,materials,parameters,
			              partition,tpartition,yg,Rmg,Gmn,nodesets,ys,ys0);

	/*Add to femmodels in model: */
	femmodels->AddObject((Object*)femmodel);

}
/*}}}1*/
/*FUNCTION Model::DeepEcho {{{1*/

void Model::DeepEcho(void){

	femmodels->Echo();

}
/*}}}1*/
/*FUNCTION Model::Echo {{{1*/

void Model::Echo(void){

	printf("Models echo: \n");
	printf("   number of fem models: %i\n",femmodels->Size());

}
/*}}}1*/
/*FUNCTION Model::FindParam(bool* pparameter,int enum_type {{{1*/

int   Model::FindParam(bool* pparameter,int enum_type){

	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);

	if(!femmodel)return 0;

	femmodel->FindParam(pparameter,enum_type);

}
/*}}}1*/
/*FUNCTION Model::FindParam(int* pparameter,int enum_type {{{1*/

int   Model::FindParam(int* pparameter,int enum_type){

	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);

	if(!femmodel)return 0;

	femmodel->FindParam(pparameter,enum_type);

}
/*}}}1*/
/*FUNCTION Model::FindParam(double* pparameter,int enum_type ){{{1*/
int   Model::FindParam(double* pparameter,int enum_type){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,enum_type);


}
/*}}}1*/
/*FUNCTION Model::FindParam(double** pparameter,int* pM, int *pN,int enum_type) {{{1*/
int   Model::FindParam(double** pparameter,int* pM, int *pN,int enum_type){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,pM, pN,enum_type);


}
/*}}}1*/
/*FUNCTION Model::FindParam(double** pparameter,int* pM, int enum_type) {{{1*/
int   Model::FindParam(double** pparameter,int* pM, int enum_type){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,pM, enum_type);


}
/*}}}1*/
/*FUNCTION Model::FindParam(char** pparameter,int enum_type) {{{1*/
int   Model::FindParam(char** pparameter,int enum_type){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,enum_type);

}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysisAndSub(bool* pparameter,int enum_type,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParamByAnalysisAndSub(bool* pparameter,int enum_type,int analysis_type,int sub_analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysisAndSub(int* pparameter,int enum_type,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParamByAnalysisAndSub(int* pparameter,int enum_type,int analysis_type,int sub_analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysisAndSub(double* pparameter,int enum_type,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParamByAnalysisAndSub(double* pparameter,int enum_type,int analysis_type,int sub_analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysisAndSub(double** pparameter,int* pM, int* pN,int enum_type,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParamByAnalysisAndSub(double** pparameter,int* pM, int* pN,int enum_type,int analysis_type,int sub_analysis_type){

	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,pM, pN,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysisAndSub(double** pparameter,int* pM, int enum_type,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParamByAnalysisAndSub(double** pparameter,int* pM, int enum_type,int analysis_type,int sub_analysis_type){

	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,pM, enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysisAndSub(char** pparameter,int enum_type,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParamByAnalysisAndSub(char** pparameter,int enum_type,int analysis_type,int sub_analysis_type){

	FemModel* femmodel=NULL;
	
	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysis(bool* pparameter,int enum_type,int analysis_type) {{{1*/
int   Model::FindParamByAnalysis(bool* pparameter,int enum_type,int analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysis(int* pparameter,int enum_type,int analysis_type) {{{1*/
int   Model::FindParamByAnalysis(int* pparameter,int enum_type,int analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysis(double* pparameter,int enum_type,int analysis_type) {{{1*/
int   Model::FindParamByAnalysis(double* pparameter,int enum_type,int analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysis(double** pparameter,int* pM, int* pN,int enum_type,int analysis_type) {{{1*/
int   Model::FindParamByAnalysis(double** pparameter,int* pM, int* pN,int enum_type,int analysis_type){

	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,pM,pN,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysis(double** pparameter,int* pM, int enum_type,int analysis_type) {{{1*/
int   Model::FindParamByAnalysis(double** pparameter,int* pM, int enum_type,int analysis_type){

	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,pM,enum_type);
}
/*}}}1*/
/*FUNCTION Model::FindParamByAnalysis(char** pparameter,int enum_type,int analysis_type) {{{1*/
int   Model::FindParamByAnalysis(char** pparameter,int enum_type,int analysis_type){

	FemModel* femmodel=NULL;
	
	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,enum_type);
}	
/*}}}1*/
/*FUNCTION Model::GetActiveFormulation {{{1*/
FemModel* Model::GetActiveFormulation(){return active;}
/*}}}1*/
/*FUNCTION Model::GetFormulation(int analysis_type,int sub_analysis_type) {{{1*/

FemModel* Model::GetFormulation(int analysis_type,int sub_analysis_type){
	
	int i;
	FemModel* femmodel=NULL;
	int femmodel_analysis_type;
	int femmodel_sub_analysis_type;
	int found=0;
	
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*find femmodel with correct analysis_type and sub_analysis_type: */
	for(i=0;i<femmodels->Size();i++){

		femmodel=(FemModel*)femmodels->GetObjectByOffset(i);
		femmodel->FindParam(&femmodel_analysis_type,AnalysisTypeEnum);
		femmodel->FindParam(&femmodel_sub_analysis_type,SubAnalysisTypeEnum);

		if((analysis_type==femmodel_analysis_type) && (sub_analysis_type==femmodel_sub_analysis_type)){
			found=1;
			break;
		}
	}

	if(!found)return NULL;
	else return femmodel;
}
/*}}}1*/
/*FUNCTION Model::GetFormulation(int analysis_type) {{{1*/

FemModel* Model::GetFormulation(int analysis_type){
	
	int i;
	FemModel* femmodel=NULL;
	int femmodel_analysis_type;
	int femmodel_sub_analysis_type;
	int found=0;
	
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*find femmodel with correct analysis_type and sub_analysis_type: */
	for(i=0;i<femmodels->Size();i++){

		femmodel=(FemModel*)femmodels->GetObjectByOffset(i);
		femmodel->FindParam(&femmodel_analysis_type,AnalysisTypeEnum);

		if((analysis_type==femmodel_analysis_type)){
			found=1;
			break;
		}
	}

	if(!found)return NULL;
	else return femmodel;
}
/*}}}1*/
/*FUNCTION Model::SetActiveFormulation {{{1*/
void* Model::SetActiveFormulation(FemModel* femmodel){
	active=femmodel;
}
/*}}}1*/
/*FUNCTION Model::UpdateInputsFromConstant(double constant, int name);{{{1*/
void  Model::UpdateInputsFromConstant(double constant, int name){

	int i;
	FemModel* femmodel=NULL;

	for(i=0;i<this->femmodels->Size();i++){
		femmodel=(FemModel*)this->femmodels->GetObjectByOffset(i);
		femmodel->UpdateInputsFromConstant(constant,name);
	}

}
/*}}}1*/
/*FUNCTION Model::UpdateInputsFromConstant(int constant, int name);{{{1*/
void  Model::UpdateInputsFromConstant(int constant, int name){

	int i;
	FemModel* femmodel=NULL;

	for(i=0;i<this->femmodels->Size();i++){
		femmodel=(FemModel*)this->femmodels->GetObjectByOffset(i);
		femmodel->UpdateInputsFromConstant(constant,name);
	}

}
/*}}}1*/
/*FUNCTION Model::UpdateInputsFromConstant(bool constant, int name);{{{1*/
void  Model::UpdateInputsFromConstant(bool constant, int name){

	int i;
	FemModel* femmodel=NULL;

	for(i=0;i<this->femmodels->Size();i++){
		femmodel=(FemModel*)this->femmodels->GetObjectByOffset(i);
		femmodel->UpdateInputsFromConstant(constant,name);
	}

}
/*}}}1*/
/*FUNCTION Model::UpdateInputsFromVector(double* vector,int name, int type);{{{1*/
void  Model::UpdateInputsFromVector(double* vector,int name, int type){

	int i;
	FemModel* femmodel=NULL;

	if(vector==NULL)return; //don't bother

	for(i=0;i<this->femmodels->Size();i++){
		femmodel=(FemModel*)this->femmodels->GetObjectByOffset(i);
		femmodel->UpdateInputsFromVector(vector,name,type);
	}

}
/*}}}1*/
/*FUNCTION Model::UpdateInputsFromVector(Vec vector,int name, int type);{{{1*/
void  Model::UpdateInputsFromVector(Vec vector,int name, int type){

	int i;
	FemModel* femmodel=NULL;
	double* serial_vector=NULL;

	if(vector==NULL)return; //don't bother

	VecToMPISerial(&serial_vector,vector);

	//call double* idential routine.
	this->UpdateInputsFromVector(serial_vector,name,type);

	/*Free ressources:*/
	xfree((void**)&serial_vector);
}
/*}}}1*/
/*FUNCTION Model::UpdateInputsFromSolution(double* vector,int analysis_type, int sub_analysis_type);{{{1*/
void  Model::UpdateInputsFromSolution(double* vector,int analysis_type, int sub_analysis_type){

	int i;
	FemModel* femmodel=NULL;

	for(i=0;i<this->femmodels->Size();i++){
		femmodel=(FemModel*)this->femmodels->GetObjectByOffset(i);
		femmodel->UpdateInputsFromSolution(vector,analysis_type,sub_analysis_type);
	}

}
/*}}}1*/
/*FUNCTION Model::UpdateFromDakota(double* variables,char** variables_descriptors,int numvariables,DataSet* parameters, double* qmu_part,int qmu_npart);{{{1:*/
void    UpdateFromDakota(double* variables,char** variables_descriptors,int numvariables,DataSet* parameters, double* qmu_part,int qmu_npart){
	ISSMERROR("not supported yet!");
}
/*}}}*/
