/*!\file:  SpawnCoreParallel.cpp
 * \brief: run core ISSM solution using Dakota inputs coming from CPU 0.
 * \sa qmu.cpp DakotaPlugin.cpp
 *
 * This routine needs to be understood simultaneously with qmu.cpp and DakotaPlugin. 
 * SpawnCoreParallel is called by all CPUS, with CPU 0 holding Dakota variable values, along 
 * with variable descriptors. 
 *
 * SpawnCoreParallel takes care of broadcasting the variables and their descriptors across the MPI 
 * ring. Once this is done, we use the variables to modify the inputs for the solution core. 
 * For ex, if "rho_ice" is provided, for ex 920, we include "rho_ice" in the inputs, then 
 * call the core with the modified inputs. This is the way we get Dakota to explore the parameter 
 * spce of the core. 
 *
 * Once the core is called, we process the results of the core, and using the processed results, 
 * we compute response functions. The responses are computed on all CPUS, but they are targeted 
 * for CPU 0, which will get these values back to the Dakota engine. 
 *
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


#include "../objects/objects.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "./Qmux.h"
#include "../include/include.h"
#include "../parallel/parallel.h"

void SpawnCoreParallel(double* responses, int numresponses, double* variables, char** variables_descriptors,int numvariables, Model* model,int analysis_type,int sub_analysis_type,int counter){

	int i;
	
	/*output from core solutions: */
	DataSet* results=NULL;
	DataSet* processed_results=NULL;

	char** responses_descriptors=NULL;
	int    num_responses_descriptors;
	char*  string=NULL;
	int    string_length;
	double* qmu_part=NULL;
	int     qmu_npart;
	int     verbose=0;
	int     dummy;

	extern int my_rank;
	
	
	/*synchronize all cpus, as CPU 0 is probably late (it is starting the entire dakota strategy!) : */
	MPI_Barrier(MPI_COMM_WORLD);
	
	/*some parameters needed: */
	model->FindParam(&verbose,VerboseEnum);
		
	/*First off, recover the response descriptors for the response functions: */
	model->GetFormulation(DiagnosticAnalysisEnum,HorizAnalysisEnum)->parameters->FindParam(&responses_descriptors,&num_responses_descriptors,ResponseDescriptorsEnum);

	/*Recover partitioning for dakota: */
	model->FindParam(&qmu_npart,QmuNPartEnum);
	model->FindParam(&qmu_part,&dummy,QmuPartEnum);

	/*broadcast variables: only cpu 0 has correct values*/
	MPI_Bcast(&numvariables,1,MPI_INT,0,MPI_COMM_WORLD); 
	if(my_rank!=0)variables=(double*)xmalloc(numvariables*sizeof(double));
	MPI_Bcast(variables,numvariables,MPI_DOUBLE,0,MPI_COMM_WORLD); 

	/*broadcast variables_descriptors: */
	if(my_rank!=0){
		variables_descriptors=(char**)xmalloc(numvariables*sizeof(char*));
	}
	for(i=0;i<numvariables;i++){
		if(my_rank==0){
			string=variables_descriptors[i];
			string_length=(strlen(string)+1)*sizeof(char);
		}
		MPI_Bcast(&string_length,1,MPI_INT,0,MPI_COMM_WORLD); 
		if(my_rank!=0)string=(char*)xmalloc(string_length);
		MPI_Bcast(string,string_length,MPI_CHAR,0,MPI_COMM_WORLD); 
		if(my_rank!=0)variables_descriptors[i]=string;
	}

	/*broadcast numresponses: */
	MPI_Bcast(&numresponses,1,MPI_INT,0,MPI_COMM_WORLD); 

	_printf_("qmu iteration: %i\n",counter);

	//initialize results:
	results=new DataSet(ResultsEnum);

	/*Modify core inputs in objects contained in model, to reflect the dakota variables inputs: */
	model->UpdateFromDakota(variables,variables_descriptors,numvariables,model->GetFormulation(DiagnosticAnalysisEnum,HorizAnalysisEnum)->parameters,qmu_part,qmu_npart); //diagnostic horiz model is the one holding the parameters for Dakota.

	/*Run the analysis core solution sequence: */
	if(analysis_type==DiagnosticAnalysisEnum){
			
		if(verbose)_printf_("Starting diagnostic core\n");

		diagnostic_core(results,model);

	}
	else if(analysis_type==ThermalAnalysisEnum){
		
		if(verbose)_printf_("Starting thermal core\n");
		thermal_core(results,model);

	}
	else if(analysis_type==PrognosticAnalysisEnum){

		if(verbose)_printf_("Starting prognostic core\n");
		prognostic_core(results,model);

	}
	else if(analysis_type==TransientAnalysisEnum){

		if(verbose)_printf_("Starting transient core\n");
		transient_core(results,model);

	}
	else ISSMERROR("%s%i%s%i%s"," analysis_type ",analysis_type," and sub_analysis_type ",sub_analysis_type," not supported yet!");
	
		

	/*Now process the outputs, before computing the dakota responses: */
	if(verbose)_printf_("process results:\n");

	ProcessResults(&processed_results,results,model,analysis_type); 

	/*compute responses on cpu 0: dummy for now! */
	if(verbose)_printf_("compute dakota responses:\n");
	DakotaResponses(responses,responses_descriptors,numresponses,model,results,processed_results,analysis_type,sub_analysis_type);

	/*Free ressources:*/
	delete results;
	delete processed_results;

	//variables only on cpu != 0
	if(my_rank!=0){
		xfree((void**)&variables);
		for(i=0;i<numvariables;i++){
			string=variables_descriptors[i];
			xfree((void**)&string);
		}
		xfree((void**)&variables_descriptors);
	}
	//responses descriptors
	for(i=0;i<numresponses;i++){
		string=responses_descriptors[i];
		xfree((void**)&string);
	}
	//rest of dynamic allocations.
	xfree((void**)&responses_descriptors);
	xfree((void**)&qmu_part);
}

