/*
 * \file DataSet.c
 * \brief: implementation of the DataSet class, and derived classes Inputs and Parameters
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Constructors/Destructors*/
/*FUNCTION DataSet::DataSet(){{{1*/
DataSet::DataSet(){
	
	sorted=0;
	sorted_ids=NULL;
	id_offsets=NULL;

}
/*}}}*/
/*FUNCTION DataSet::DataSet(int dataset_enum){{{1*/
DataSet::DataSet(int dataset_enum){
	enum_type=dataset_enum;
	
	sorted=0;
	sorted_ids=NULL;
	id_offsets=NULL;

}
/*}}}*/
/*FUNCTION DataSet::Copy{{{1*/
DataSet*   DataSet::Copy(void){


	DataSet* copy=NULL;
	vector<Object*>::iterator object;
	Object* object_copy=NULL;

	copy=new DataSet(enum_type);

	copy->sorted=sorted;
	copy->presorted=presorted;
	if(sorted_ids){
		copy->sorted_ids=(int*)xmalloc(objects.size()*sizeof(int));
		memcpy(copy->sorted_ids,sorted_ids,objects.size()*sizeof(int));
	}
	if(id_offsets){
		copy->id_offsets=(int*)xmalloc(objects.size()*sizeof(int));
		memcpy(copy->id_offsets,id_offsets,objects.size()*sizeof(int));
	}

	/*Now we need to deep copy the objects: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		object_copy = (*object)->copy();
		copy->AddObject(object_copy);
	}
	return copy;
}
/*}}}*/
/*FUNCTION DataSet::~DataSet{{{1*/
DataSet::~DataSet(){
	clear();
	xfree((void**)&sorted_ids);
	xfree((void**)&id_offsets);
}
/*}}}*/

/*I/O*/
/*FUNCTION DataSet::Marshall{{{1*/
char* DataSet::Marshall(){

	vector<Object*>::iterator object;
	int                       object_size;
	int                       marshalled_dataset_size=0;
	char*                     marshalled_dataset=NULL;
	char*                     old_marshalled_dataset=NULL;

	/*First get size of marshalled dataset: */
	object_size=(int)objects.size();

	marshalled_dataset_size=MarshallSize();
	
	/*Allocate marshalled dataset: */
	marshalled_dataset=(char*)xmalloc(marshalled_dataset_size*sizeof(char)); 

	/*Keep track of old_marshalled_dataset: */
	old_marshalled_dataset=marshalled_dataset;

	/*Store internals of dataset first: */
	memcpy(marshalled_dataset,&object_size,sizeof(int)); marshalled_dataset+=sizeof(int);
	memcpy(marshalled_dataset,&sorted,sizeof(int)); marshalled_dataset+=sizeof(int);
	if(sorted){
		if(object_size)memcpy(marshalled_dataset,sorted_ids,object_size*sizeof(int)); marshalled_dataset+=object_size*sizeof(int);
		if(object_size)memcpy(marshalled_dataset,id_offsets,object_size*sizeof(int)); marshalled_dataset+=object_size*sizeof(int);
	}

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		(*object)->Marshall(&marshalled_dataset);
	}

	/* Ok, marshalled_dataset now points to the end of the original marshalled_dataset pointer 
	 * before  we started the loop on objects. Get object to point right again: */
	marshalled_dataset-=marshalled_dataset_size;

	/*We should be back to old_marshalled_dataset: check and abort if that's not the case, 
	 * because this is a nasty error: */
	if (marshalled_dataset!=old_marshalled_dataset){
		ISSMERROR("final marshalled dataset is different from initial one!"); 
		abort();
	}

	/*Return: */
	return marshalled_dataset;
}
/*}}}*/
/*FUNCTION DataSet::MarshallSize{{{1*/
int DataSet::MarshallSize(){

	vector<Object*>::iterator object;
	int                      marshalled_dataset_size=0;


	for ( object=objects.begin() ; object < objects.end(); object++ ){
		marshalled_dataset_size+= (*object)->MarshallSize();
	}

	marshalled_dataset_size+=sizeof(int); //objects size
	marshalled_dataset_size+=sizeof(int); //sorted size
	if(sorted){
		marshalled_dataset_size+=(int)objects.size()*sizeof(int); //sorted ids
		marshalled_dataset_size+=(int)objects.size()*sizeof(int); //id offsets
	}

	return marshalled_dataset_size;
}
/*}}}*/
/*FUNCTION DataSet::Demarshall{{{1*/
DataSet* DataSetDemarshall(char* marshalled_dataset){

	return DataSetDemarshallRaw(&marshalled_dataset);

}
/*}}}*/
/*FUNCTION DataSet::DemarshallRaw{{{1*/
DataSet* DataSetDemarshallRaw(char** pmarshalled_dataset){

	int i;

	DataSet* dataset=NULL;
	int      numobjects=0;
	int      enum_type;
	Object*  object=NULL;
	int      sorted;
	int*     sorted_ids=NULL;
	int*     id_offsets=NULL;
	char*    marshalled_dataset=NULL;

	/*recover marshalled_dataset pointer: */
	marshalled_dataset=*pmarshalled_dataset;

	/*initialize dataset: */
	dataset=new DataSet();

	/*Get internals first: */
	memcpy(&numobjects,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);
	memcpy(&sorted,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);
	if(sorted){
		if(numobjects){
			sorted_ids=(int*)xmalloc(numobjects*sizeof(int));
			id_offsets=(int*)xmalloc(numobjects*sizeof(int));
			memcpy(sorted_ids,marshalled_dataset,numobjects*sizeof(int)); marshalled_dataset+=numobjects*sizeof(int);
			memcpy(id_offsets,marshalled_dataset,numobjects*sizeof(int)); marshalled_dataset+=numobjects*sizeof(int);
		}
		dataset->SetSorting(sorted_ids,id_offsets);
	}

	for(i=0;i<numobjects;i++){

		/*get enum type of object: */
		memcpy(&enum_type,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);

		if(enum_type==NodeEnum){
			Node* node=NULL;
			node=new Node();
			node->Demarshall(&marshalled_dataset);
			dataset->AddObject(node);
		}
		else if(enum_type==VertexEnum){
			Vertex* vertex=NULL;
			vertex=new Vertex();
			vertex->Demarshall(&marshalled_dataset);
			dataset->AddObject(vertex);
		}
		else if(enum_type==DoubleParamEnum){
			DoubleParam* doubleparam=NULL;
			doubleparam=new DoubleParam();
			doubleparam->Demarshall(&marshalled_dataset);
			dataset->AddObject(doubleparam);
		}
		else if(enum_type==TriaEnum){
			Tria* tria=NULL;
			tria=new Tria();
			tria->Demarshall(&marshalled_dataset);
			dataset->AddObject(tria);
		}
		else if(enum_type==TriaVertexInputEnum){
			TriaVertexInput* triavertexinput=NULL;
			triavertexinput=new TriaVertexInput();
			triavertexinput->Demarshall(&marshalled_dataset);
			dataset->AddObject(triavertexinput);
		}
		else if(enum_type==SingEnum){
			Sing* sing=NULL;
			sing=new Sing();
			sing->Demarshall(&marshalled_dataset);
			dataset->AddObject(sing);
		}
		else if(enum_type==BeamEnum){
			Beam* beam=NULL;
			beam=new Beam();
			beam->Demarshall(&marshalled_dataset);
			dataset->AddObject(beam);
		}
		else if(enum_type==PentaEnum){
			Penta* penta=NULL;
			penta=new Penta();
			penta->Demarshall(&marshalled_dataset);
			dataset->AddObject(penta);
		}
		else if(enum_type==MaticeEnum){
			Matice* matice=NULL;
			matice=new Matice();
			matice->Demarshall(&marshalled_dataset);
			dataset->AddObject(matice);
		}
		else if(enum_type==MatparEnum){
			Matpar* matpar=NULL;
			matpar=new Matpar();
			matpar->Demarshall(&marshalled_dataset);
			dataset->AddObject(matpar);
		}
		else if(enum_type==SpcEnum){
			Spc* spc=NULL;
			spc=new Spc();
			spc->Demarshall(&marshalled_dataset);
			dataset->AddObject(spc);
		}
		else if(enum_type==PengridEnum){
			Pengrid* pengrid=NULL;
			pengrid=new Pengrid();
			pengrid->Demarshall(&marshalled_dataset);
			dataset->AddObject(pengrid);
		}
		else if(enum_type==PenpairEnum){
			Penpair* penpair=NULL;
			penpair=new Penpair();
			penpair->Demarshall(&marshalled_dataset);
			dataset->AddObject(penpair);
		}
		else if(enum_type==IcefrontEnum){
			Icefront* icefront=NULL;
			icefront=new Icefront();
			icefront->Demarshall(&marshalled_dataset);
			dataset->AddObject(icefront);
		}
		else if(enum_type==NumericalfluxEnum){
			Numericalflux* numericalflux=NULL;
			numericalflux=new Numericalflux();
			numericalflux->Demarshall(&marshalled_dataset);
			dataset->AddObject(numericalflux);
		}
		else if(enum_type==RgbEnum){
			Rgb* rgb=NULL;
			rgb=new Rgb();
			rgb->Demarshall(&marshalled_dataset);
			dataset->AddObject(rgb);
		}
		else if(enum_type==RiftfrontEnum){
			Riftfront* riftfront=NULL;
			riftfront=new Riftfront();
			riftfront->Demarshall(&marshalled_dataset);
			dataset->AddObject(riftfront);
		}
		else{
			ISSMERROR("%s%i"," could not recognize enum type: ",enum_type);
		}

	}

	/*Assign output pointers:*/
	*pmarshalled_dataset=marshalled_dataset;
	
	return dataset;
}
/*}}}*/
/*FUNCTION DataSet::Spawn{{{1*/
DataSet* DataSet::Spawn(int* indices, int num){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Specific methods*/
/*FUNCTION DataSet::AddObject{{{1*/
int  DataSet::AddObject(Object* object){

	objects.push_back(object);

	return 1;
}
/*}}}*/
/*FUNCTION DataSet::clear{{{1*/
void  DataSet::clear(){

	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		delete (*object);
	}
	objects.clear();
}
/*}}}*/
/*FUNCTION DataSet::DeleteObject{{{1*/
int  DataSet::DeleteObject(Object* object){

	vector<Object*>::iterator iterator;

	if(object){
		iterator = find(objects.begin(), objects.end(),object);
		delete *iterator;
		objects.erase(iterator);
	}

}
/*}}}*/
/*FUNCTION DataSet::DeepEcho{{{1*/
void DataSet::DeepEcho(){


	vector<Object*>::iterator object;

	if(this==NULL)ISSMERROR(" trying to echo a NULL dataset");

	_printf_("DataSet echo: %i objects\n",objects.size());

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call deep echo on object: */
		(*object)->DeepEcho();

	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::Echo{{{1*/
void DataSet::Echo(){


	vector<Object*>::iterator object;

	if(this==NULL)ISSMERROR(" trying to echo a NULL dataset");

	_printf_("DataSet echo: %i objects\n",objects.size());

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		(*object)->Echo();

	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::FindResult(Vec* presult,char* name){{{1*/
int   DataSet::FindResult(Vec* presult,char* name){

	/*Go through a dataset, and find a Result* object 
	 *whith result name is "name" : */
	
	vector<Object*>::iterator object;
	Result* result=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ResultEnum){

			/*Ok, this object is a result,recover it and ask which name it has: */
			result=(Result*)(*object);

			if (strcmp(result->GetFieldName(),name)==0){
				/*Ok, this is the one! Recover the value of this result: */
				result->GetField(presult);
				found=1;
				break;
			}
		}
	}
	return found;
}
/*}}}*/
/*FUNCTION DataSet::FindResult(void* pvalue, char* name){{{1*/
int   DataSet::FindResult(void* pvalue, char* name){

	/*Go through a dataset, and find a Result* object 
	 *which field name is "name" : */
	
	vector<Object*>::iterator object;
	Result* result=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ResultEnum){

			/*Ok, this object is a result, recover it and ask which name it has: */
			result=(Result*)(*object);

			if (strcmp(result->GetFieldName(),name)==0){
				/*Ok, this is the one! Recover the value of this result: */
				double** field=NULL;
				field=(double**)pvalue; //not very safe, but hey!
				result->GetField(field);
				found=1;
				break;
			}
		}
	}
	return found;
}
/*}}}*/
/*FUNCTION DataSet::GetEnum{{{1*/
int  DataSet::GetEnum(){
	return enum_type;
}
/*}}}*/
/*FUNCTION DataSet::GetEnum(int offset){{{1*/
int   DataSet::GetEnum(int offset){

	return objects[offset]->Enum();

}
/*}}}*/
/*FUNCTION DataSet::GetObjectByOffset{{{1*/
Object* DataSet::GetObjectByOffset(int offset){

	return objects[offset];

}
/*}}}*/
/*FUNCTION DataSet::GetObjectById{{{1*/
Object* DataSet::GetObjectById(int* poffset,int eid){

	int id_offset;
	int offset;
	int i;

	if(!sorted)ISSMERROR(" trying to binary search on a non-sorted dataset!");

	/*Carry out a binary search on the sorted_ids: */
	if(!binary_search(&id_offset,eid, sorted_ids,objects.size())){
		ISSMERROR("%s%i"," could not find object with id ",eid);
	}

	/*Convert  the id offset into sorted offset: */
	offset=id_offsets[id_offset];

	/*Assign output pointers if requested:*/
	if (poffset)*poffset=offset;

	/*Return object at offset position in objects :*/
	return objects[offset];
}
/*}}}*/
/*FUNCTION DataSet::Ranks{{{1*/
void   DataSet::Ranks(int* ranks){

	/*Go through a dataset, and for each object, report it cpu: */

	int rank;
	int id;
	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Ok, this object is a node, ask which rank it has: */
		rank=(*object)->MyRank();

		/*Which id does it have: */
		id=(*object)->Id();

		/*Plug rank into ranks, according to id: */
		ranks[id-1]=rank; //Matlab ids start at 1.
	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::Presort{{{1*/
void DataSet::Presort(){

	/*vector of objects is already sorted, just allocate the sorted ids and their
	 * offsets:*/
	int i;

	if(objects.size()){
		sorted_ids=(int*)xmalloc(objects.size()*sizeof(int));
		id_offsets=(int*)xmalloc(objects.size()*sizeof(int));
		for(i=0;i<objects.size();i++){
			id_offsets[i]=i;
			sorted_ids[i]=objects[i]->Id();
		}
	}

	/*set sorted flag: */
	sorted=1;
}
/*}}}*/
/*FUNCTION DataSet::SetSorting{{{1*/
void DataSet::SetSorting(int* in_sorted_ids,int* in_id_offsets){

	sorted=1;
	sorted_ids=in_sorted_ids;
	id_offsets=in_id_offsets;
}
/*}}}*/
/*FUNCTION DataSet::Size{{{1*/
int  DataSet::Size(void){

	return objects.size();
}
/*}}}*/
/*FUNCTION DataSet::Sort{{{1*/
void DataSet::Sort(){

	/*Only sort if we are not already sorted: */
	if(!sorted){
		ISSMERROR(" not implemented yet!");
	}
}
/*}}}*/

/*Objects methods*/
/*FUNCTION DataSet::ComputeBasalStress{{{1*/
void DataSet::ComputeBasalStress(Vec sigma_b,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->ComputeBasalStress(sigma_b,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::ComputePressure{{{1*/
void DataSet::ComputePressure(Vec p_g,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->ComputePressure(p_g,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::ComputeStrainRate{{{1*/
void DataSet::ComputeStrainRate(Vec eps,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->ComputeStrainRate(eps,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::Configure{{{1*/
void DataSet::Configure(DataSet* elements,DataSet* loads, DataSet* nodes, DataSet* vertices, DataSet* materials,Parameters* parameters){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Configure(loads,nodes,materials,parameters);
		}
		if(EnumIsLoad((*object)->Enum())){
			load=(Load*)(*object);
			load->Configure(elements,loads,nodes,vertices,materials,parameters);
		}

		if((*object)->Enum()==NodeEnum){
			node=(Node*)(*object);
			node->Configure(nodes,vertices);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::CostFunction{{{1*/
void  DataSet::CostFunction(double* pJ,int analysis_type,int sub_analysis_type){

	double J=0;;

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			J+=element->CostFunction(analysis_type,sub_analysis_type);

		}
	}

	/*Assign output pointers:*/
	*pJ=J;

}
/*}}}*/
/*FUNCTION DataSet::CreateKMatrix{{{1*/
void  DataSet::CreateKMatrix(Mat Kgg,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		
		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->CreateKMatrix(Kgg,analysis_type,sub_analysis_type);
		}
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->CreateKMatrix(Kgg,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::CreatePartitioningVector{{{1*/
void  DataSet::CreatePartitioningVector(Vec* ppartition,int numberofobjects){

	/*output: */
	Vec partition=NULL;
	vector<Object*>::iterator object;
	DofObject* dofobject=NULL;

	/*Create partition vector: */
	partition=NewVec(numberofobjects);

	/*Go through all objects, and ask each object to plug its doflist in 
	 * partition: */

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		dofobject=dynamic_cast<DofObject*>(*object);
		dofobject->CreatePartition(partition);
	}

	/*Assemble the petsc vector: */
	VecAssemblyBegin(partition);
	VecAssemblyEnd(partition);

	/*Assign output pointers: */
	*ppartition=partition;
}
/*}}}*/
/*FUNCTION DataSet::CreatePVector{{{1*/
void  DataSet::CreatePVector(Vec pg,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->CreatePVector(pg,analysis_type,sub_analysis_type);
		}
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->CreatePVector(pg,analysis_type,sub_analysis_type);
		}		
	}

}
/*}}}*/
/*FUNCTION DataSet::DistributeDofs{{{1*/
void  DataSet::DistributeDofs(int numberofobjects,int numberofdofsperobject){

	extern int num_procs;
	extern int my_rank;

	int  i;
	
	int  dofcount=0;
	int* alldofcount=NULL;
	int* truedofs=NULL;
	int* alltruedofs=NULL;
	vector<Object*>::iterator object;
	DofObject* dofobject=NULL;

	/*Go through objects, and distribute dofs locally, from 0 to numberofdofsperobject: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		dofobject=dynamic_cast<DofObject*>(*object);
		dofobject->DistributeDofs(&dofcount);

	}

	/*Ok, now every object has distributed dofs, but locally, and with a dof count starting from 
	 *0. This means the dofs between all the cpus are not synchronized! We need to synchronize all 
	 *dof on all cpus, and use those to update the dofs of every object: */

	alldofcount=(int*)xmalloc(num_procs*sizeof(int));
	MPI_Gather(&dofcount,1,MPI_INT,alldofcount,1,MPI_INT,0,MPI_COMM_WORLD);
	MPI_Bcast(alldofcount,num_procs,MPI_INT,0,MPI_COMM_WORLD);

	/*Ok, now every cpu should start its own dof count at the end of the dofcount 
	 * from cpu-1. : */
	dofcount=0;
	if(my_rank==0){
		dofcount=0;
	}
	else{
		for(i=0;i<my_rank;i++){
			dofcount+=alldofcount[i];
		}
	}


	/*Ok, now every cpu knows where his dofs should start. Update the dof count: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		dofobject=dynamic_cast<DofObject*>(*object);
		dofobject->OffsetDofs(dofcount);
	
	}

	/*Finally, remember that cpus may have skipped some objects, because they were clones. For every 
	 * object that is not a clone, tell them to show their dofs, so that later on, they can get picked 
	 * up by their clones: */
	truedofs=(int*)xcalloc(numberofobjects*numberofdofsperobject,sizeof(int));
	alltruedofs=(int*)xcalloc(numberofobjects*numberofdofsperobject,sizeof(int));

	for ( object=objects.begin() ; object < objects.end(); object++ ){
			
		/*Ok, let this object show its true dofs, if is is a true dof: */
		dofobject=dynamic_cast<DofObject*>(*object);
		dofobject->ShowTrueDofs(truedofs);
	}
	
	MPI_Allreduce ( (void*)truedofs,(void*)alltruedofs,numberofobjects*numberofdofsperobject,MPI_INT,MPI_MAX,MPI_COMM_WORLD);

	/*Ok, now every cpu knows the true dofs of everyone else that is not a clone. Let the clones recover those true dofs: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		dofobject=dynamic_cast<DofObject*>(*object);
		dofobject->UpdateCloneDofs(alltruedofs);

	}

	/* Free ressources: */
	xfree((void**)&alldofcount);
	xfree((void**)&truedofs);
	xfree((void**)&alltruedofs);

}
/*}}}*/
/*FUNCTION DataSet::Du{{{1*/
void  DataSet::Du(Vec du_g,int analysis_type,int sub_analysis_type){


	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Du(du_g,analysis_type,sub_analysis_type);
		}
	}


}
/*}}}*/
/*FUNCTION DataSet::FieldAverageOntoVertices{{{1*/
void DataSet::FieldAverageOntoVertices(Vec fieldsum,Vec connectivity,double* field){

	vector<Object*>::iterator object;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==NodeEnum){
			node=(Node*)(*object);
			node->FieldAverageOntoVertices(fieldsum,connectivity,field);
		}

	}

}
/*}}}*/
/*FUNCTION DataSet::FieldDepthAverageAtBase{{{1*/
void  DataSet::FieldDepthAverageAtBase(Vec field,double* field_serial,char* fieldname){

	vector<Object*>::iterator object;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==NodeEnum){

			node=(Node*)(*object);
			node->FieldDepthAverageAtBase(field,field_serial,fieldname);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::FieldExtrude{{{1*/
void  DataSet::FieldExtrude(Vec field,double* field_serial,char* field_name, int collapse){

	vector<Object*>::iterator object;
	Penta* penta=NULL;
	Node*  node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==PentaEnum){

			penta=(Penta*)(*object);
			penta->FieldExtrude(field,field_serial,field_name,collapse);

		}
		if((*object)->Enum()==NodeEnum){

			node=(Node*)(*object);
			node->FieldExtrude(field,field_serial,field_name);

		}
	}

}
/*}}}*/
/*FUNCTION DataSet::FlagClones{{{1*/
void  DataSet::FlagClones(int numberofobjects){

	int i;
	extern int num_procs;

	int* ranks=NULL;
	int* minranks=NULL;

	vector<Object*>::iterator object;
	DofObject* dofobject=NULL;

	/*Allocate ranks: */
	ranks=(int*)xmalloc(numberofobjects*sizeof(int));
	minranks=(int*)xmalloc(numberofobjects*sizeof(int));

	for(i=0;i<numberofobjects;i++)ranks[i]=num_procs; //no cpu can have rank num_procs. This is the maximum limit.

	/*Now go through all our objects and ask them to report to who they belong (which rank): */
	Ranks(ranks);

	/*We need to take the minimum rank for each vertex, and every cpu needs to get that result. That way, 
	 * when we start building the dof list for all vertexs, a cpu can check whether its vertex already has been 
	 * dealt with by another cpu. We take the minimum because we are going to manage dof assignment in increasing 
	 * order of cpu rank. This is also why we initialized this array to num_procs.*/
	MPI_Allreduce ( (void*)ranks,(void*)minranks,numberofobjects,MPI_INT,MPI_MIN,MPI_COMM_WORLD);

	/*Now go through all objects, and use minranks to flag which objects are cloned: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*For this object, decide whether it is a clone: */
		dofobject=dynamic_cast<DofObject*>(*object);
		dofobject->SetClone(minranks);
	}

	/*Free ressources: */
	xfree((void**)&ranks); 
	xfree((void**)&minranks);

}
/*}}}*/
/*FUNCTION DataSet::FlagNodeSets{{{1*/
void DataSet::FlagNodeSets(Vec pv_g, Vec pv_m, Vec pv_n, Vec pv_f, Vec pv_s){

	vector<Object*>::iterator object;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==NodeEnum){

			node=(Node*)(*object);

			/*Plug set values intp our 4 set vectors: */
			node->CreateVecSets(pv_g,pv_m,pv_n,pv_f,pv_s);

		}
	}

	/*Assemble: */
	VecAssemblyBegin(pv_g);
	VecAssemblyEnd(pv_g);

	VecAssemblyBegin(pv_m);
	VecAssemblyEnd(pv_m);

	VecAssemblyBegin(pv_n);
	VecAssemblyEnd(pv_n);

	VecAssemblyBegin(pv_f);
	VecAssemblyEnd(pv_f);

	VecAssemblyBegin(pv_s);
	VecAssemblyEnd(pv_s);

}
/*}}}*/
/*FUNCTION DataSet::Gradj{{{1*/
void  DataSet::Gradj(Vec grad_g,int analysis_type,int sub_analysis_type,char* control_type){


	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Gradj(grad_g,analysis_type,sub_analysis_type,control_type);
		}
	}


}		
/*}}}*/
/*FUNCTION DataSet::MeltingIsPresent{{{1*/
int   DataSet::MeltingIsPresent(){

	int found=0;
	int mpi_found=0;

	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==PengridEnum){
			found=1;
			break;
		}
	}	
	
	#ifdef _PARALLEL_
	MPI_Reduce (&found,&mpi_found,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&mpi_found,1,MPI_INT,0,MPI_COMM_WORLD);                
	found=mpi_found;
	#endif

	return found;
}
/*}}}*/
/*FUNCTION DataSet::MeltingConstraints{{{1*/
void  DataSet::MeltingConstraints(int* pconverged, int* pnum_unstable_constraints,int analysis_type,int sub_analysis_type){

	/* generic object pointer: */
	vector<Object*>::iterator object;
	Pengrid* pengrid=NULL;

	int unstable=0;
	int num_unstable_constraints=0;
	int converged=0;
	int sum_num_unstable_constraints=0;

	num_unstable_constraints=0;	

	/*Enforce constraints: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		
		if((*object)->Enum()==PengridEnum){

			pengrid=(Pengrid*)(*object);

			pengrid->PenaltyConstrain(&unstable,analysis_type,sub_analysis_type);

			num_unstable_constraints+=unstable;
		}
	}

	#ifdef _PARALLEL_
	MPI_Reduce (&num_unstable_constraints,&sum_num_unstable_constraints,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&sum_num_unstable_constraints,1,MPI_INT,0,MPI_COMM_WORLD);                
	num_unstable_constraints=sum_num_unstable_constraints;
	#endif

	/*Have we converged? : */
	if (num_unstable_constraints==0) converged=1;
	else converged=0;

	/*Assign output pointers: */
	*pconverged=converged;
	*pnum_unstable_constraints=num_unstable_constraints;
}
/*}}}*/
/*FUNCTION DataSet::Misfit{{{1*/
void  DataSet::Misfit(double* pJ,int analysis_type,int sub_analysis_type){

	double J=0;;

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			J+=element->Misfit(analysis_type,sub_analysis_type);

		}
	}

	/*Assign output pointers:*/
	*pJ=J;

}
/*}}}*/
/*FUNCTION DataSet::NumberOfDofs{{{1*/
int   DataSet::NumberOfDofs(){


	vector<Object*>::iterator object;
	Node* node=NULL;
	int   numdofs=0;
	int   allnumdofs;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to plug values into partition: */
			if (!node->IsClone()){

				numdofs+=node->GetNumberOfDofs();

			}
		}
	}

	/*Gather from all cpus: */
	MPI_Allreduce ( (void*)&numdofs,(void*)&allnumdofs,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);

	return allnumdofs;
}
/*}}}*/
/*FUNCTION DataSet::NumberOfRgbs{{{1*/
int   DataSet::NumberOfRgbs(){

	vector<Object*>::iterator object;
	Rgb* rgb=NULL;
	int  count=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==RgbEnum){ //we assume uniqueness of all Rgbs, no error checking here.

			count++;

		}
	}

	return count;
}
/*}}}*/
/*FUNCTION DataSet::OutputRifts{{{1*/
void  DataSet::OutputRifts(Vec riftproperties){

	vector<Object*>::iterator object;
	Riftfront* riftfront=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==RiftfrontEnum){

			riftfront=(Riftfront*)(*object);
			riftfront->OutputProperties(riftproperties);
		}
	}



}
/*}}}*/
/*FUNCTION DataSet::PenaltyCreateKMatrix{{{1*/
void  DataSet::PenaltyCreateKMatrix(Mat Kgg,double kmax,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->PenaltyCreateKMatrix(Kgg,kmax,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::PenaltyCreatePVector{{{1*/
void  DataSet::PenaltyCreatePVector(Vec pg,double kmax,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->PenaltyCreatePVector(pg,kmax,analysis_type,sub_analysis_type);
		}		
	}

}
/*}}}*/
/*FUNCTION DataSet::RiftIsPresent{{{1*/
int   DataSet::RiftIsPresent(){

	int i;

	vector<Object*>::iterator object;
	Penpair* penpair=NULL;
	int found=0;
	int mpi_found=0;

	/*go though loads, and figure out if one of the loads is a PenPair with numdof=2: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==PenpairEnum){

			penpair=(Penpair*)(*object);
		}
	}

#ifdef _PARALLEL_
	MPI_Reduce (&found,&mpi_found,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&mpi_found,1,MPI_INT,0,MPI_COMM_WORLD);                
	found=mpi_found;
#endif

	return found;
}
/*}}}*/
/*FUNCTION DataSet::SetupMpcs{{{1*/
void DataSet::SetupMpcs(Mat Rmg,DataSet* nodes){

	vector<Object*>::iterator object;
	Rgb* rgb=NULL;
	int  nodeid1;
	int  nodeid2;
	int  dof;

	int  dof1;
	int  dof2;


	Node* node1=NULL;
	Node* node2=NULL;

	int count=-1;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==RgbEnum){ //we assume uniqueness of all Rgbs, no error checking here.

			/*we found an rgb, increment counter, so that row index for Rmg is up to date: */
			count++;
			rgb=(Rgb*)(*object);


			nodeid1=rgb->GetNodeId1();
			nodeid2=rgb->GetNodeId2();
			dof=rgb->GetDof();

			/*For this rgb, find the nodes that go with it: */
			node1=(Node*)nodes->GetObjectById(NULL,nodeid1);
			node2=(Node*)nodes->GetObjectById(NULL,nodeid2);

			if ((node1 && !node2) || (!node1 && node2)){
				/*we are missing one node, not good!*/
				ISSMERROR("%s%p%s%p"," in Rgb, missing one node. node1: ",node1," node2: ",node2);
			}

			if(!node1 && !node2){
				/*That's ok, this Rgb can't find those nodes, so leave them alone. They are probably not on this 
				 * cpu!*/
			}
			else{
				/*Ok, this cpu owns both nodes. Put dof for node1 into m set, unless it is already there, 
				 * in which case node2 gets into the m set: */
				if(node1->DofIsInMSet(dof-1)){
					node2->DofInMSet(dof-1);
				}
				else{
					node1->DofInMSet(dof-1);
				}

				/*Plug values into Rmg. We essentially want dofs from node1 and node2 to be the 
				 *same: */
				dof1=node1->GetDof(dof-1); //matlab indexing
				dof2=node2->GetDof(dof-1); //matlab indexing

				MatSetValue(Rmg,count,dof1,1.0,INSERT_VALUES);
				MatSetValue(Rmg,count,dof2,-1.0,INSERT_VALUES);

			}
		}
	}
}
/*}}}*/
/*FUNCTION DataSet::SetupSpcs{{{1*/
void   DataSet::SetupSpcs(DataSet* nodes,Vec yg){

	vector<Object*>::iterator object;
	Spc* spc=NULL;
	Node* node=NULL;

	int nodeid;
	int dof;
	double value;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==SpcEnum){

			spc=(Spc*)(*object);

			/*Ok, this object is a constraint. Get the nodeid from the node it applies to: */
			nodeid=spc->GetNodeId();
			dof=spc->GetDof();
			value=spc->GetValue();

			/*Now, chase through nodes and find the corect node: */
			node=(Node*)nodes->GetObjectById(NULL,nodeid);

			/*Apply constraint: */
			if(node){ //in case the spc is dealing with a node on another cpu
				node->ApplyConstraint(yg,dof,value);
			}

		}
	}

	/*Assemble yg: */
	VecAssemblyBegin(yg);
	VecAssemblyEnd(yg);
}
/*}}}*/
/*FUNCTION DataSet::SurfaceArea{{{1*/
void  DataSet::SurfaceArea(double* pS,int analysis_type,int sub_analysis_type){

	double S=0;

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			S+=element->SurfaceArea(analysis_type,sub_analysis_type);

		}
	}

	/*Assign output pointers:*/
	*pS=S;

}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromSolution{{{1*/
void  DataSet::UpdateInputsFromSolution(double* solution, int analysis_type, int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->UpdateInputsFromSolution(solution,analysis_type,sub_analysis_type);
		}
		else ISSMERROR("%s%i%s"," object with id: ",(*object)->Id()," is not an element, in a function that deals only with elements!");
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromConstant{{{1*/
void  DataSet::UpdateInputsFromConstant(double constant, int name){

	vector<Object*>::iterator object;
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		(*object)->UpdateInputsFromConstant(constant,name);
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromConstant{{{1*/
void  DataSet::UpdateInputsFromConstant(int constant, int name){

	vector<Object*>::iterator object;
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		(*object)->UpdateInputsFromConstant(constant,name);
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromConstant{{{1*/
void  DataSet::UpdateInputsFromConstant(bool constant, int name){

	vector<Object*>::iterator object;
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		(*object)->UpdateInputsFromConstant(constant,name);
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromVector{{{1*/
void  DataSet::UpdateInputsFromVector(double* vec, int name, int type){

	vector<Object*>::iterator object;
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		(*object)->UpdateInputsFromVector(vec,name,type);
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromVector{{{1*/
void  DataSet::UpdateInputsFromVector(int* vec, int name, int type){

	vector<Object*>::iterator object;
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		(*object)->UpdateInputsFromVector(vec,name,type);
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateInputsFromVector{{{1*/
void  DataSet::UpdateInputsFromVector(bool* vec, int name, int type){

	vector<Object*>::iterator object;
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		(*object)->UpdateInputsFromVector(vec,name,type);
	}
}
/*}}}*/
/*FUNCTION DataSet::UpdateVertexPositions{{{1*/
void  DataSet::UpdateVertexPositions(double* thickness,double* bed){

	vector<Object*>::iterator object;
	Vertex* vertex=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==VertexEnum){

			vertex=(Vertex*)(*object);
			vertex->UpdatePosition(thickness,bed);
		}
	}
}
/*}}}*/
