/*!\file Dofx
 * \brief: establish degrees of freedom for all nodes
 */

#include "./Dofx.h"

#include "../shared/shared.h"
#include "../include/include.h"
#include "../toolkits/toolkits.h"
#include "../EnumDefinitions/EnumDefinitions.h"

int Dofx( DofVec** ppartition, DofVec** ptpartition, DataSet* elements,DataSet* nodes, DataSet* vertices, Parameters* parameters) {

	int noerr=1;
	int i;

	int  found=0;
	extern int num_procs;
	extern int my_rank;
	
	/*intermediary: */
	int  numberofnodes;
	int  numberofvertices;
	int  numberofdofspernode;

	/*output: */
	DofVec* partition=NULL;
	DofVec* tpartition=NULL;

	/*Check that vertices and nodes are not empty*/
	ISSMASSERT(nodes && vertices);

	/*Initialize dofvecs: */
	partition=new DofVec("partition");
	tpartition=new DofVec("tpartition");

	/*First, recover number of vertices and nodes from parameters: */
	found=parameters->FindParam(&numberofvertices,NumberOfVerticesEnum);
	if(!found)ISSMERROR(" could not find numberofvertices in parameters");
	
	found=parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	if(!found)ISSMERROR(" could not find numberofnodes in parameters");

	/*Recover number of dofs per node: */
	found=parameters->FindParam(&numberofdofspernode,NumberOfDofsPerNodeEnum);
	if(!found)ISSMERROR(" could not find numberofdofspernode in parameters");

	/*Ensure that only for each cpu, the partition border nodes only will be taken into account once 
	 * across the cluster. To do so, we flag all the clone nodes: */
	vertices->FlagClones(numberofvertices);
	nodes->FlagClones(numberofnodes);

	/*Go through all vertices and nodes, and build degree of freedom lists. Each vertex/node gets a fixed number of dofs. When 
	 *a  vertex/node has already been distributed dofs on one cpu, all other cpus with the same vertex/node cannot distribute it 
	 *anymore. Use clone field to be sure of that: */
	vertices->DistributeDofs(numberofvertices,1); //only 1 dof per vertex.
	nodes->DistributeDofs(numberofnodes,numberofdofspernode);

	/*Now that dofs have been distributed, create partitioning vector and its transpose: */
	vertices->CreatePartitioningVector(&partition->vector,numberofvertices);

	/*Transpose partition into tpartition: */
	VecTranspose(&tpartition->vector,partition->vector);

	/*Now, setup numdof and numentries for dofvec: */
	partition->numdofs=1;
	VecGetSize(partition->vector,&partition->numentries);

	tpartition->numdofs=1;
	VecGetSize(tpartition->vector,&tpartition->numentries);

	/*Assign output pointers: */
	*ppartition=partition;
	*ptpartition=tpartition;
	
	return noerr;
}
