function results=diagnostic_core(model);
%DIAGNOSTIC_CORE - compute the core velocity field 
%
%   Usage:
%      results=diagnostic_core(models);
%

%recover models
m_dh=models.dh;
m_dv=models.dv;
m_ds=models.ds;
m_dhu=models.dhu;
m_sl=models.sl;

%recover parameters common to all solutions
verbose=m_dhu.parameters.verbose;
dim=m_dh.parameters.dim;
ishutter=m_dhu.parameters.ishutter;
ismacayealpattyn=m_dh.parameters.ismacayealpattyn;
isstokes=m_ds.parameters.isstokes;
numrifts=m_dhu.parameters.numrifts;
qmu_analysis=m_dh.parameters.qmu_analysis;

	
%for qmu analysis, be sure the velocity input we are starting from  is the one in the parameters: 
if qmu_analysis,
	ModelUpdateInputsFromVector(models,m_dh.vx,VxEnum,VertexEnum);
	ModelUpdateInputsFromVector(models,m_dh.vy,VyEnum,VertexEnum);
	ModelUpdateInputsFromVector(models,m_dh.vz,VzEnum,VertexEnum);
end

%Compute slopes: 
[surfaceslopex,surfaceslopey]=slope_core(m_sl,SurfaceAnalysisEnum);
[bedslopex,bedslopey]=slope_core(m_sl,BedAnalysisEnum);

%Update:
ModelUpdateInputsFromVector(models,surfaceslopex,SurfaceSlopexEnum,VertexEnum);
ModelUpdateInputsFromVector(models,surfaceslopey,SurfaceSlopeyEnum,VertexEnum);
ModelUpdateInputsFromVector(models,bedslopex,BedSlopexEnum,VertexEnum);
ModelUpdateInputsFromVector(models,bedslopey,BedSlopeyEnum,VertexEnum);

if ishutter,

	displaystring(verbose,'\n%s',['computing hutter velocities...']);
	u_g=diagnostic_core_linear(m_dhu,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

	displaystring(verbose,'\n%s',['computing pressure according to MacAyeal...']);
	p_g=ComputePressure(m_dhu.elements,m_dhu.nodes,m_dhu.vertices,m_dhu.loads,m_dhu.materials,m_dhu.parameters,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

	displaystring(verbose,'\n%s',['update boundary conditions for macyeal pattyn using hutter results...']);
	if ismacayealpattyn,
		m_dh.y_g=u_g;
		[m_dh.ys m_dh.ys0]=Reducevectorgtos(m_dh.y_g,m_dh.nodesets);
	end

end
		
if ismacayealpattyn,

	displaystring(verbose,'\n%s',['computing horizontal velocities...']);
	[u_g m_dh.loads]=diagnostic_core_nonlinear(m_dh,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

	if dim==2,
		displaystring(verbose,'\n%s',['computing pressure according to MacAyeal...']);
		p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.vertices,m_dh.loads,m_dh.materials,m_dh.parameters,DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	end
end
	
if dim==3,

	displaystring(verbose,'\n%s',['extruding horizontal velocities...']);
	u_g_horiz=FieldExtrude(m_dh.elements,m_dh.nodes,m_dh.vertices,m_dh.loads,m_dh.materials,m_dh.parameters,u_g,'velocity',1);

	[vx,vy]=SplitSolutionVector(ug,numberofnodes,numberofdofspernode_dh);
	model->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
	model->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
		


	displaystring(verbose,'\n%s',['computing vertical velocities...']);
	inputs=add(inputs,'velocity',u_g_horiz,'doublevec',m_dh.parameters.numberofdofspernode,m_dh.parameters.numberofnodes);
	u_g_vert=diagnostic_core_linear(m_dv,inputs,DiagnosticAnalysisEnum(),VertAnalysisEnum());

	displaystring(verbose,'\n%s',['combining horizontal and vertical velocities...']);
	u_g=zeros(m_dh.parameters.numberofnodes*3,1);
	u_g(dofsetgen([1,2],3,m_dh.parameters.numberofnodes*3))=u_g_horiz;
	u_g(dofsetgen([3],3,m_dh.parameters.numberofnodes*3))=u_g_vert;

	displaystring(verbose,'\n%s',['computing pressure according to Pattyn...']);
	p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.vertices,m_dh.loads,m_dh.materials,m_dh.parameters,inputs,DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	
	if isstokes,

		%"recondition" pressure 
		p_g=p_g/m_ds.parameters.stokesreconditioning;

		displaystring(verbose,'\n%s',['computing bed slope (x and y derivatives)...']);
		slopex=diagnostic_core_linear(m_sl,inputs,SlopecomputeAnalysisEnum(),BedXAnalysisEnum());
		slopey=diagnostic_core_linear(m_sl,inputs,SlopecomputeAnalysisEnum(),BedYAnalysisEnum());
		slopex=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.vertices,m_sl.loads,m_sl.materials,m_sl.parameters,slopex,'slopex',0);
		slopey=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.vertices,m_sl.loads,m_sl.materials,m_sl.parameters,slopey,'slopey',0);

		inputs=add(inputs,'bedslopex',slopex,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);
		inputs=add(inputs,'bedslopey',slopey,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);
		
		%recombine u_g and p_g: 
		u_g_stokes=zeros(m_ds.nodesets.gsize,1);
		u_g_stokes(dofsetgen([1,2,3],4,m_ds.nodesets.gsize))=u_g;
		u_g_stokes(dofsetgen([4],4,m_ds.nodesets.gsize))=p_g;

		inputs=add(inputs,'velocity',u_g_stokes,'doublevec',4,m_ds.parameters.numberofnodes);

		displaystring(verbose,'\n%s',['update boundary conditions for stokes using velocities previously computed...']);
		m_ds.y_g=zeros(m_ds.nodesets.gsize,1);
		m_ds.y_g(dofsetgen([1,2,3],4,m_ds.nodesets.gsize))=u_g;
		[m_ds.ys m_ds.ys0]=Reducevectorgtos(m_ds.y_g,m_ds.nodesets);

		displaystring(verbose,'\n%s',['computing stokes velocities and pressure ...']);
		u_g=diagnostic_core_nonlinear(m_ds,inputs,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	
		%"decondition" pressure
		p_g=u_g(4:4:end)*m_dh.parameters.stokesreconditioning;
	end
end
%load onto results
results.step=1;
results.time=0;
results.u_g=u_g;
results.p_g=p_g;

if numrifts,
	results.riftproperties=OutputRifts(m_dh.loads,m_dh.parameters);
end
