/*!\file FemModel.c
 * \brief: implementation of the FemModel object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "./objects.h"

/*constructors/destructors*/
/*FUNCTION FemModel::FemModel {{{1*/
FemModel::FemModel(){

	elements=NULL;
	nodes=NULL;
	vertices=NULL;
	constraints=NULL;
	loads=NULL;
	materials=NULL;
	parameters=NULL;

	partition=NULL;
	tpartition=NULL;
	yg=NULL;
	Rmg=NULL;
	nodesets=NULL;
	ys=NULL;
	ys0=NULL;
	Gmn=NULL;

}
/*}}}*/
/*FUNCTION FemModel::FemModel {{{1*/
FemModel::FemModel(DataSet* femmodel_elements,DataSet* femmodel_nodes,DataSet* femmodel_vertices, DataSet* femmodel_constraints,DataSet* femmodel_loads,
		DataSet* femmodel_materials,Parameters* femmodel_parameters, DofVec* femmodel_partition,DofVec* femmodel_tpartition,DofVec* femmodel_yg,
		Mat femmodel_Rmg,Mat femmodel_Gmn,NodeSets* femmodel_nodesets,Vec femmodel_ys,Vec femmodel_ys0){


	elements=femmodel_elements;
	nodes=femmodel_nodes;
	vertices=femmodel_vertices;
	constraints=femmodel_constraints;
	loads=femmodel_loads;
	materials=femmodel_materials;
	parameters=femmodel_parameters;

	partition=femmodel_partition;
	tpartition=femmodel_tpartition;
	yg=femmodel_yg;
	Rmg=femmodel_Rmg;
	nodesets=femmodel_nodesets;
	ys=femmodel_ys;
	ys0=femmodel_ys0;
	Gmn=femmodel_Gmn;

}
/*}}}*/
/*FUNCTION FemModel::~FemModel {{{1*/
FemModel::~FemModel(){

	delete elements;
	delete nodes;
	delete vertices;
	delete loads;
	delete constraints;
	delete materials;
	delete parameters;

	delete partition;
	delete tpartition;
	delete yg;
	MatFree(&Rmg);
	delete nodesets;
	VecFree(&ys);
	VecFree(&ys0);
	MatFree(&Gmn);

}
/*}}}*/

/*object functions*/
/*FUNCTION FemModel::Echo {{{1*/
void FemModel::Echo(void){

	printf("FemModels echo: \n");
	printf("   elements: %p\n",elements);
	printf("   nodes: %p\n",nodes);
	printf("   vertices: %p\n",vertices);
	printf("   loads: %p\n",loads);
	printf("   materials: %p\n",materials);
	printf("   parameters: %p\n",parameters);
	
	printf("   partition: %p\n",partition);
	printf("   tpartition: %p\n",tpartition);
	printf("   yg: %p\n",yg);
	printf("   Rmg: %p\n",Rmg);
	printf("   nodesets: %p\n",nodesets);
	printf("   ys: %p\n",ys);
	printf("   ys0: %p\n",ys0);
	printf("   Gmn: %p\n",Gmn);

}
/*}}}*/
/*FUNCTION FemModel::DeepEcho {{{1*/
void FemModel::DeepEcho(void){
	
	printf("FemModels echo: \n");
	printf("   elements: \n");
	elements->Echo();
	printf("   nodes: \n");
	nodes->Echo();
	printf("   vertices: \n");
	vertices->Echo();
	printf("   loads: \n");
	nodes->Echo();
	printf("   materials: \n");
	nodes->Echo();
	printf("   parameters: \n");
	nodes->Echo();
	
	printf("   partition: \n");
	partition->Echo();
	printf("   tpartition: \n");
	tpartition->Echo();
	printf("   yg: \n");
	yg->Echo();
	printf("   Rmg: \n");
	MatView(Rmg,PETSC_VIEWER_STDOUT_WORLD);
	printf("   nodesets: \n");
	nodesets->Echo();
	printf("   ys: \n");
	VecView(ys,PETSC_VIEWER_STDOUT_WORLD);
	printf("   ys0: \n");
	VecView(ys0,PETSC_VIEWER_STDOUT_WORLD);
	printf("   Gmn: \n");
	MatView(Gmn,PETSC_VIEWER_STDOUT_WORLD);

}
/*}}}*/
/*FUNCTION FemModel::Id {{{1*/
int   FemModel::Id(void){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::MyRank {{{1*/
int   FemModel::MyRank(void){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::Marshall {{{1*/
void  FemModel::Marshall(char** pmarshalled_dataset){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::MarshallSize {{{1*/
int   FemModel::MarshallSize(void){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::Demarshall {{{1*/
void  FemModel::Demarshall(char** pmarshalled_dataset){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::Enum {{{1*/
int   FemModel::Enum(void){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::copy {{{1*/
Object* FemModel::copy(void){
	ISSMERROR("not implemented yet!");
}
/*}}}*/
/*FUNCTION FemModel::FindParam(bool* pinteger,int enum_type) {{{1*/
int FemModel::FindParam(bool* pinteger,int enum_type){
	
	return parameters->FindParam(pinteger,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(int* pinteger,int enum_type) {{{1*/
int FemModel::FindParam(int* pinteger,int enum_type){
	
	return parameters->FindParam(pinteger,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(double* pscalar,int enum_type) {{{1*/
int FemModel::FindParam(double* pscalar,int enum_type){
	
	return parameters->FindParam(pscalar,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(char** pstring,int enum_type) {{{1*/
int FemModel::FindParam(char** pstring,int enum_type){
	
	return parameters->FindParam(pstring,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(char*** pstringarray,int* pM,int enum_type) {{{1*/
int FemModel::FindParam(char*** pstringarray,int* pM,int enum_type){
	
	return parameters->FindParam(pstringarray,pM,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(double** pdoublearray,int* pM,int* pN,int enum_type) {{{1*/
int FemModel::FindParam(double** pdoublearray,int* pM,int* pN,int enum_type){
	
	return parameters->FindParam(pdoublearray,pM,pN,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(double** pdoublearray,int* pM,int enum_type) {{{1*/
int FemModel::FindParam(double** pdoublearray,int* pM,int enum_type){
	
	return parameters->FindParam(pdoublearray,pM,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(Vec* pvec,int enum_type) {{{1*/
int FemModel::FindParam(Vec* pvec,int enum_type){
	
	return parameters->FindParam(pvec,enum_type);

}
/*}}}*/
/*FUNCTION FemModel::FindParam(Mat* pmat,int enum_type) {{{1*/
int FemModel::FindParam(Mat* pmat,int enum_type){
	
	return parameters->FindParam(pmat,enum_type);

}
/*}}}*/

/*access to internal data: */
/*FUNCTION FemModel::get_elements {{{1*/
DataSet* FemModel::get_elements(void){
	return elements;
}
/*}}}*/
/*FUNCTION FemModel::get_nodes {{{1*/
DataSet* FemModel::get_nodes(void){
	return nodes;
}
/*}}}*/
/*FUNCTION FemModel::get_vertices {{{1*/
DataSet* FemModel::get_vertices(void){
	return vertices;
}
/*}}}*/
/*FUNCTION FemModel::get_constraints {{{1*/
DataSet* FemModel::get_constraints(void){
	return constraints;
}
/*}}}*/
/*FUNCTION FemModel::get_loads {{{1*/
DataSet* FemModel::get_loads(void){
	return loads;
}
/*}}}*/
/*FUNCTION FemModel::get_materials {{{1*/
DataSet* FemModel::get_materials(void){
	return materials;
}
/*}}}*/
/*FUNCTION FemModel::get_parameters {{{1*/
Parameters* FemModel::get_parameters(void){
	return parameters;
}
/*}}}*/
/*FUNCTION FemModel::get_partition {{{1*/
DofVec* FemModel::get_partition(void){
	return partition;
}
/*}}}*/
/*FUNCTION FemModel::get_tpartition {{{1*/
DofVec* FemModel::get_tpartition(void){
	return tpartition;
}
/*}}}*/
/*FUNCTION FemModel::get_yg {{{1*/
DofVec* FemModel::get_yg(void){
	return yg;
}
/*}}}*/
/*FUNCTION FemModel::get_Rmg {{{1*/
Mat FemModel::get_Rmg(void){
	return Rmg;
}
/*}}}*/
/*FUNCTION FemModel::get_nodesets {{{1*/
NodeSets* FemModel::get_nodesets(void){
	return nodesets;
}
/*}}}*/
/*FUNCTION FemModel::get_ys {{{1*/
Vec FemModel::get_ys(void){
	return ys;
}
/*}}}*/
/*FUNCTION FemModel::get_ys0 {{{1*/
Vec FemModel::get_ys0(void){
	return ys0;
}
/*}}}*/
/*FUNCTION FemModel::get_Gmn {{{1*/
Mat FemModel::get_Gmn(void){
	return Gmn;
}
/*}}}*/
/*FUNCTION FemModel::UpdateInputsFromVector(double* vector, int name, int type){{{1*/
void FemModel::UpdateInputsFromVector(double* vector, int name, int type){

	if(vector==NULL)return; //don't bother

	elements->UpdateInputsFromVector(vector,name,type);
	nodes->UpdateInputsFromVector(vector,name,type);
	vertices->UpdateInputsFromVector(vector,name,type);
	loads->UpdateInputsFromVector(vector,name,type);
	materials->UpdateInputsFromVector(vector,name,type);
	parameters->UpdateInputsFromVector(vector,name,type);

}
/*}}}*/
/*FUNCTION FemModel::UpdateInputsFromConstant(double constant, int name){{{1*/
void    FemModel::UpdateInputsFromConstant(double constant, int name){
	
	elements->UpdateInputsFromConstant(constant,name);
	nodes->UpdateInputsFromConstant(constant,name);
	vertices->UpdateInputsFromConstant(constant,name);
	loads->UpdateInputsFromConstant(constant,name);
	materials->UpdateInputsFromConstant(constant,name);
	parameters->UpdateInputsFromConstant(constant,name);

}
/*}}}*/
/*FUNCTION FemModel::UpdateInputsFromConstant(int constant, int name){{{1*/
void    FemModel::UpdateInputsFromConstant(int constant, int name){
	
	elements->UpdateInputsFromConstant(constant,name);
	nodes->UpdateInputsFromConstant(constant,name);
	vertices->UpdateInputsFromConstant(constant,name);
	loads->UpdateInputsFromConstant(constant,name);
	materials->UpdateInputsFromConstant(constant,name);
	parameters->UpdateInputsFromConstant(constant,name);

}
/*}}}*/
/*FUNCTION FemModel::UpdateInputsFromConstant(bool constant, int name){{{1*/
void    FemModel::UpdateInputsFromConstant(bool constant, int name){
	
	elements->UpdateInputsFromConstant(constant,name);
	nodes->UpdateInputsFromConstant(constant,name);
	vertices->UpdateInputsFromConstant(constant,name);
	loads->UpdateInputsFromConstant(constant,name);
	materials->UpdateInputsFromConstant(constant,name);
	parameters->UpdateInputsFromConstant(constant,name);

}
/*}}}*/
/*FUNCTION FemModel::UpdateInputsFromSolution(Vec vector, int name, int type){{{1*/
void FemModel::UpdateInputsFromSolution(Vec vector,int name, int type){

	double* serial_vector=NULL;

	VecToMPISerial(&serial_vector,vector);

	elements->UpdateInputsFromSolution(serial_vector,name,type);
	nodes->UpdateInputsFromSolution(serial_vector,name,type);
	vertices->UpdateInputsFromSolution(serial_vector,name,type);
	loads->UpdateInputsFromSolution(serial_vector,name,type);
	materials->UpdateInputsFromSolution(serial_vector,name,type);
	parameters->UpdateInputsFromSolution(serial_vector,name,type);

	/*Free ressources:*/
	xfree((void**)&serial_vector);

}
/*}}}*/
/*FUNCTION FemModel::UpdateInputsFromSolution(double* vector, int name, int type){{{1*/
void FemModel::UpdateInputsFromSolution(double* vector,int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
