/*!\file:  DakotaResponses.cpp
 * \brief  compute dakota responses, using a list of response descriptors.
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../DataSet/DataSet.h"    
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../MassFluxx/MassFluxx.h"
#include "../Misfitx/Misfitx.h"

void DakotaResponses(double* responses,char** responses_descriptors,int numresponses,Model* model,DataSet* results,DataSet* processed_results,int analysis_type,int sub_analysis_type){

	int i,j;
	int found=0;
	char* response_descriptor=NULL;
	int numberofnodes;
	extern int my_rank;

	/*some data needed across the responses: */
	model->FindParam(&numberofnodes,NumberOfNodesEnum);


	for(i=0;i<numresponses;i++){

		response_descriptor=responses_descriptors[i];

		//'min_vx' 'max_vx' 'max_abs_vx' 'min_vy' 'max_vy' 'max_abs_vy' 'min_vel' 'max_vel, mass_flux'

		if(strcmp(response_descriptor,"min_vel")==0){
			double* vel=NULL;
			double min_vel=0;
		
			found=processed_results->FindResult((void*)&vel,"vel");
			if(!found)ISSMERROR(" could not find vel to compute min_vel");

			min_vel=vel[0];
			for(j=1;j<numberofnodes;j++){
				if (vel[j]<min_vel)min_vel=vel[j];
			}

			if(my_rank==0)responses[i]=min_vel;
			
			/*Free ressources:*/
			xfree((void**)&vel);

			
		}
		else if(strcmp(response_descriptor,"max_vel")==0){
			double* vel=NULL;
			double max_vel=0;

			found=processed_results->FindResult((void*)&vel,"vel");
			if(!found)ISSMERROR(" could not find vel to compute max_vel");

			max_vel=vel[0];
			for(j=1;j<numberofnodes;j++){
				if (vel[j]>max_vel)max_vel=vel[j];
			}
			if(my_rank==0)responses[i]=max_vel;
			
			/*Free ressources:*/
			xfree((void**)&vel);

		}
		else if(strcmp(response_descriptor,"min_vx")==0){
			double* vx=NULL;
			double min_vx=0;
			
			found=processed_results->FindResult((void*)&vx,"vx");
			if(!found)ISSMERROR(" could not find vx to compute min_vx");

			min_vx=vx[0];
			for(j=1;j<numberofnodes;j++){
				if (vx[j]<min_vx)min_vx=vx[j];
			}
			if(my_rank==0)responses[i]=min_vx;
			
			/*Free ressources:*/
			xfree((void**)&vx);

		}
		else if(strcmp(response_descriptor,"max_vx")==0){
			double* vx=NULL;
			double max_vx=0;
			
			found=processed_results->FindResult((void*)&vx,"vx");
			if(!found)ISSMERROR(" could not find vx to compute max_vx");

			max_vx=vx[0];
			for(j=1;j<numberofnodes;j++){
				if (vx[j]>max_vx)max_vx=vx[j];
			}
			if(my_rank==0)responses[i]=max_vx;
			
			/*Free ressources:*/
			xfree((void**)&vx);

		}
		else if(strcmp(response_descriptor,"max_abs_vx")==0){
			double* vx=NULL;
			double max_abs_vx=0;
			
			found=processed_results->FindResult((void*)&vx,"vx");
			if(!found)ISSMERROR(" could not find vx to compute max_abs_vx");

			max_abs_vx=fabs(vx[0]);
			for(j=1;j<numberofnodes;j++){
				if (fabs(vx[j])>max_abs_vx)max_abs_vx=fabs(vx[j]);
			}
			if(my_rank==0)responses[i]=max_abs_vx;
			
			/*Free ressources:*/
			xfree((void**)&vx);

		}
		else if(strcmp(response_descriptor,"min_vy")==0){
			double* vy=NULL;
			double min_vy=0;
			
			found=processed_results->FindResult((void*)&vy,"vy");
			if(!found)ISSMERROR(" could not find vy to compute min_vy");

			min_vy=vy[0];
			for(j=1;j<numberofnodes;j++){
				if (vy[j]<min_vy)min_vy=vy[j];
			}
			if(my_rank==0)responses[i]=min_vy;
			
			/*Free ressources:*/
			xfree((void**)&vy);

		}
		else if(strcmp(response_descriptor,"max_vy")==0){
			double* vy=NULL;
			double max_vy=0;
			
			found=processed_results->FindResult((void*)&vy,"vy");
			if(!found)ISSMERROR(" could not find vy to compute max_vy");

			max_vy=vy[0];
			for(j=1;j<numberofnodes;j++){
				if (vy[j]>max_vy)max_vy=vy[j];
			}
			if(my_rank==0)responses[i]=max_vy;
			
			/*Free ressources:*/
			xfree((void**)&vy);

		}
		else if(strcmp(response_descriptor,"max_abs_vy")==0){
			double* vy=NULL;
			double max_abs_vy=0;
			
			found=processed_results->FindResult((void*)&vy,"vy");
			if(!found)ISSMERROR(" could not find vy to compute max_abs_vy");

			max_abs_vy=fabs(vy[0]);
			for(j=1;j<numberofnodes;j++){
				if (fabs(vy[j])>max_abs_vy)max_abs_vy=fabs(vy[j]);
			}
			if(my_rank==0)responses[i]=max_abs_vy;
			
			/*Free ressources:*/
			xfree((void**)&vy);

		}
		else if(strcmp(response_descriptor,"min_vz")==0){
			double* vz=NULL;
			double min_vz=0;
			
			found=processed_results->FindResult((void*)&vz,"vz");
			if(!found)ISSMERROR(" could not find vz to compute min_vz");

			min_vz=vz[0];
			for(j=1;j<numberofnodes;j++){
				if (vz[j]<min_vz)min_vz=vz[j];
			}
			if(my_rank==0)responses[i]=min_vz;
			
			/*Free ressources:*/
			xfree((void**)&vz);

		}
		else if(strcmp(response_descriptor,"max_vz")==0){
			double* vz=NULL;
			double max_vz=0;
			
			found=processed_results->FindResult((void*)&vz,"vz");
			if(!found)ISSMERROR(" could not find vz to compute max_vz");

			max_vz=vz[0];
			for(j=1;j<numberofnodes;j++){
				if (vz[j]>max_vz)max_vz=vz[j];
			}
			if(my_rank==0)responses[i]=max_vz;
			
			/*Free ressources:*/
			xfree((void**)&vz);

		}
		else if(strcmp(response_descriptor,"max_abs_vz")==0){
			double* vz=NULL;
			double max_abs_vz=0;
			
			found=processed_results->FindResult((void*)&vz,"vz");
			if(!found)ISSMERROR(" could not find vz to compute max_abs_vz");

			max_abs_vz=fabs(vz[0]);
			for(j=1;j<numberofnodes;j++){
				if (fabs(vz[j])>max_abs_vz)max_abs_vz=fabs(vz[j]);
			}
			if(my_rank==0)responses[i]=max_abs_vz;
			
			/*Free ressources:*/
			xfree((void**)&vz);
		}
		else if(strcmp(response_descriptor,"misfit")==0){
			
			int isstokes,ismacayealpattyn,ishutter;
			FemModel* femmodel=NULL;
			double J=0;
			int numberofdofspernode,numberofnodes;
			Vec u_g=NULL;
			double* u_g_double=NULL;
			double* vx=NULL;
			double* vy=NULL;
			double* vz=NULL;
			double* fit=NULL;

			/*retrieve active fem model: */
			model->FindParam(&isstokes,IsStokesEnum);
			model->FindParam(&ismacayealpattyn,IsMacAyealPattynEnum);
			model->FindParam(&ishutter,IsHutterEnum);

			if(isstokes){
				femmodel=model->GetFormulation(DiagnosticAnalysisEnum,StokesAnalysisEnum);
			}
			if(ismacayealpattyn){
				femmodel=model->GetFormulation(DiagnosticAnalysisEnum,HorizAnalysisEnum);
			}
			if(ishutter){
				femmodel=model->GetFormulation(DiagnosticAnalysisEnum,HutterAnalysisEnum);
			}	


			/*Recover some parameters: */
			femmodel->parameters->FindParam(&numberofdofspernode,NumberOfDofsPernodeEnum);
			femmodel->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
			femmodel->parameters->FindParam(&fit,NULL,NULL,FitEnum);

			/*Recover velocity: */
			found=results->FindResult(&u_g,"u_g");
			VecToMPISerial(&u_g_double,u_g);
			if(!found)ISSMERROR(" could not find velocity to compute misfit");

			SplitSolutionVector(u_g,numberofnodes,numberofdofspernode,&vx,&vy,&vz);
			SplitSolutionVector(u_g,numberofnodes,2,temperature,melting);

			/*Add to inputs: */
			femmodel->elements->UpdateInputsFromVector(vx,VxEnum,VertexEnum);
			femmodel->elements->UpdateInputsFromVector(vy,VyEnum,VertexEnum);
			femmodel->elements->UpdateInputsFromVector(vz,VzEnum,VertexEnum);
			femmodel->elements->UpdateInputsFromVector(fit[0],FitEnum,ConstantEnum);

			/*Compute misfit: */
			Misfitx( &J, femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,analysis_type,sub_analysis_type);
			

			if(my_rank==0)responses[i]=J;

			/*Some cleanup: */
			VecFree(&u_g);
			xfree((void**)&u_g_double);
			xfree((void**)&fit);

		}
		else if(strcmp(response_descriptor,"mass_flux")==0){

			int isstokes,ismacayealpattyn,ishutter;
			Vec       ug=NULL;
			double*   ug_serial=NULL;
			double    mass_flux=0;
			double*   segments=NULL;
			int       num_segments;
			DataSet*  elements=NULL;
			DataSet*  nodes=NULL;
			DataSet*  materials=NULL;
			DataSet*  parameters=NULL;
			FemModel* femmodel=NULL;
			Param*    param=NULL;

			/*retrieve velocities: */
			found=results->FindResult(&ug,"u_g");
			if(!found)ISSMERROR(" could not find velocity to compute mass_flux");
			VecToMPISerial(&ug_serial,ug);
		
			/*retrieve active fem model: */
			model->FindParam(&isstokes,IsStokesEnum);
			model->FindParam(&ismacayealpattyn,IsMacAyealPattynEnum);
			model->FindParam(&ishutter,IsHutterEnum);

			if(isstokes){
				femmodel=model->GetFormulation(DiagnosticAnalysisEnum,StokesAnalysisEnum);
			}
			if(ismacayealpattyn){
				femmodel=model->GetFormulation(DiagnosticAnalysisEnum,HorizAnalysisEnum);
			}
			if(ishutter){
				femmodel=model->GetFormulation(DiagnosticAnalysisEnum,HutterAnalysisEnum);
			}

			/*retrieve qmu_mass_flux_segments: */
			param=(Param*)femmodel->parameters->FindParamObject(QmuMassFluxSegmentsEnum);
			if(!param)ISSMERROR(" could not find qmu_mass_flux_segments to compute mass_flux");
			
			param->GetParameterValue(&segments);
			num_segments=param->GetM();

			/*call mass flux module: */
			MassFluxx(&mass_flux,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,segments,num_segments,ug_serial);
			
			if(my_rank==0)responses[i]=mass_flux;
			
			/*Free ressources:*/
			VecFree(&ug);
			xfree((void**)&ug_serial);
			xfree((void**)&segments);
		}
		else{
			if(my_rank==0)printf("%s%s%s"," response descriptor : ",response_descriptor," not supported yet!");
			ISSMERROR("%s%s%s"," response descriptor : ",response_descriptor," not supported yet!");
		}
	}

}
