/*!\file:  objectivefunctionC
 * \brief  objective function that returns a misfit, for a certain parameter.
 */ 

#include "../issm.h"
#include "./parallel.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

double objectivefunctionC(double search_scalar,OptArgs* optargs){

	int i;  
	
	/*output: */
	double J;
	
	/*parameters: */
	Model*    model=NULL;
	FemModel* femmodel=NULL;
	DataSet* diagnostic_results=NULL;
	double* param_g=NULL;
	double* grad_g=NULL;
	ParameterInputs* inputs=NULL;
	int numberofdofspernode;
	int n;

	/*intermediary:*/
	int gsize;
	double* optscal=NULL;
	double* fit=NULL;
	double  cm_min;
	double  cm_max;
	char*   control_type=NULL;
	double* param_g_copy=NULL;
	int     analysis_type;
	int     sub_analysis_type;
	int     control_steady;
	Vec     u_g=NULL;
	Vec     u_g_full=NULL;
	double* u_g_double=NULL;
	int     numberofnodes;
		
	/*steadystate: */
	int     dt=0;
	int     isstokes=0;
	DataSet* results_steadystate=NULL;
	int dofs01[2]={0,1};
	double* dofset=NULL;

	/*Recover active model: */
	model=optargs->model;
	femmodel=model->GetActiveFormulation();

	/*Recover parameters: */
	param_g=optargs->param_g;
	grad_g=optargs->grad_g;
	inputs=optargs->inputs;
	n=optargs->n;

	gsize=femmodel->nodesets->GetGSize();
	femmodel->parameters->FindParam(&optscal,NULL,NULL,OptScalEnum);
	femmodel->parameters->FindParam(&control_type,ControlTypeEnum);
	femmodel->parameters->FindParam(&fit,NULL,NULL,FitEnum);
	femmodel->parameters->FindParam(&cm_min,CmMinEnum);
	femmodel->parameters->FindParam(&cm_max,CmMaxEnum);
	femmodel->parameters->FindParam(&control_type,ControlTypeEnum);
	femmodel->parameters->FindParam(&control_steady,ControlSteadyEnum);
	femmodel->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	femmodel->parameters->FindParam(&sub_analysis_type,SubAnalysisTypeEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&numberofnodes,NumberOfNodesEnum);
	femmodel->parameters->FindParam(&numberofdofspernode,NumberOfDofsPerNodeEnum);

	/*First copy param_g so we don't modify it: */
	param_g_copy=(double*)xmalloc(numberofnodes*sizeof(double));
	memcpy(param_g_copy,param_g,numberofnodes*sizeof(double));

	/*First, update param_g using search_scalar: */
	for(i=0;i<numberofnodes;i++)param_g_copy[i]=param_g_copy[i]+search_scalar*optscal[n]*grad_g[i];

	/*Constrain:*/
	ControlConstrainx(param_g_copy,numberofnodes,cm_min,cm_max,control_type);

	/*Add new parameter to inputs: */
	inputs->Add(control_type,param_g_copy,1,numberofnodes);

	/*Run diagnostic with updated parameters.*/
	if(!control_steady){
		diagnostic_core_nonlinear(&u_g,NULL,NULL,NULL,femmodel,inputs,DiagnosticAnalysisEnum,sub_analysis_type);
		VecToMPISerial(&u_g_double,u_g); VecFree(&u_g);
		inputs->Add("velocity",u_g_double,numberofdofspernode,numberofnodes);
	}
	else{
		//We need a 3D velocity!! (vz is required for the next thermal run)
		diagnostic_results=new DataSet(ResultsEnum); 
		diagnostic_core(diagnostic_results,model, inputs);

		//extract u_g and add it to input (3d velocity needed by thermal_core)
		diagnostic_results->FindResult(&u_g,"u_g");
		inputs->Add("velocity",u_g,3,numberofnodes);
		delete diagnostic_results;
	}

	/*Compute misfit for this velocity field.*/
	inputs->Add("fit",fit[n]);
	CostFunctionx( &J, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters,inputs,analysis_type,sub_analysis_type);

	/*Free ressources:*/
	xfree((void**)&fit);
	xfree((void**)&optscal);
	xfree((void**)&control_type);
	xfree((void**)&param_g_copy);
	xfree((void**)&u_g_double);
	xfree((void**)&dofset);

	return J;
}
