/*!\file Beam.c
 * \brief: implementation of the Beam object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
#include "../../include/typedefs.h"
#include "../../include/macros.h"

/*Object constructors and destructor*/
/*FUNCTION Beam::Beam(){{{1*/
Beam::Beam(){
	this->inputs=NULL;
	this->parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION Beam::Beam(int id, int* node_ids, int matice_id, int matpar_id){{{1*/
Beam::Beam(int beam_id,int* beam_node_ids, int beam_matice_id, int beam_matpar_id):
	hnodes(beam_node_ids,2),
	hmatice(&beam_matice_id,1),
	hmatpar(&beam_matpar_id,1)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=beam_id;
	this->parameters=NULL;
	this->inputs=new Inputs();
}
/*}}}*/
/*FUNCTION Beam::Beam(int id, Hook* hnodes, Hook* hmatice, Hook* hmatpar, Parameters* beam_parameters, ElementProperties* properties){{{1*/
Beam::Beam(int beam_id,Hook* beam_hnodes, Hook* beam_hmatice, Hook* beam_hmatpar, Parameters* beam_parameters, Inputs* beam_inputs):
	hnodes(beam_hnodes),
	hmatice(beam_hmatice),
	hmatpar(beam_hmatpar)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=beam_id;
	if(beam_inputs){
		this->inputs=(Inputs*)beam_inputs->Copy();
	}
	else{
		this->inputs=new Inputs();
	}
	/*point parameters: */
	this->parameters=beam_parameters;
}
/*}}}*/
/*FUNCTION Beam::Beam(int id, int i, IoModel* iomodel){{{1*/
Beam::Beam(int beam_id, int index,IoModel* iomodel){

	int i;

	/*beam constructor input: */
	int   beam_matice_id;
	int   beam_matpar_id;
	int   beam_node_ids[2];
	double nodeinputs[2];

	/*id: */
	this->id=beam_id;

	/*hooks: */
	beam_matice_id=index+1; //refers to the corresponding material property card
	beam_matpar_id=iomodel->numberofvertices2d*(iomodel->numlayers-1)+1;//refers to the corresponding matpar property card
	beam_node_ids[0]=index+1;
	beam_node_ids[1]=(int)iomodel->uppernodes[index]; //grid that lays right on top
	
	this->hnodes.Init(beam_node_ids,2);
	this->hmatice.Init(&beam_matice_id,1);
	this->hmatpar.Init(&beam_matpar_id,1);

	//intialize inputs, and add as many inputs per element as requested: 
	this->inputs=new Inputs();

	if (iomodel->thickness) {
		nodeinputs[0]=iomodel->thickness[index];
		nodeinputs[1]=iomodel->thickness[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		nodeinputs[0]=iomodel->surface[index];
		nodeinputs[1]=iomodel->surface[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(SurfaceEnum,nodeinputs));
	}	
	if (iomodel->bed) {
		nodeinputs[0]=iomodel->bed[index];
		nodeinputs[1]=iomodel->bed[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(BedEnum,nodeinputs));
	}	
	if (iomodel->drag_coefficient) {
		nodeinputs[0]=iomodel->drag_coefficient[index];
		nodeinputs[1]=iomodel->drag_coefficient[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(DragCoefficientEnum,nodeinputs));
	}
	
	if (iomodel->vx) {
		nodeinputs[0]=iomodel->vx[index];
		nodeinputs[1]=iomodel->vx[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(VxEnum,nodeinputs));
	}
	if (iomodel->vy) {
		nodeinputs[0]=iomodel->vy[index];
		nodeinputs[1]=iomodel->vy[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(VyEnum,nodeinputs));
	}
	if (iomodel->vz) {
		nodeinputs[0]=iomodel->vz[index];
		nodeinputs[1]=iomodel->vz[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(VzEnum,nodeinputs));
	}
	if (iomodel->vx_obs) {
		nodeinputs[0]=iomodel->vx_obs[index];
		nodeinputs[1]=iomodel->vx_obs[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(VxEnum,nodeinputs));
	}
	if (iomodel->vy_obs) {
		nodeinputs[0]=iomodel->vy_obs[index];
		nodeinputs[1]=iomodel->vy_obs[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(VyEnum,nodeinputs));
	}
	if (iomodel->vz_obs) {
		nodeinputs[0]=iomodel->vz_obs[index];
		nodeinputs[1]=iomodel->vz_obs[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(VzEnum,nodeinputs));
	}
	if (iomodel->weights) {
		nodeinputs[0]=iomodel->weights[index];
		nodeinputs[1]=iomodel->weights[(int)(iomodel->uppernodes[index]-1)];
		this->inputs->AddInput(new BeamVertexInput(WeightsEnum,nodeinputs));
	}	
	if (iomodel->gridonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->gridonbed[index]));

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;


}
/*}}}*/
/*FUNCTION Beam::~Beam(){{{1*/
Beam::~Beam(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object management*/
/*FUNCTION Beam::Configure{{{1*/
void  Beam::Configure(DataSet* loadsin, DataSet* nodesin, DataSet* materialsin, Parameters* parametersin){

	int i;
	
	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	hmatice.configure(materialsin);
	hmatpar.configure(materialsin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Beam::copy{{{1*/
Object* Beam::copy() {
	
	return new Beam(this->id,&this->hnodes,&this->hmatice,&this->hmatpar,this->parameters,this->inputs);

}
/*}}}*/
/*FUNCTION Beam::DeepEcho{{{1*/
void Beam::DeepEcho(void){

	printf("Beam:\n");
	printf("   id: %i\n",id);
	hnodes.DeepEcho();
	hmatice.DeepEcho();
	hmatpar.DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Beam::Demarshall{{{1*/
void  Beam::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	hmatice.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);

	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Beam::Echo {{{1*/
void Beam::Echo(void){

	printf("Beam:\n");
	printf("   id: %i\n",id);
	hnodes.Echo();
	hmatice.Echo();
	hmatpar.Echo();
	printf("   parameters\n");
	parameters->Echo();
	printf("   inputs\n");
	inputs->Echo();

	return;
}
/*}}}*/
/*FUNCTION Beam::Marshall{{{1*/
void  Beam::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Beam: */
	enum_type=BeamEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Beam data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);

	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	hmatice.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;
	
	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Beam::MarshallSize{{{1*/
int   Beam::MarshallSize(){
	
	return sizeof(id)
		+hnodes.MarshallSize()
		+hmatice.MarshallSize()
		+hmatpar.MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Beam::UpdateInputsFromSolution {{{1*/
void  Beam::UpdateInputsFromSolution(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Object functions*/
/*FUNCTION Beam::ComputeBasalStress{{{1*/
void  Beam::ComputeBasalStress(Vec eps,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Beam::ComputePressure{{{1*/
void  Beam::ComputePressure(Vec p_g,int analysis_type,int sub_analysis_type){

	int i;
	const int numgrids=2;
	int doflist[numgrids];
	double pressure[numgrids];
	double surface[numgrids];
	double rho_ice,g;
	double xyz_list[numgrids][3];
	double gauss[numgrids][numgrids]={{1,0},{0,1}};

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
        
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*recover value of surface at gauss points (0,1) and (1,0): */
	inputs->GetParameterValues(&surface[0],&gauss[0][0],2,SurfaceEnum);
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(surface[i]-xyz_list[i][2]);
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(p_g,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Beam::ComputeStrainRate{{{1*/
void  Beam::ComputeStrainRate(Vec eps,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Beam::CostFunction{{{1*/
double Beam::CostFunction(int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::CreateKMatrix{{{1*/

void  Beam::CreateKMatrix(Mat Kgg,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum) {
	
		if (sub_analysis_type==HutterAnalysisEnum) {

			CreateKMatrixDiagnosticHutter( Kgg,analysis_type,sub_analysis_type);
		}
		else 
			ISSMERROR("%s%i%s\n","sub_analysis_type: ",sub_analysis_type," not supported yet");
	}
	else{
		ISSMERROR("%s%i%s\n","analysis: ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Beam::CreateKMatrixDiagnosticHutter{{{1*/

void  Beam::CreateKMatrixDiagnosticHutter(Mat Kgg,int analysis_type,int sub_analysis_type){
	
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    Ke_gg[numdofs][numdofs]={{0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0}};
	int       numberofdofspernode;
	bool onbed;
	
	
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	GetDofList(&doflist[0],&numberofdofspernode);

	if (onbed){
		Ke_gg[0][0]=1;
		Ke_gg[1][1]=1;
		Ke_gg[2][0]=-1;
		Ke_gg[2][2]=1;
		Ke_gg[3][1]=-1;
		Ke_gg[3][3]=1;
	}
	else{
		Ke_gg[2][0]=-1;
		Ke_gg[2][2]=1;
		Ke_gg[3][1]=-1;
		Ke_gg[3][3]=1;
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Beam::CreatePVector{{{1*/
void  Beam::CreatePVector(Vec pg,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum) {
		if (sub_analysis_type==HutterAnalysisEnum) {
			CreatePVectorDiagnosticHutter( pg,analysis_type,sub_analysis_type);
		}
		else
			ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}
	else{
		ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Beam::CreatePVectorDiagnosticHutter{{{1*/

void Beam::CreatePVectorDiagnosticHutter( Vec pg,  int analysis_type,int sub_analysis_type){

	int i,j,k;
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    pe_g[4]={0,0,0,0};
	double    pe_g_gaussian[4]={0,0,0,0};
	int       found=0;
	int       dofs[1]={0};
	double    xyz_list[numgrids][3];
	double    z_list[numgrids];
	double    constant_part;
	int       numberofdofspernode;

	/*gaussian points: */
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double   gauss_coord;
	double*  gauss_weights=NULL;
	double   gauss_weight;
	double   gauss1[2]={1,0};
	int      ig;
	double   l1l2[2];
	double   slope[2];
	double   slope2;

	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;
	double   surface,thickness;
	
	bool onbed;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*recover doflist: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover some inputs: */
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	/*recover parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();

	//recover extra inputs
	inputs->GetParameterValue(&slope[0],&gauss1[0],SurfaceSlopexEnum);
	inputs->GetParameterValue(&slope[1],&gauss1[0],SurfaceSlopeyEnum);
	inputs->GetParameterValue(&surface,&gauss1[0],SurfaceEnum);
	inputs->GetParameterValue(&thickness,&gauss1[0],ThicknessEnum);

	//Get all element grid data:
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	for(i=0;i<numgrids;i++)z_list[i]=xyz_list[i][2];
	
	//compute slope2 slopex and slopy
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//%compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	GaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	//Start  looping on the number of gaussian points:
	for(ig=0;ig<num_gauss;ig++){

		//Pick up the gaussian point and its weight:
		gauss_weight=gauss_weights[ig];
		gauss_coord=segment_gauss_coord[ig];

		//compute constant_part
		GetParameterValue(&z_g, &z_list[0],gauss_coord);

		//Get Jacobian determinant:
		GetJacobianDeterminant(&Jdet, &z_list[0],gauss_coord);
		
		for(j=0;j<NDOF2;j++){
			pe_g_gaussian[NDOF2+j]=constant_part*pow((surface-z_g)/B,n)*slope[j]*Jdet*gauss_weight;
		}
		
		//add pe_gaussian vector to pe:
		for(j=0;j<numdofs;j++){
			pe_g[j]+=pe_g_gaussian[j];
		}
	} //for(ig=0;ig<num_gauss;ig++)

	//Deal with lower surface
	if (onbed){

		//compute ub
		constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*thickness;
		ub=constant_part*slope[0];
		vb=constant_part*slope[1];

		//Add to pe: 
		pe_g[0]+=ub;
		pe_g[1]+=vb;
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&segment_gauss_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Beam::Du{{{1*/
void  Beam::Du(Vec,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Enum{{{1*/
int Beam::Enum(void){

	return BeamEnum;

}
/*}}}*/
/*FUNCTION Beam::GetBedList{{{1*/
void  Beam::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetDofList{{{1*/
void  Beam::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	for(i=0;i<2;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Beam::GetDofList1{{{1*/
void  Beam::GetDofList1(int* doflist){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	int i;
	for(i=0;i<2;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Beam::Id{{{1*/
int    Beam::Id(void){ return id; }
/*}}}*/
/*FUNCTION Beam::GetJacobianDeterminant{{{1*/
void Beam::GetJacobianDeterminant(double* pJdet,double* z_list, double gauss_coord){


	double Jdet;

	Jdet=1.0/2.0*(z_list[1]-z_list[0]);

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}
	
	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Beam::GetMatPar{{{1*/
void* Beam::GetMatPar(){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	return matpar;
}
/*}}}*/
/*FUNCTION Beam::GetNodalFunctions{{{1*/
void Beam::GetNodalFunctions(double* l1l2, double gauss_coord){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2[0]=.5*gauss_coord+.5;

	/*Second nodal function: */
	l1l2[1]=-.5*gauss_coord+.5;
}
/*}}}*/
/*FUNCTION Beam::GetNodes{{{1*/
void  Beam::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=NULL;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	
	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Beam::GetOnBed{{{1*/
bool   Beam::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetParameterValue{{{1*/
void Beam::GetParameterValue(double* pvalue, double* value_list,double gauss_coord){

	double l1l2[2];
	
	GetNodalFunctions(&l1l2[0],gauss_coord);

	*pvalue=l1l2[0]*value_list[0]+l1l2[1]*value_list[1];
}
/*}}}*/
/*FUNCTION Beam::GetShelf{{{1*/
bool   Beam::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GetThicknessList{{{1*/
void  Beam::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Gradj{{{1*/
void  Beam::Gradj(Vec, int, int,char*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GradjB{{{1*/
void  Beam::GradjB(Vec, int, int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::GradjDrag{{{1*/
void  Beam::GradjDrag(Vec, int,int ){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MassFlux{{{1*/
double Beam::MassFlux( double* segment,double* ug){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::Misfit{{{1*/
double Beam::Misfit(int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam::MyRank{{{1*/
int    Beam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Beam::SurfaceArea{{{1*/
double Beam::SurfaceArea(int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
//*FUNCTION Beam::SetClone {{{1*/
void  Beam::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/

