/*
 * \file Inputs.c
 * \brief: implementation of the Inputs class, derived from DataSet class
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/
/*Inputs: {{{1*/

/*Object constructors and destructor*/
/*FUNCTION Inputs::Inputs(){{{2*/
Inputs::Inputs(){
	return;
}
/*}}}*/
/*FUNCTION Inputs::Inputs(int in_enum){{{2*/
Inputs::Inputs(int in_enum): DataSet(in_enum) {
	//do nothing;
	return;
}
/*}}}*/
/*FUNCTION Inputs::~Inputs(){{{2*/
Inputs::~Inputs(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type){{{2*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,gauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,double* gauss,int enum_type,double defaultvalue){{{2*/
void Inputs::GetParameterValue(double* pvalue,double* gauss, int enum_type,double defaultvalue){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. Return the default value: */
		*pvalue=defaultvalue;
	}
	else{
		input->GetParameterValue(pvalue,gauss);
	}
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){{{2*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValues(values,gauss_pointers,numgauss);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue, Node* node, int enum_type){{{2*/
void Inputs::GetParameterValue(double* pvalue,Node* node,int enum_type){

	/*given a node, instead of a gauss point, we want to recover a value: probably in an element!: */

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,node);
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue, Node* node1, Node* node2,int enum_type){{{2*/
void Inputs::GetParameterValue(double* pvalue,Node* node1, Node* node2,double gauss_coord,int enum_type){

	/*given a node, instead of a gauss point, we want to recover a value: probably in an element!: */

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the values: */
	input->GetParameterValue(pvalue,node1,node2,gauss_coord);
}
/*}}}*/
/*FUNCTION Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){{{2*/
void Inputs::GetParameterValues(double* values,double* gauss_pointers, int numgauss,int enum_type,double* defaultvalues){

	int i;
	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. Return the default values: */
		for(i=0;i<numgauss;i++)values[i]=defaultvalues[i];
	}
	else{
		input->GetParameterValues(values,gauss_pointers,numgauss);
	}

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(bool* pvalue,int enum-type){{{2*/
void Inputs::GetParameterValue(bool* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(int* pvalue,int enum-type){{{2*/
void Inputs::GetParameterValue(int* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,int enum-type){{{2*/
void Inputs::GetParameterValue(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){{{2*/
void Inputs::GetParameterDerivativeValue(double* derivativevalues, double* xyz_list, double* gauss,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type)break;
	}

	if (!input){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i"," could not find input with enum type ",enum_type);
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterDerivativeValue(derivativevalues,xyz_list,gauss);
}
/*}}}*/
/*FUNCTION Inputs::GetStrainRate(double* epsilon,double* xyz_list, double* gauss, int xenum, int yenum){{{2*/
void Inputs::GetStrainRate(double* epsilon,double* xyz_list, double* gauss, int xenum, int yenum){

	vector<Object*>::iterator object;
	Input* xinput=NULL;
	Input* yinput=NULL;

	/*Go through inputs and find data for xenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		xinput=(Input*)(*object); 
		if (xinput->EnumType()==xenum)break;
	}
	/*Go through inputs and find data for yenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		yinput=(Input*)(*object); 
		if (yinput->EnumType()==yenum)break;
	}

	if (!xinput | !yinput){
		/*we could not find one input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i%s%i\n"," could not find input with enum type ",xenum," or enum type ",yenum);
	}

	/*Ok, we have the inputs, call bilinear operator: */
	xinput->GetStrainRate(epsilon,yinput,xyz_list,gauss);

}
/*}}}*/
/*FUNCTION Inputs::GetStrainRateStokes(double* epsilon,double* xyz_list, double* gauss, int xenum, int yenum,int zenum){{{2*/
void Inputs::GetStrainRateStokes(double* epsilon,double* xyz_list, double* gauss, int xenum, int yenum,int zenum){

	vector<Object*>::iterator object;
	Input* xinput=NULL;
	Input* yinput=NULL;
	Input* zinput=NULL;

	/*Go through inputs and find data for xenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		xinput=(Input*)(*object); 
		if (xinput->EnumType()==xenum)break;
	}
	/*Go through inputs and find data for yenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		yinput=(Input*)(*object); 
		if (yinput->EnumType()==yenum)break;
	}
	/*Go through inputs and find data for zenum: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		zinput=(Input*)(*object); 
		if (zinput->EnumType()==zenum)break;
	}

	if (!xinput | !yinput | !zinput){
		/*we could not find one input with the correct enum type. No defaults values were provided, 
		 * error out: */
		ISSMERROR("%s%i%s%i%s%i\n"," could not find input with enum type ",xenum," or enum type ",yenum, " or enum type ",zenum);
	}

	/*Ok, we have the inputs, call bilinear operator: */
	xinput->GetStrainRateStokes(epsilon,yinput,zinput,xyz_list,gauss);

}
/*}}}*/
/*FUNCTION Inputs::AddInput{{{2*/
int  Inputs::AddInput(Input* in_input){

	/*First, go through dataset of inputs and check whether any input 
	 * with the same name is already in. If so, erase the corresponding 
	 * object before adding this new one: */
	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==in_input->EnumType()){
			this->DeleteObject(input);
			break;
		}
	}
	this->AddObject(in_input);
}
/*}}}*/

/*Object functions*/


/*}}}*/
