/* \file FetchParams.c:
 * \brief: interface for fetching  matlab parameters into a "C" dataset 
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#ifdef _SERIAL_

#include <mex.h>
#include "./io.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../EnumDefinitions/EnumDefinitions.h"

void FetchParams(DataSet** pparameters, DataHandle dataref){

	int i,j;
	int count;

	/*output: */
	Param* param=NULL;
	Parameters* parameters=NULL;

	/*intermediary: */
	int M,N;
	double* tmatrix=NULL;
	double* matrix=NULL;
	char**  stringarray=NULL;
	int nfields;
	char* name=NULL;
	mxArray* pfield=NULL;
	mxArray* pfield2=NULL;
	int enum_type;


	/*First, create parameters : */
	parameters=new Parameters();

	/*go through matlab params structure, and create Param object for each field: */

	nfields=mxGetNumberOfFields(dataref);

	for(count=0;count<nfields;count++){

		/*Get i'th field: */
		name=(char*)mxGetFieldNameByNumber(dataref,count);
		enum_type=StringAsEnum(name);
		pfield=mxGetFieldByNumber(dataref,0,count);

		/*Check type of field: */
		if (mxIsDouble(pfield)){
			/*could be  DOUBLE, DOUBLEVEC or DOUBLEMAT, depends on dimensions: */

			M=mxGetM(pfield);
			N=mxGetN(pfield);

			if (M==0 | N==0){
				ISSMERROR("%s%i (%s) %s%i%s%i%s","array in parameters structure field ",count,name," is of size (",M,",",N,")");
			}
			if (M==1 && N==1){
				/*we have a simple scalar: */
				param= new DoubleParam(enum_type,*mxGetPr(pfield));
				parameters->AddObject(param);

			}
			else{
				if (N==1){
					
					/*vector: */
					param= new DoubleVecParam(enum_type,mxGetPr(pfield),M);
					parameters->AddObject(param);

				}
				else{
					/*matrix: first, transpose, then plug into Param */
					matrix=mxGetPr(pfield);
					tmatrix=(double*)xmalloc(M*N*sizeof(double));
					for (i=0;i<M;i++){
						for(j=0;j<N;j++){
							*(tmatrix+N*i+j)=*(matrix+M*j+i);
						}
					}

					param= new DoubleMatParam(enum_type,tmatrix,M,N);
					parameters->AddObject(param);
	
					/*Free ressources:*/
					xfree((void**)&tmatrix);
				}
			}

		}
		else if (mxIsChar(pfield)){
			
			/* we have a string parameter:*/
			
			int stringlen;
			char* string=NULL;
			
			stringlen = mxGetM(pfield)*mxGetN(pfield)+1;
			string = (char*)xmalloc(sizeof(mxChar)*stringlen);
			mxGetString(pfield,string,stringlen);

			param= new StringParam(enum_type,string);
			parameters->AddObject(param);

			xfree((void**)&string);
		}
		else if (mxIsCell(pfield)){

			/*string array: */
			M=mxGetM(pfield);
			stringarray=(char**)xmalloc(M*sizeof(char*));

			for(i=0;i<M;i++){
				char* descriptor=NULL;
				pfield2=mxGetCell(pfield,i);
				FetchData(&descriptor,pfield2);
				stringarray[i]=descriptor;
			}
		
			param= new StringArrayParam(enum_type,stringarray,M);
			parameters->AddObject(param);

			/*Free ressources:*/
			for(i=0;i<M;i++){
				char* descriptor=stringarray[i];
				xfree((void**)&descriptor);
			}
			xfree((void**)&stringarray);

		}
		else ISSMERROR("%s%i","unknow type in parameters structure field ",i);
	}

	/*Assign output pointers:*/
	*pparameters=parameters;

}
#endif
