/*!\file PetscMatParam.c
 * \brief: implementation of the PetscMatParam object
 */

/*header files: */
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../DataSet/DataSet.h"
#include "../../include/typedefs.h"
#include "../../include/types.h"
#include "../../include/macros.h"
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION PetscMatParam::PetscMatParam(){{{1*/
PetscMatParam::PetscMatParam(){
	return;
}
/*}}}*/
/*FUNCTION PetscMatParam::PetscMatParam(int enum_type,IssmPetscMat value){{{1*/
PetscMatParam::PetscMatParam(int in_enum_type,Mat in_value){

	enum_type=in_enum_type;

	value=NULL;

	if(in_value){
		MatDuplicate(in_value,MAT_COPY_VALUES,&value);
	}
}
/*}}}*/
/*FUNCTION PetscMatParam::~PetscMatParam(){{{1*/
PetscMatParam::~PetscMatParam(){
	MatFree(&value);
}
/*}}}*/

/*Object methods*/
/*FUNCTION PetscMatParam::copy{{{1*/
Object* PetscMatParam::copy() {
	
	return new PetscMatParam(this->enum_type,this->value);

}
/*}}}*/
/*FUNCTION PetscMatParam::DeepEcho{{{1*/
void PetscMatParam::DeepEcho(void){

	int i;
	printf("PetscMatParam:\n");
	printf("   enum: %i\n",this->enum_type);
	MatView(value,PETSC_VIEWER_STDOUT_WORLD);
}
/*}}}*/
/*FUNCTION PetscMatParam::Demarshall{{{1*/
void  PetscMatParam::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i,j;
	int   M,N;
	double* serial_mat=NULL;
	int*    idxm=NULL;
	int*    idxn=NULL;
	double sparsity=.001;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*data: */
	memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
	memcpy(&N,marshalled_dataset,sizeof(N));marshalled_dataset+=sizeof(N);
	if(M!=0 && N!=0){
		serial_mat=(double*)xmalloc(M*N*sizeof(double));
		memcpy(serial_mat,marshalled_dataset,M*N*sizeof(double));marshalled_dataset+=(M*N*sizeof(double));
		value=NewMat(M,N,&sparsity,NULL,NULL);
		idxm=(int*)xmalloc(M*sizeof(int));
		idxn=(int*)xmalloc(N*sizeof(int));
		for(i=0;i<M;i++)idxm[i]=i;
		for(i=0;i<N;i++)idxn[i]=i;
		MatSetValues(value,M,idxm,N,idxn,serial_mat,INSERT_VALUES); 
		MatAssemblyBegin(value,MAT_FINAL_ASSEMBLY); 
		MatAssemblyEnd(value,MAT_FINAL_ASSEMBLY);

			}
	else{
		value=NULL;
	}

	/*Free ressources:*/
	xfree((void**)&serial_mat);
	xfree((void**)&idxm);
	xfree((void**)&idxn);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION PetscMatParam::Echo {{{1*/
void PetscMatParam::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION PetscMatParam::Enum{{{1*/
int PetscMatParam::Enum(void){

	return PetscMatParamEnum;

}
/*}}}*/
/*FUNCTION PetscMatParam::Id{{{1*/
int    PetscMatParam::Id(void){ return -1; }
/*}}}*/
/*FUNCTION PetscMatParam::Marshall{{{1*/
void  PetscMatParam::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;
	int   M,N;
	double* serial_mat=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of PetscMatParam: */
	enum_value=PetscMatParamEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall PetscMatParam data: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	if(value){
		MatGetSize(value,&M,&N);
		MatToSerial(&serial_mat,value);
		memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
		memcpy(marshalled_dataset,&N,sizeof(N));marshalled_dataset+=sizeof(N);
		memcpy(marshalled_dataset,serial_mat,M*N*sizeof(double));marshalled_dataset+=(M*N*sizeof(double));
	}
	else{
		M=0;
		N=0;
		memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
		memcpy(marshalled_dataset,&N,sizeof(N));marshalled_dataset+=sizeof(N);
	}
	
	/*Free ressources:*/
	xfree((void**)&serial_mat);

	/*return:*/
	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION PetscMatParam::MarshallSize{{{1*/
int   PetscMatParam::MarshallSize(){

	int M=0;
	int N=0;
	if(value)MatGetSize(value,&M,&N);
			
	return sizeof(M)+
		sizeof(N)+
		M*N*sizeof(double)+
		+sizeof(enum_type)+
		+sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION PetscMatParam::MyRank{{{1*/
int    PetscMatParam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION PetscMatParam::GetParameterValue(Mat* pvalue){{{1*/
void  PetscMatParam::GetParameterValue(Mat* poutput){
	Mat output=NULL;

	if(value){
		MatDuplicate(value,MAT_COPY_VALUES,&output);
	}
	*poutput=output;
}
/*}}}*/
/*FUNCTION PetscMatParam::GetParameterName(void); {{{1*/
char* PetscMatParam::GetParameterName(void){
	return  EnumAsString(this->enum_type);
}
/*}}}*/
/*FUNCTION PetscMatParam::SetMatlabField(mxArray* dataref);{{{1*/
void  PetscMatParam::SetMatlabField(mxArray* dataref){
	
	char* name=NULL;
	int   M,N;
	double* doublemat=NULL;
	mxArray* pfield=NULL;

	MatToSerial(&doublemat,value);
	MatGetSize(value,&M,&N);
	name=this->GetParameterName();
				
	pfield=mxCreateDoubleMatrix(0,0,mxREAL);
	mxSetM(pfield,M);
	mxSetN(pfield,N);
	mxSetPr(pfield,doublemat);
	mxSetField( dataref, 0, name, pfield);

	/*Free ressources:*/
	xfree((void**)&name);
	xfree((void**)&doublemat);
}
/*}}}*/
