/*!\file DofVec.c
 * \brief: implementation of the DofVec object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./objects.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../EnumDefinitions/EnumDefinitions.h"

/*Object constructors and destructor*/
/*FUNCTION DofVec::DofVec() {{{1*/
DofVec::DofVec(){
	return;
}
/*}}}*/
/*FUNCTION DofVec::DofVec(char* name) {{{1*/
DofVec::DofVec(char* name){

	strcpy(this->name,name);
	this->numdofs=0;
	this->numentries=0;
	this->vector=NULL;

}
/*}}}*/
/*FUNCTION DofVec::DofVec(int total_size,char* name) {{{1*/
DofVec::DofVec(int total_size,char* name){

	strcpy(this->name,name);
	this->numdofs=1;
	this->numentries=total_size;
	this->vector=NewVec(total_size);
}
/*}}}*/
/*FUNCTION DofVec::DofVec(int total_size,double default_value,char* name) {{{1*/
DofVec::DofVec(int total_size,double default_value,char* name){

	strcpy(this->name,name);
	this->numdofs=1;
	this->numentries=total_size;
	this->vector=NewVec(total_size);
	VecSet(this->vector,default_value);
}
/*}}}*/
/*FUNCTION DofVec::DofVec(int dofvec_numdofs,int dofvec_numentries,char* name) {{{1*/
DofVec::DofVec(int dofvec_numdofs,int dofvec_numentries,char* name){

	strcpy(this->name,name);
	this->numdofs=dofvec_numdofs;
	this->numentries=dofvec_numentries;
	this->vector=NewVec(dofvec_numdofs*dofvec_numentries);
}
/*}}}*/
/*FUNCTION DofVec::DofVec(int dofvec_numdofs,int dofvec_numentries,double default_value,char* name) {{{1*/
DofVec::DofVec(int dofvec_numdofs,int dofvec_numentries,double default_value,char* name){

	strcpy(this->name,name);
	this->numdofs=dofvec_numdofs;
	this->numentries=dofvec_numentries;
	this->vector=NewVec(dofvec_numdofs*dofvec_numentries);
	VecSet(this->vector,default_value);

}
/*}}}*/
/*FUNCTION DofVec::DofVec(double* serial_vector,int total_size,char* name) {{{1*/
DofVec::DofVec(double* serial_vector,int total_size,char* name){
	
	strcpy(this->name,name);
	this->numdofs=1;
	this->numentries=total_size;
	this->vector=SerialToVec(serial_vector,total_size);
}
/*}}}*/
/*FUNCTION DofVec::DofVec(double* serial_vector,int dofvec_numdofs,int dofvec_numentries,char* name) {{{1*/
DofVec::DofVec(double* serial_vector,int dofvec_numdofs,int dofvec_numentries,char* name){
	
	strcpy(this->name,name);
	this->numdofs=dofvec_numdofs;
	this->numentries=dofvec_numentries;
	this->vector=SerialToVec(serial_vector,dofvec_numdofs*dofvec_numentries);
}
/*}}}*/
/*FUNCTION DofVec::DofVec(Vec vec,char* name) {{{1*/
DofVec::DofVec(Vec vec,char* name){

	strcpy(this->name,name);
	this->numdofs=1;
	VecGetSize(vec,&this->numentries);
	VecDuplicatePatch(&this->vector,vec);
	
}
/*}}}*/
/*FUNCTION DofVec::DofVec(Vec vec,int dofvec_numdofs,int dofvec_numentries,char* name) {{{1*/
DofVec::DofVec(Vec vec,int dofvec_numdofs,int dofvec_numentries,char* name){

	strcpy(this->name,name);
	this->numdofs=dofvec_numdofs;
	this->numentries=dofvec_numentries;
	VecDuplicatePatch(&this->vector,vec);
}
/*}}}*/
/*FUNCTION DofVec::~DofVec {{{1*/
DofVec::~DofVec(){
	VecFree(&this->vector);
}
/*}}}*/

/*Object management: */
/*FUNCTION DofVec::copy {{{1*/
Object* DofVec::copy() {

	DofVec* dofvec=NULL;

	dofvec=new DofVec(*this); 

	/*do the dynamic allocation copy: */
	VecDuplicatePatch(&dofvec->vector,this->vector);

	return (Object*)dofvec;

}
/*}}}*/
/*FUNCTION DofVec::DeepEcho {{{1*/
void DofVec::DeepEcho(void){

	printf("DofVec:\n");
	printf("   name: %s\n",this->name);
	printf("   numdofs: %i\n",this->numdofs);
	printf("   numentries: %i\n",this->numentries);
	printf("   totalsize: %i\n",this->numdofs*this->numentries);
	if(this->vector)VecView(this->vector,PETSC_VIEWER_STDOUT_WORLD);
}
/*}}}*/
/*FUNCTION DofVec::Demarshall {{{1*/
void  DofVec::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;
	double* serial_vector=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&name,marshalled_dataset,sizeof(name));marshalled_dataset+=sizeof(name);
	memcpy(&numdofs,marshalled_dataset,sizeof(numdofs));marshalled_dataset+=sizeof(numdofs);
	memcpy(&numentries,marshalled_dataset,sizeof(numentries));marshalled_dataset+=sizeof(numentries);

	serial_vector=(double*)xmalloc(this->numentries*this->numdofs*sizeof(double));
	memcpy(serial_vector,marshalled_dataset,this->numentries*this->numdofs*sizeof(double));marshalled_dataset+=this->numentries*this->numdofs*sizeof(double);

	this->vector=SerialToVec(serial_vector,this->numentries*this->numdofs);

	/*Free ressources:*/
	xfree((void**)&serial_vector);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION DofVec::Echo {{{1*/
void DofVec::Echo(void){

	printf("DofVec:\n");
	printf("   name: %s\n",this->name);
	printf("   numdofs: %i\n",this->numdofs);
	printf("   numentries: %i\n",this->numentries);
	printf("   totalsize: %i\n",this->numentries*this->numdofs);
	printf("   vector: %p\n",this->vector);

}
/*}}}*/
/*FUNCTION DofVec::Marshall {{{1*/
void  DofVec::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	double* serial_vector=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of DofVec: */
	enum_type=DofVecEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall DofVec data: */
	memcpy(marshalled_dataset,&name,sizeof(name));marshalled_dataset+=sizeof(name);
	memcpy(marshalled_dataset,&numdofs,sizeof(numdofs));marshalled_dataset+=sizeof(numdofs);
	memcpy(marshalled_dataset,&numentries,sizeof(numentries));marshalled_dataset+=sizeof(numentries);

	/*Serialize vector before marshalling it: */
	VecToMPISerial(&serial_vector,this->vector);
	
	memcpy(marshalled_dataset,serial_vector,this->numentries*this->numdofs*sizeof(double));marshalled_dataset+=this->numentries*this->numdofs*sizeof(double);

	/*Free ressources: */
	xfree((void**)&serial_vector);
	
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION DofVec::MarshallSize {{{1*/
int   DofVec::MarshallSize(){
	return sizeof(name)
		+sizeof(numdofs)
		+sizeof(numentries)
		+this->numentries*this->numdofs*sizeof(double)
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/

/*Object functions*/
/*FUNCTION DofVec::Enum {{{1*/
int DofVec::Enum(void){

	return DofVecEnum;

}
/*}}}*/
/*FUNCTION DofVec::Id {{{1*/
int DofVec::Id(void){ 
	ISSMERROR(" no id function for DofVec class");
}

/*FUNCTION DofVec::GetVec {{{1*/
Vec     DofVec::GetVec(void){
	return vector;
}
/*}}}*/
/*FUNCTION DofVec::GetVecFromDof(double** pvector,int dof) {{{1*/
void DofVec::GetVecFromDof(double** pvector,int dof){

	double* dofset=NULL;
	double* serial_vector=NULL;
	Vec     vector=NULL;

	
	/*check we are not out of bounds: */
	if (dof>=numdofs){
		ISSMERROR("%s%i%s%i\n"," error message: dof ",dof," requested is out of bounds for vector with number of dofs",this->numdofs);
	}

	dofset=dofsetgen(1,&dof,this->numdofs,this->numdofs*this->numentries);
	VecPartition(&vector, this->vector, dofset, this->numdofs*this->numentries);

	VecToMPISerial(&serial_vector,vector);

	/*Free ressources:*/
	VecFree(&vector);
	xfree((void**)&dofset);

	/*Assign output pointers:*/
	*pvector=serial_vector;
}
/*}}}*/
/*FUNCTION DofVec::GetVecFromDof(Vec* pvector, int dof) {{{1*/
void DofVec::GetVecFromDof(Vec* pvector, int dof){

	double* dofset=NULL;
	Vec     vector=NULL;

	
	/*check we are not out of bounds: */
	if (dof>=numdofs){
		ISSMERROR("%s%i%s%i\n"," error message: dof ",dof," requested is out of bounds for vector with number of dofs",this->numdofs);
	}

	dofset=dofsetgen(1,&dof,this->numdofs,this->numdofs*this->numentries);
	VecPartition(&vector, this->vector, dofset, this->numdofs*this->numentries);

	/*Free ressources:*/
	xfree((void**)&dofset);

	/*Assign output pointers:*/
	*pvector=vector;
}
/*}}}*/
/*FUNCTION DofVec::GetVecFromDofList(double** pvector, int requested_numdofs,int* requested_doflist) {{{1*/
void DofVec::GetVecFromDofList(double** pvector, int requested_numdofs,int* requested_doflist){

	int i;
	double* dofset=NULL;
	double* serial_vector=NULL;
	Vec     vector=NULL;
	
	/*check we are not out of bounds: */
	if (requested_numdofs>=this->numdofs) ISSMERROR("%s%i%s%i\n"," error message: requested dof list has size  ",requested_numdofs," which is out of bounds for vector with number of dofs",this->numdofs);

	for(i=0;i<requested_numdofs;i++){
		if (requested_doflist[i]>=this->numdofs)ISSMERROR("%s%i%s%i\n"," error message: requested dof   ",requested_doflist[i]," in list, is out of bounds for vector with number of dofs",this->numdofs);
	}

	
	dofset=dofsetgen(requested_numdofs,requested_doflist,this->numdofs,this->numdofs*this->numentries);
	VecPartition(&vector, this->vector, dofset, this->numdofs*this->numentries);

	VecToMPISerial(&serial_vector,vector);

	/*Free ressources:*/
	VecFree(&vector);
	xfree((void**)&dofset);


	/*Assign output pointers:*/
	*pvector=serial_vector;
}
/*}}}*/
/*FUNCTION DofVec::GetVecFromDofList(Vec* pvector, int requested_numdofs,int* requested_doflist) {{{1*/
void DofVec::GetVecFromDofList(Vec* pvector, int requested_numdofs,int* requested_doflist){

	int i;
	double* dofset=NULL;
	Vec     vector=NULL;
	
	/*check we are not out of bounds: */
	if (requested_numdofs>=this->numdofs) ISSMERROR("%s%i%s%i\n"," error message: requested dof list has size  ",requested_numdofs," which is out of bounds for vector with number of dofs",this->numdofs);

	for(i=0;i<requested_numdofs;i++){
		if (requested_doflist[i]>=this->numdofs)ISSMERROR("%s%i%s%i\n"," error message: requested dof   ",requested_doflist[i]," in list, is out of bounds for vector with number of dofs",this->numdofs);
	}

	
	dofset=dofsetgen(requested_numdofs,requested_doflist,this->numdofs,this->numdofs*this->numentries);
	VecPartition(&vector, this->vector, dofset, this->numdofs*this->numentries);


	/*Free ressources:*/
	xfree((void**)&dofset);

	/*Assign output pointers:*/
	*pvector=vector;
}
/*}}}*/
/*FUNCTION DofVec::SetValuesFromVecAndDofList(double* serial_vector,int vector_size,int requested_numdofs,int* requested_doflist) {{{1*/
void DofVec::SetValuesFromVecAndDofList(double* serial_vector,int vector_size,int requested_numdofs,int* requested_doflist){

	double* dofset=NULL;
	Vec     vector=NULL;

	/*First thing, is the vector of the right size? :*/
	if (vector_size!=requested_numdofs*this->numentries)ISSMERROR("%s%i%s%i"," error message: input vector size ",vector_size," is not equal to the expected numdofs*numentries",requested_numdofs*this->numentries);

	/*Create dofset: */
	dofset=dofsetgen(requested_numdofs,requested_doflist,this->numdofs,this->numdofs*this->numentries);

	/*Make a parallel vector out of the serial input vector: */
	vector=SerialToVec(serial_vector,vector_size);
	
	/*Merge vector into this->vector: */
	VecMerge(this->vector,vector,dofset,vector_size);

	/*Free ressources:*/
	xfree((void**)&dofset);
	VecFree(&vector);
}
/*}}}*/
/*FUNCTION DofVec::SetValuesFromVecAndDofList(Vec vector,int requested_numdofs,int* requested_doflist) {{{1*/
void DofVec::SetValuesFromVecAndDofList(Vec vector,int requested_numdofs,int* requested_doflist){

	double* dofset=NULL;
	int    vector_size;

	VecGetSize(vector,&vector_size);

	/*First thing, is the vector of the right size? :*/
	if (vector_size!=requested_numdofs*this->numentries)ISSMERROR("%s%i%s%i"," error message: input vector size ",vector_size," is not equal to the expected numdofs*numentries",requested_numdofs*this->numentries);

	/*Create dofset: */
	dofset=dofsetgen(requested_numdofs,requested_doflist,this->numdofs,this->numdofs*this->numentries);

	/*Merge vector into this->vector: */
	VecMerge(this->vector,vector,dofset,vector_size);

	/*Free ressources:*/
	xfree((void**)&dofset);
}
/*}}}*/
/*FUNCTION DofVec::SetValuesFromVecAndDofList(DofVec* vector,int requested_numdofs,int* requested_doflist) {{{1*/
void DofVec::SetValuesFromVecAndDofList(DofVec* vector,int requested_numdofs,int* requested_doflist){

	double* dofset=NULL;
	int    vector_size;

	vector_size=vector->Size();

	/*First thing, is the vector of the right size? :*/
	if (vector_size!=requested_numdofs*this->numentries)ISSMERROR("%s%i%s%i"," error message: input vector size ",vector_size," is not equal to the expected numdofs*numentries",requested_numdofs*this->numentries);

	/*Create dofset: */
	dofset=dofsetgen(requested_numdofs,requested_doflist,this->numdofs,this->numdofs*this->numentries);

	/*Merge vector into this->vector: */
	VecMerge(this->vector,vector->GetVec(),dofset,vector_size);

	/*Free ressources:*/
	xfree((void**)&dofset);
}
/*}}}*/
/*FUNCTION DofVec::GetValuesFromIndex {{{1*/
double* DofVec::GetValuesFromIndex(int index_size,int* index){

	double* values=NULL;
	double* serial_vector=NULL;
	int i;

	/*Serialize vector first, then allocate a buffer and fill it with the correct 
	 * values: */
	VecToMPISerial(&serial_vector,this->vector);

	values=(double*)xmalloc(index_size*sizeof(double));
	for(i=0;i<index_size;i++){
		values[i]=serial_vector[index[i]];
	}

	/*Free ressources:*/
	xfree((void**)&serial_vector);

	return values;
}
/*}}}*/
/*FUNCTION DofVec::MyRank {{{1*/
int    DofVec::MyRank(void){ 

	extern int my_rank;
	return my_rank; 
}

/*FUNCTION DofVec::Size {{{1*/
int DofVec::Size(){
	return this->numdofs*this->numentries;
}
/*}}}*/
