/*!\file Tria.c
 * \brief: implementation of the Tria object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./TriaVertexInput.h"
#include "./Object.h"
#include "./Hook.h"
#include "./Tria.h"
#include "./Hook.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../DataSet/Inputs.h"
#include "../include/typedefs.h"
#include "../include/macros.h"
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Tria::Tria(){{{1*/
Tria::Tria(){
	this->inputs=NULL;
	this->parameters=NULL;
}
/*}}}*/
/*FUNCTION Tria::Tria(int id, int* node_ids, int matice_id, int matpar_id){{{1*/
Tria::Tria(int tria_id,int* tria_node_ids, int tria_matice_id, int tria_matpar_id): 
	hnodes(tria_node_ids,3),
	hmatice(&tria_matice_id,1),
	hmatpar(&tria_matpar_id,1)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=tria_id;
	this->parameters=NULL;
	this->inputs=new Inputs();

}
/*}}}*/
/*FUNCTION Tria::Tria(int id, Hook* hnodes, Hook* hmatice, Hook* hmatpar, Parameters* parameters, Inputs* tria_inputs) {{{1*/
Tria::Tria(int tria_id,Hook* tria_hnodes, Hook* tria_hmatice, Hook* tria_hmatpar, Parameters* tria_parameters, Inputs* tria_inputs):
	hnodes(tria_hnodes),
	hmatice(tria_hmatice),
	hmatpar(tria_hmatpar)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=tria_id;
	if(tria_inputs){
		this->inputs=(Inputs*)tria_inputs->Copy();
	}
	else{
		this->inputs=new Inputs();
	}
	/*point parameters: */
	this->parameters=tria_parameters;
}
/*}}}*/
/*FUNCTION Tria::Tria(int id, int index, IoModel* iomodel){{{1*/
Tria::Tria(int tria_id, int index, IoModel* iomodel){ //i is the element index

	int    i;
	int    j;
	int    tria_node_ids[3];
	int    tria_matice_id;
	int    tria_matpar_id;
	double nodeinputs[3];

	/*id: */
	this->id=tria_id;
	
	/*hooks: */
	//go recover node ids, needed to initialize the node hook.
	if (iomodel->analysis_type==Prognostic2AnalysisEnum || iomodel->analysis_type==Balancedthickness2AnalysisEnum){
		/*Discontinuous Galerkin*/
		tria_node_ids[0]=3*index+1;
		tria_node_ids[1]=3*index+2;
		tria_node_ids[2]=3*index+3;
	}
	else{
		/*Continuous Galerkin*/
		for(i=0;i<3;i++){ 
			tria_node_ids[i]=(int)*(iomodel->elements+3*index+i); //ids for vertices are in the elements array from Matlab
		}
	}
	tria_matice_id=index+1; //refers to the corresponding ice material object
	tria_matpar_id=iomodel->numberofelements+1; //refers to the constant material parameters object

	this->hnodes.Init(tria_node_ids,3);
	this->hmatice.Init(&tria_matice_id,1);
	this->hmatpar.Init(&tria_matpar_id,1);

	//intialize inputs, and add as many inputs per element as requested: 
	this->inputs=new Inputs();
	
	if (iomodel->thickness) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->thickness[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->surface[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(SurfaceEnum,nodeinputs));
	}
	if (iomodel->bed) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->bed[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(BedEnum,nodeinputs));
	}
	if (iomodel->drag_coefficient) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->drag_coefficient[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(DragCoefficientEnum,nodeinputs));

		if (iomodel->drag_p) this->inputs->AddInput(new DoubleInput(DragPEnum,iomodel->drag_p[index]));
		if (iomodel->drag_q) this->inputs->AddInput(new DoubleInput(DragQEnum,iomodel->drag_q[index]));
		this->inputs->AddInput(new IntInput(DragTypeEnum,iomodel->drag_type));

	}
	if (iomodel->melting_rate) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->melting_rate[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(MeltingRateEnum,nodeinputs));
	}
	if (iomodel->accumulation_rate) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->accumulation_rate[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(AccumulationRateEnum,nodeinputs));
	}
	if (iomodel->geothermalflux) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->geothermalflux[tria_node_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(GeothermalFluxEnum,nodeinputs));
	}	

	if (iomodel->elementoniceshelf) this->inputs->AddInput(new BoolInput(ElementOnIceShelfEnum,(IssmBool)iomodel->elementoniceshelf[index]));
	if (iomodel->elementonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->elementonbed[index]));
	if (iomodel->elementonwater) this->inputs->AddInput(new BoolInput(ElementOnWaterEnum,(IssmBool)iomodel->elementonwater[index]));
	if (iomodel->elementonsurface) this->inputs->AddInput(new BoolInput(ElementOnSurfaceEnum,(IssmBool)iomodel->elementonsurface[index]));

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;


}
/*}}}*/
/*FUNCTION Tria::~Tria(){{{1*/
Tria::~Tria(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object management: */
/*FUNCTION Tria::Configure {{{1*/
void  Tria::Configure(DataSet* loadsin, DataSet* nodesin, DataSet* materialsin, Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	hmatice.configure(materialsin);
	hmatpar.configure(materialsin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Tria::copy {{{1*/
Object* Tria::copy() {

	return new Tria(this->id,&this->hnodes,&this->hmatice,&this->hmatpar,this->parameters,this->inputs);

}

/*}}}*/
/*FUNCTION Tria::Demarshall {{{1*/
void  Tria::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	hmatice.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Tria::DeepEcho{{{1*/

void Tria::DeepEcho(void){

	printf("Tria:\n");
	printf("   id: %i\n",id);
	hnodes.DeepEcho();
	hmatice.DeepEcho();
	hmatpar.DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
	
	return;
}
/*}}}*/
/*FUNCTION Tria::Echo{{{1*/

void Tria::Echo(void){

	printf("Tria:\n");
	printf("   id: %i\n",id);
	hnodes.Echo();
	hmatice.Echo();
	hmatpar.Echo();
	printf("   parameters\n");
	parameters->Echo();
	printf("   inputs\n");
	inputs->Echo();
}
/*}}}*/
/*FUNCTION Tria::Marshall {{{1*/
void  Tria::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Tria: */
	enum_type=TriaEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Tria data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);

	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	hmatice.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Tria::MarshallSize {{{1*/
int   Tria::MarshallSize(){
	
	return sizeof(id)
		+hnodes.MarshallSize()
		+hmatice.MarshallSize()
		+hmatpar.MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/

/*Updates: */
/*FUNCTION Tria::UpdateFromDakota {{{1*/
void  Tria::UpdateFromDakota(void* vinputs){

	int     i;
	int     dofs[1]={0};
	double  temperature_list[3];
	double  temperature_average;
	double  B_list[3];
	double  B_average;
	double  new_h[3];

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*Update internal data if inputs holds new values: */
	/*inputs->Recover("thickness",&this->properties.h[0],1,dofs,3,(void**)nodes);
	if(inputs->Recover("thickness",&new_h[0],1,dofs,3,(void**)nodes)){
	//density, needed later:
	double di=(this->matpar->GetRhoIce()/this->matpar->GetRhoWater());
	//Go through grids:
	for (i=0;i<3;i++){
	if(nodes[i]->IsOnShelf()){
	this->b[i]=this->b[i]-di*(new_h[i]-h[i]); //hydrostatic equilibrium;
	}
	this->s[i]=this->b[i]+new_h[i];
	this->h[i]=new_h[i];
	}
	}*/

	ISSMERROR("not supported yet!");

}
/*}}}*/
/*FUNCTION Tria::UpdateInputs {{{1*/
void  Tria::UpdateInputs(double* solution, int analysis_type, int sub_analysis_type){

	/*Just branch to the correct UpdateInputs generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum){
		
		UpdateInputsDiagnosticHoriz( solution,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==DiagnosticAnalysisEnum){
	
		if (sub_analysis_type==HorizAnalysisEnum){

			UpdateInputsDiagnosticHoriz( solution,analysis_type,sub_analysis_type);
		}
		else ISSMERROR("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet");

	}
	else if (analysis_type==SlopecomputeAnalysisEnum){

		UpdateInputsSlopeCompute( solution,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==PrognosticAnalysisEnum){

		UpdateInputsPrognostic( solution,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==Prognostic2AnalysisEnum){

		UpdateInputsPrognostic2(solution,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){

		UpdateInputsBalancedthickness( solution,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==Balancedthickness2AnalysisEnum){

		UpdateInputsBalancedthickness2( solution,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){

		UpdateInputsBalancedvelocities( solution,analysis_type,sub_analysis_type);
	}
	else{

		ISSMERROR("%s%i%s\n","analysis: ",analysis_type," not supported yet");
	}
}
/*}}}*/
/*FUNCTION Tria::UpdateInputsDiagnosticHoriz {{{1*/
void  Tria::UpdateInputsDiagnosticHoriz(double* solution, int analysis_type, int sub_analysis_type){
	
	
	int i;

	const int    numvertices=3;
	const int    numdofpervertex=2;
	const int    numdof=numdofpervertex*numvertices;
	
	int          doflist[numdof];
	double       values[numdof];
	double       vx[numvertices];
	double       vy[numvertices];

	int          dummy;
	
	/*Get dof list: */
	GetDofList(&doflist[0],&dummy);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++){
		values[i]=solution[doflist[i]];
	}

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numvertices;i++){
		vx[i]=values[i*numdofpervertex+0];
		vy[i]=values[i*numdofpervertex+1];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new TriaVertexInput(VyEnum,vy));
}

/*}}}*/
/*FUNCTION Tria::UpdateInputsSlopeCompute {{{1*/
void  Tria::UpdateInputsSlopeCompute(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::UpdateInputsPrognostic {{{1*/
void  Tria::UpdateInputsPrognostic(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::UpdateInputsPrognostic2 {{{1*/
void  Tria::UpdateInputsPrognostic2(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::UpdateInputsBalancedthickness {{{1*/
void  Tria::UpdateInputsBalancedthickness(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::UpdateInputsBalancedthickness2 {{{1*/
void  Tria::UpdateInputsBalancedthickness2(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::UpdateInputsBalancedvelocities {{{1*/
void  Tria::UpdateInputsBalancedvelocities(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Object functions*/
/*FUNCTION Tria::ComputeBasalStress {{{1*/
void  Tria::ComputeBasalStress(Vec eps,int analysis_type,int sub_analysis_type){

	int i;
	const int numgrids=3;
	int doflist[numgrids];
	double value;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*plug local pressure values into global pressure vector: */
	ISSMERROR("Not Implemented yet");
	//VecSetValues(eps,1,2,(const double*)&value,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::ComputePressure {{{1*/
void  Tria::ComputePressure(Vec pg,int analysis_type,int sub_analysis_type){

	int i;
	const int numvertices=3;
	int doflist[numvertices];
	double pressure[numvertices];
	double thickness[numvertices];
	double rho_ice,g;
	double gauss[numvertices][numvertices]={{1,0,0},{0,1,0},{0,0,1}};
	
	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();

	/*recover value of thickness at gauss points (0,0,1), (0,1,0),(1,0,0): */
	inputs->GetParameterValues(&thickness[0],&gauss[0][0],3,ThicknessEnum);

	for(i=0;i<numvertices;i++){
		pressure[i]=rho_ice*g*thickness[i];
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(pg,numvertices,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::ComputeStrainRate {{{1*/
void  Tria::ComputeStrainRate(Vec eps,int analysis_type,int sub_analysis_type){

	int i;
	const int numgrids=3;
	int doflist[numgrids];
	double value;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=   NULL;
	Matpar* matpar    =NULL;
	Matice* matice= NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*plug local pressure values into global pressure vector: */
	ISSMERROR("Not Implemented yet");
	//VecSetValues(eps,1,2,(const double*)&value,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Tria::CostFunction {{{1*/
double Tria::CostFunction(int analysis_type,int sub_analysis_type){

	int i;

	/* output: */
	double Jelem;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	int          dofs1[1]={0};
	int          dofs2[2]={0,1};
	double       xyz_list[numgrids][3];

	/* grid data: */
	double B[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  k_gauss;
	double  B_gauss;

	/* parameters: */
	double  dk[NDOF2]; 
	double  dB[NDOF2]; 
	char*   control_type=NULL;
	double  cm_noisedmp;

	/* Jacobian: */
	double Jdet;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*inputs: */
	bool onwater;
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&control_type,ControlTypeEnum);
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/*First, get Misfit*/
	Jelem=Misfit(analysis_type,sub_analysis_type);

	  /* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	  GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Add Tikhonov regularization term to misfit*/
		if (strcmp(control_type,"drag")==0){
			if (shelf){
				inputs->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],&gauss_l1l2l3[0],DragCoefficientEnum);
				Jelem+=cm_noisedmp*1/2*(pow(dk[0],2)+pow(dk[1],2))*Jdet*gauss_weight;

			}
		}
		else if (strcmp(control_type,"B")==0){
			inputs->GetParameterDerivativeValue(&dB[0], &xyz_list[0][0], &gauss_l1l2l3[0],RheologyBEnum);
			Jelem+=cm_noisedmp*1/2*(pow(dB[0],2)+pow(dB[1],2))*Jdet*gauss_weight;
		}
		else{
			ISSMERROR("%s%s","unsupported control type: ",control_type);
		}

	}

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
	xfree((void**)&control_type);

	/*Return: */
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrix {{{1*/

void  Tria::CreateKMatrix(Mat Kgg,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum){
		
		CreateKMatrixDiagnosticHoriz( Kgg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==DiagnosticAnalysisEnum){
	
		if (sub_analysis_type==HorizAnalysisEnum){

			CreateKMatrixDiagnosticHoriz( Kgg,analysis_type,sub_analysis_type);
		}
		else ISSMERROR("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet");

	}
	else if (analysis_type==SlopecomputeAnalysisEnum){

		CreateKMatrixSlopeCompute( Kgg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==PrognosticAnalysisEnum){

		CreateKMatrixPrognostic( Kgg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==Prognostic2AnalysisEnum){

		CreateKMatrixPrognostic2(Kgg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){

		CreateKMatrixBalancedthickness( Kgg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==Balancedthickness2AnalysisEnum){

		CreateKMatrixBalancedthickness2( Kgg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){

		CreateKMatrixBalancedvelocities( Kgg,analysis_type,sub_analysis_type);
	}
	else{

		ISSMERROR("%s%i%s\n","analysis: ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedthickness {{{1*/
void  Tria::CreateKMatrixBalancedthickness(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};//local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_thickness1[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_thickness2[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.

	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  vx_list[numgrids];
	double  vy_list[numgrids];
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};


	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	int     dofs[2]={0,1};
	int     found=0;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*parameters: */
	double artdiff;


	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Recover velocity: */
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&artdiff,ArtDiffEnum);

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=ONETHIRD; gauss_l1l2l3[1]=ONETHIRD; gauss_l1l2l3[2]=ONETHIRD;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		v_gauss[0]=ONETHIRD*(vx_list[0]+vx_list[1]+vx_list[2]);
		v_gauss[1]=ONETHIRD*(vy_list[0]+vy_list[1]+vy_list[2]);

		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		this->GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		this->GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);

		this->GetParameterDerivativeValue(&dvx[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3);
		this->GetParameterDerivativeValue(&dvy[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=gauss_weight*Jdettria;

		//Create DL and DLprime matrix
		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		//Ke_gg_thickness=B'*DLprime*Bprime;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&B[0][0],2,numdof,0,
					&Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_thickness2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness2[i][j];

		if(artdiff){

			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		}
	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedthickness2 {{{1*/
void  Tria::CreateKMatrixBalancedthickness2(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg2[numdof][numdof]={0.0};
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  vx,vy;
	int     dofs[1]={0};
	int     found;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;


	/*recover objects from hooks: */
	nodes=(Node**)   hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get B  and B prime matrix: */
		/*WARNING: B and Bprime are inverted compared to usual prognostic!!!!*/
		GetB_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		inputs->GetParameterValue(&vx, &gauss_l1l2l3[0],VxEnum);
		inputs->GetParameterValue(&vy, &gauss_l1l2l3[0],VyEnum);

		DL_scalar=-gauss_weight*Jdettria;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg2[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedvelocities {{{1*/
void  Tria::CreateKMatrixBalancedvelocities(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};//local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_velocities1[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_velocities2[numdof][numdof]={0.0}; //stiffness matrix evaluated at the gaussian point.
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  surface_normal[3];
	double  surface_list[3];
	double  nx,ny,norm;
	double  vx_list[numgrids]={0.0};
	double  vy_list[numgrids]={0.0};
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};
	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	int     dofs[2]={0,1};
	int     found=0;

	/*parameters: */
	double artdiff;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*Recover velocity: */
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&artdiff,ArtDiffEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Modify z so that it reflects the surface*/
	inputs->GetParameterValues(&surface_list[0],&gaussgrids[0][0],3,SurfaceEnum);
	for(i=0;i<numgrids;i++) xyz_list[i][2]=surface_list[i];

	/*Get normal vector to the surface*/
	nx=(vx_list[0]+vx_list[1]+vx_list[2])/3;
	ny=(vy_list[0]+vy_list[1]+vy_list[2])/3;
	if(nx==0 && ny==0){
		SurfaceNormal(&surface_normal[0],xyz_list);
		nx=surface_normal[0];
		ny=surface_normal[1];
	}
	if(nx==0 && ny==0){
		nx=0;
		ny=1;
	}
	norm=pow( pow(nx,2)+pow(ny,2) , (double).5);
	nx=nx/norm;
	ny=ny/norm;

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=ONETHIRD; gauss_l1l2l3[1]=ONETHIRD; gauss_l1l2l3[2]=ONETHIRD;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		v_gauss[0]=ONETHIRD*(vx_list[0]+vx_list[1]+vx_list[2]);
		v_gauss[1]=ONETHIRD*(vy_list[0]+vy_list[1]+vy_list[2]);

		K[0][0]=pow(10,2)*pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]); //pow should be zero!!
		K[1][1]=pow(10,2)*pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);

		GetParameterDerivativeValue(&dvx[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterDerivativeValue(&dvy[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=gauss_weight*Jdettria;

		DLprime[0][0]=DL_scalar*nx;
		DLprime[1][1]=DL_scalar*ny;

		//Do the triple product tL*D*L. 
		//Ke_gg_velocities=B'*DLprime*Bprime;
		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_velocities2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_velocities2[i][j];

		if(artdiff){

			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		}

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticHoriz {{{1*/
void  Tria::CreateKMatrixDiagnosticHoriz(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* material data: */
	double viscosity; //viscosity
	double newviscosity; //viscosity
	double oldviscosity; //viscosity

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/
	double oldepsilon[3]; /* oldepsilon=[exx,eyy,exy];*/

	/* matrices: */
	double B[3][numdof];
	double Bprime[3][numdof];
	double D[3][3]={0.0};  // material matrix, simple scalar matrix.
	double D_scalar;

	/*parameters: */
	double viscosity_overshoot;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.

	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double  thickness;
	int     dofs[2]={0,1};

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*inputs: */
	bool onwater,shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

	/*First, if we are on water, return empty matrix: */
	if(onwater) return;

	/*recover objects from hooks: */
	nodes =(Node**) hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);


		/*Compute thickness at gaussian point: */
		inputs->GetParameterValue(&thickness, gauss_l1l2l3,ThicknessEnum);

		/*Get strain rate from velocity: */
		inputs->GetStrainRate(&epsilon[0],&xyz_list[0][0],gauss_l1l2l3,VxEnum,VyEnum);
		inputs->GetStrainRate(&oldepsilon[0],&xyz_list[0][0],gauss_l1l2l3,VxOldEnum,VyOldEnum);

		/*Get viscosity: */
		matice->GetViscosity2d(&viscosity, &epsilon[0]);
		matice->GetViscosity2d(&oldviscosity, &oldepsilon[0]);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
			onto this scalar matrix, so that we win some computational time: */
		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=newviscosity*thickness*gauss_weight*Jdet;

		for (i=0;i<3;i++){
			D[i][i]=D_scalar;
		}

		/*Get B and Bprime matrices: */
		GetB(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		/*  Do the triple product tB*D*Bprime: */
		TripleMultiply( &B[0][0],3,numdof,1,
					&D[0][0],3,3,0,
					&Bprime[0][0],3,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	/*Do not forget to include friction: */
	if(!shelf){
		CreateKMatrixDiagnosticHorizFriction(Kgg,analysis_type,sub_analysis_type);
	}

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticHorizFriction {{{1*/
void  Tria::CreateKMatrixDiagnosticHorizFriction(Mat Kgg,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* matrices: */
	double L[2][numdof];
	double DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	
	double Jdet;
	
	/*slope: */
	double  slope[2]={0.0,0.0};
	double  slope_magnitude;

	/*input parameters for structural analysis (diagnostic): */
	double  vx_list[numgrids];
	double  vy_list[numgrids];
	double  thickness_list[numgrids];
	double  bed_list[numgrids];
	double  dragcoefficient_list[numgrids];
	double  drag_p,drag_q;

	/*friction: */
	double alpha2_list[numgrids]={0.0,0.0,0.0};
	double alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	/*inputs: */
	bool shelf;
	int  drag_type;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;


	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	if (shelf){
		/*no friction, do nothing*/
		return;
	}

	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");

	/*Recover inputs: */
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);
	inputs->GetParameterValues(&dragcoefficient_list[0],&gaussgrids[0][0],3,DragCoefficientEnum);
	inputs->GetParameterValues(&bed_list[0],&gaussgrids[0][0],3,BedEnum);
	inputs->GetParameterValues(&thickness_list[0],&gaussgrids[0][0],3,ThicknessEnum);
	inputs->GetParameterValue(&drag_p,DragPEnum);
	inputs->GetParameterValue(&drag_q,DragQEnum);

	/*Build alpha2_list used by drag stiffness matrix*/
	Friction* friction=NewFriction();
	
	/*Initialize all fields: */
	friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
	strcpy(friction->element_type,"2d");
	
	friction->gravity=matpar->GetG();
	friction->rho_ice=matpar->GetRhoIce();
	friction->rho_water=matpar->GetRhoWater();
	friction->K=&dragcoefficient_list[0];
	friction->bed=&bed_list[0];
	friction->thickness=&thickness_list[0];
	friction->vx=&vx_list[0];
	friction->vy=&vy_list[0];
	friction->p=drag_p;
	friction->q=drag_q;

	/*Compute alpha2_list: */
	FrictionGetAlpha2(&alpha2_list[0],friction);

	/*Erase friction object: */
	DeleteFriction(&friction);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);


		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		inputs->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0],SurfaceEnum);
		slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));

		if (slope_magnitude>MAXSLOPE){
			alpha2_list[0]=pow((double)10,MOUNTAINKEXPONENT);
			alpha2_list[1]=pow((double)10,MOUNTAINKEXPONENT);
			alpha2_list[2]=pow((double)10,MOUNTAINKEXPONENT);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/*Now, take care of the basal friction if there is any: */
		GetParameterValue(&alpha2, &alpha2_list[0],gauss_l1l2l3);

		DL_scalar=alpha2*gauss_weight*Jdet;
		for (i=0;i<2;i++){
			DL[i][i]=DL_scalar;
		}
		
		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&L[0][0],2,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}	
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticSurfaceVert {{{1*/
void  Tria::CreateKMatrixDiagnosticSurfaceVert(Mat Kgg,int analysis_type,int sub_analysis_type){

	int i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];


	/* surface normal: */
	double x4,y4,z4;
	double x5,y5,z5;
	double x6,y6,z6;
	double v46[3];
	double v56[3];
	double normal[3];
	double norm_normal;
	double nz;

	/*Matrices: */
	double DL_scalar;
	double L[3];
	double Jdet;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*Build normal vector to the surface:*/

	x4=xyz_list[0][0];
	y4=xyz_list[0][1];
	z4=xyz_list[0][2];

	x5=xyz_list[1][0];
	y5=xyz_list[1][1];
	z5=xyz_list[1][2];

	x6=xyz_list[2][0];
	y6=xyz_list[2][1];
	z6=xyz_list[2][2];

	v46[0]=x4-x6;
	v46[1]=y4-y6;
	v46[2]=z4-z6;

	v56[0]=x5-x6;
	v56[1]=y5-y6;
	v56[2]=z5-z6;

	normal[0]=(y4-y6)*(z5-z6)-(z4-z6)*(y5-y6);
	normal[1]=(z4-z6)*(x5-x6)-(x4-x6)*(z5-z6);
	normal[2]=(x4-x6)*(y5-y6)-(y4-y6)*(x5-x6);

	norm_normal=sqrt(pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2));
	nz=1.0/norm_normal*normal[2];

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		//Get L matrix if viscous basal drag present:
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/**********************Do not forget the sign**********************************/
		DL_scalar=- gauss_weight*Jdet*nz; 
		/******************************************************************************/

		/*  Do the triple producte tL*D*L: */
		TripleMultiply( L,1,3,1,
					&DL_scalar,1,1,0,
					L,1,3,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];


	} //for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixMelting {{{1*/
void  Tria::CreateKMatrixMelting(Mat Kgg,int analysis_type,int sub_analysis_type){

	/*indexing: */
	int i,j;

	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;

	/*Grid data: */
	double     xyz_list[numgrids][3];

	/*Material constants */
	double     heatcapacity,latentheat;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double     Jdet;
	double     D_scalar;
	double     K_terms[numdof][numdof]={0.0};
	double     L[3];
	double     tLD[3];
	double     Ke_gaussian[numdof][numdof]={0.0};

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*Recover constants of ice */
	latentheat=matpar->GetLatentHeat();
	heatcapacity=matpar->GetHeatCapacity();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights: */
	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss  (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get L matrix : */
		GetL(&L[0], &xyz_list[0][0], gauss_coord,NDOF1);

		/*Calculate DL on gauss point */
		D_scalar=latentheat/heatcapacity*gauss_weight*Jdet;

		/*  Do the triple product tL*D*L: */
		MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
		MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<numgrids;i++){
			for(j=0;j<numgrids;j++){
				K_terms[i][j]+=Ke_gaussian[i][j];
			}
		}
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic {{{1*/
void  Tria::CreateKMatrixPrognostic(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof]={0.0};
	double Ke_gg_thickness1[numdof][numdof]={0.0};
	double Ke_gg_thickness2[numdof][numdof]={0.0};
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  vx_list[numgrids]={0.0};
	double  vy_list[numgrids]={0.0};
	double  dvx[2];
	double  dvy[2];
	double  vx,vy;
	double  dvxdx,dvydy;
	double  v_gauss[2]={0.0};
	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	int     dofs[2]={0,1};
	int     found;

	/*parameters: */
	double dt;
	double artdiff;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*Recover velocity: */
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);

	//Create Artificial diffusivity once for all if requested
	if(artdiff==1){
		//Get the Jacobian determinant
		gauss_l1l2l3[0]=ONETHIRD; gauss_l1l2l3[1]=ONETHIRD; gauss_l1l2l3[2]=ONETHIRD;
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		//Build K matrix (artificial diffusivity matrix)
		v_gauss[0]=ONETHIRD*(vx_list[0]+vx_list[1]+vx_list[2]);
		v_gauss[1]=ONETHIRD*(vy_list[0]+vy_list[1]+vy_list[2]);

		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		DL_scalar=gauss_weight*Jdettria;

		/*  Do the triple product tL*D*L: */
		TripleMultiply( &L[0],1,numdof,1,
					&DL_scalar,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/*Get B  and B prime matrix: */
		GetB_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		GetParameterValue(&vx, &vx_list[0],gauss_l1l2l3);
		GetParameterValue(&vy, &vy_list[0],gauss_l1l2l3);

		GetParameterDerivativeValue(&dvx[0], &vx_list[0],&xyz_list[0][0], gauss_l1l2l3);
		GetParameterDerivativeValue(&dvy[0], &vy_list[0],&xyz_list[0][0], gauss_l1l2l3);

		dvxdx=dvx[0];
		dvydy=dvy[1];

		DL_scalar=dt*gauss_weight*Jdettria;

		//Create DL and DLprime matrix
		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		//Ke_gg_thickness=B'*DL*B+B'*DLprime*Bprime;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&B[0][0],2,numdof,0,
					&Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_thickness2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_thickness2[i][j];

		if(artdiff==1){

			/* Compute artificial diffusivity */
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add artificial diffusivity matrix */
			for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

		}

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic2 {{{1*/
void  Tria::CreateKMatrixPrognostic2(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[numgrids];
	double B[2][numgrids];
	double Bprime[2][numgrids];
	double DL[2][2]={0.0};
	double DLprime[2][2]={0.0};
	double DL_scalar;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg1[numdof][numdof]={0.0};
	double Ke_gg2[numdof][numdof]={0.0};
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  vx,vy;
	int     dofs[1]={0};
	int     found;

	/*parameters: */
	double dt;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)   hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		DL_scalar=gauss_weight*Jdettria;

		/*  Do the triple product tL*D*L: */
		TripleMultiply( &L[0],1,numdof,1,
					&DL_scalar,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg1[0][0],0);

		/*Get B  and B prime matrix: */
		/*WARNING: B and Bprime are inverted compared to usual prognostic!!!!*/
		GetB_prog(&Bprime[0][0], &xyz_list[0][0], gauss_l1l2l3);
		GetBPrime_prog(&B[0][0], &xyz_list[0][0], gauss_l1l2l3);

		//Get vx, vy and their derivatives at gauss point
		inputs->GetParameterValue(&vx,&gauss_l1l2l3[0],VxAverageEnum);
		inputs->GetParameterValue(&vy,&gauss_l1l2l3[0],VyAverageEnum);

		DL_scalar=-dt*gauss_weight*Jdettria;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		//Do the triple product tL*D*L. 
		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg2[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);


cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixSlopeCompute {{{1*/

void  Tria::CreateKMatrixSlopeCompute(Mat Kgg,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrices: */
	double L[1][3];
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]={0.0}; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	
	double Jdet;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		
		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		DL_scalar=gauss_weight*Jdet;

		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],1,3,1,
					&DL_scalar,1,1,0,
					&L[0][0],1,3,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	} //for (ig=0; ig<num_gauss; ig++

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);
		
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixThermal {{{1*/
void  Tria::CreateKMatrixThermal(Mat Kgg,int analysis_type,int sub_analysis_type){

	int i,j;
	int found=0;
	
	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	double mixed_layer_capacity;
	double thermal_exchange_velocity;
	double rho_water;
	double rho_ice;
	double heatcapacity;

	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];

	/*matrices: */
	double  Jdet;
	double  K_terms[numdof][numdof]={0.0};
	double  Ke_gaussian[numdof][numdof]={0.0};
	double  l1l2l3[numgrids];
	double     tl1l2l3D[3];
	double  D_scalar;

	/*parameters: */
	double dt;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();


	GaussTria (&num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss (nodes on the bedrock) */
	for (ig=0; ig<num_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);
		
		//Get the Jacobian determinant
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss_coord);
		
		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);
				
		/*Calculate DL on gauss point */
		D_scalar=gauss_weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity/(heatcapacity*rho_ice);
		if(dt){
			D_scalar=dt*D_scalar;
		}

		/*  Do the triple product tL*D*L: */
		MatrixMultiply(&l1l2l3[0],numdof,1,0,&D_scalar,1,1,0,&tl1l2l3D[0],0);
		MatrixMultiply(&tl1l2l3D[0],numdof,1,0,&l1l2l3[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<3;i++){
			for(j=0;j<3;j++){
				K_terms[i][j]+=Ke_gaussian[i][j];
			}
		}
	}
	
	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	cleanup_and_return:
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVector {{{1*/
void  Tria::CreatePVector(Vec pg,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum){
		
		CreatePVectorDiagnosticHoriz( pg,analysis_type,sub_analysis_type);
	
	}
	else if (analysis_type==DiagnosticAnalysisEnum){
		if (sub_analysis_type==HorizAnalysisEnum){
		
			CreatePVectorDiagnosticHoriz( pg,analysis_type,sub_analysis_type);
		}
		else ISSMERROR("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet");
	}
	else if (analysis_type==SlopecomputeAnalysisEnum){
		
		CreatePVectorSlopeCompute( pg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==PrognosticAnalysisEnum){

		CreatePVectorPrognostic( pg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==Prognostic2AnalysisEnum){

		CreatePVectorPrognostic2( pg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum){

		CreatePVectorBalancedthickness( pg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==Balancedthickness2AnalysisEnum){

		CreatePVectorBalancedthickness2( pg,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum){

		CreatePVectorBalancedvelocities( pg,analysis_type,sub_analysis_type);
	}
	else{
		ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness {{{1*/
void  Tria::CreatePVectorBalancedthickness(Vec pg ,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting at gauss point */
		inputs->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0],AccumulationRateEnum);
		inputs->GetParameterValue(&melting_g, &gauss_l1l2l3[0],MeltingRateEnum);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(accumulation_g-melting_g)*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness2 {{{1*/
void  Tria::CreatePVectorBalancedthickness2(Vec pg ,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;
	double  dhdt_g;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)   hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		inputs->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0],AccumulationRateEnum);
		inputs->GetParameterValue(&melting_g, &gauss_l1l2l3[0],MeltingRateEnum);
		inputs->GetParameterValue(&dhdt_g, &gauss_l1l2l3[0],DhDtEnum);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(accumulation_g-melting_g+dhdt_g)*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedvelocities {{{1*/
void  Tria::CreatePVectorBalancedvelocities(Vec pg ,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting at gauss point */
		inputs->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0],AccumulationRateEnum);
		inputs->GetParameterValue(&melting_g, &gauss_l1l2l3[0],MeltingRateEnum);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(accumulation_g-melting_g)*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticBaseVert {{{1*/
void  Tria::CreatePVectorDiagnosticBaseVert(Vec pg,int analysis_type,int sub_analysis_type){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/* matrices: */
	double L[numgrids];

	/*input parameters for structural analysis (diagnostic): */
	double  vx,vy;
	double  meltingvalue;
	double  slope[2];
	double  dbdx,dbdy;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/*For icesheets: */
	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get melting at gaussian point: */
		inputs->GetParameterValue(&meltingvalue, &gauss_l1l2l3[0],MeltingRateEnum);

		/*Get velocity at gaussian point: */
		inputs->GetParameterValue(&vx, &gauss_l1l2l3[0],VxEnum);
		inputs->GetParameterValue(&vy, &gauss_l1l2l3[0],VyEnum);

		/*Get bed slope: */
		inputs->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0],BedEnum);
		dbdx=slope[0];
		dbdy=slope[1];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		//Get L matrix if viscous basal drag present:
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,NDOF1);


		/*Build gaussian vector: */
		for(i=0;i<numgrids;i++){
			pe_g_gaussian[i]=-Jdet*gauss_weight*(vx*dbdx+vy*dbdy-meltingvalue)*L[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticHoriz {{{1*/
void Tria::CreatePVectorDiagnosticHoriz( Vec pg,  int analysis_type,int sub_analysis_type){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* parameters: */
	double  plastic_stress; 
	double  slope[NDOF2];
	double  driving_stress_baseline;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];

	/*input parameters for structural analysis (diagnostic): */
	double  thickness;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*inputs: */
	bool onwater;
	int  drag_type;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);

	/*First, if we are on water, return empty vector: */
	if(onwater)return;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);


	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Compute thickness at gaussian point: */
		inputs->GetParameterValue(&thickness, &gauss_l1l2l3[0],ThicknessEnum);
		inputs->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0],SurfaceEnum);
		
		/*In case we have plastic basal drag, compute plastic stress at gaussian point from k1, k2 and k3 fields in the 
		 * element itself: */
		if(drag_type==1){
			inputs->GetParameterValue(&plastic_stress, &gauss_l1l2l3[0],DragCoefficientEnum);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Compute driving stress: */
		driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG()*thickness;

		/*Build pe_g_gaussian vector: */
		if(drag_type==1){
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=(-driving_stress_baseline*slope[j]-plastic_stress)*Jdet*gauss_weight*l1l2l3[i]; 
				}
			}
		}
		else {
			for (i=0;i<numgrids;i++){
				for (j=0;j<NDOF2;j++){
					pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l2l3[i];
				}
			}
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	} //for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic {{{1*/
void  Tria::CreatePVectorPrognostic(Vec pg ,int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;
	double  thickness_g;

	/*parameters: */
	double  dt;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		inputs->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0],AccumulationRateEnum);
		inputs->GetParameterValue(&melting_g, &gauss_l1l2l3[0],MeltingRateEnum);
		inputs->GetParameterValue(&thickness_g, &gauss_l1l2l3[0],ThicknessEnum);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic2 {{{1*/
void  Tria::CreatePVectorPrognostic2(Vec pg ,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* matrix */
	double pe_g[numgrids]={0.0};
	double L[numgrids];
	double Jdettria;

	/*input parameters for structural analysis (diagnostic): */
	double  accumulation_g;
	double  melting_g;
	double  thickness_g;
	double  dt;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss_l1l2l3);

		/*Get L matrix: */
		GetL(&L[0], &xyz_list[0][0], gauss_l1l2l3,numberofdofspernode);

		/* Get accumulation, melting and thickness at gauss point */
		inputs->GetParameterValue(&accumulation_g, &gauss_l1l2l3[0],AccumulationRateEnum);
		inputs->GetParameterValue(&melting_g, &gauss_l1l2l3[0],MeltingRateEnum);
		inputs->GetParameterValue(&thickness_g, &gauss_l1l2l3[0],ThicknessEnum);

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i]+=Jdettria*gauss_weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorSlopeCompute {{{1*/

void Tria::CreatePVectorSlopeCompute( Vec pg,  int analysis_type,int sub_analysis_type){

	int             i,j;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	
	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*element vector at the gaussian points: */
	double  pe_g[numdof]={0.0};
	double  pe_g_gaussian[numdof];
	double  slope[2];

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);


	/* Get gaussian points and weights: */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2); /*We need higher order because our load is order 2*/


	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		if ( (sub_analysis_type==SurfaceXAnalysisEnum) || (sub_analysis_type==SurfaceYAnalysisEnum)){
			inputs->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0],SurfaceEnum);
		}
		if ( (sub_analysis_type==BedXAnalysisEnum) || (sub_analysis_type==BedYAnalysisEnum)){
			inputs->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],&gauss_l1l2l3[0],BedEnum);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		 /*Get nodal functions: */
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build pe_g_gaussian vector: */
		if ( (sub_analysis_type==SurfaceXAnalysisEnum) || (sub_analysis_type==BedXAnalysisEnum)){
			for(i=0;i<numdof;i++) pe_g_gaussian[i]=Jdet*gauss_weight*slope[0]*l1l2l3[i];
		}
		if ( (sub_analysis_type==SurfaceYAnalysisEnum) || (sub_analysis_type==BedYAnalysisEnum)){
			for(i=0;i<numdof;i++) pe_g_gaussian[i]=Jdet*gauss_weight*slope[1]*l1l2l3[i];
		}

		/*Add pe_g_gaussian vector to pe_g: */
		for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

	} //for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorThermalShelf {{{1*/
void Tria::CreatePVectorThermalShelf( Vec pg,  int analysis_type,int sub_analysis_type){

	int i,found;
	
	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;
	double       xyz_list[numgrids][3];

	double mixed_layer_capacity;
	double thermal_exchange_velocity;
	double rho_water;
	double rho_ice;
	double heatcapacity;
	double beta;
	double meltingpoint;

	/*inputs: */
	double dt;
	double pressure;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];
	int     dofs1[1]={0};

	/*matrices: */
	double  Jdet;
	double  P_terms[numdof]={0.0};
	double  l1l2l3[numgrids];

	double  t_pmp;
	double  scalar_ocean;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();
	
	/*retrieve some solution parameters: */
	this->parameters->FindParam(&dt,DtEnum);

	/* Ice/ocean heat exchange flux on ice shelf base */

	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);

		/*Get geothermal flux and basal friction */
		inputs->GetParameterValue(&pressure, &gauss_coord[0],PressureEnum);
		t_pmp=meltingpoint-beta*pressure;

		/*Calculate scalar parameter*/
		scalar_ocean=gauss_weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity*(t_pmp)/(heatcapacity*rho_ice);
		if(dt){
			scalar_ocean=dt*scalar_ocean;
		}

		for(i=0;i<3;i++){
			P_terms[i]+=scalar_ocean*l1l2l3[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::CreatePVectorThermalSheet {{{1*/
void Tria::CreatePVectorThermalSheet( Vec pg,  int analysis_type,int sub_analysis_type){

	int i,found;
	
	const int  numgrids=3;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;
	double       xyz_list[numgrids][3];

	double rho_ice;
	double heatcapacity;

	/*inputs: */
	double dt;
	double pressure_list[3];
	double pressure;
	double alpha2_list[3];
	double basalfriction_list[3];
	double basalfriction;
	double geothermalflux_value;

	double  vx_list[numgrids];
	double  vy_list[numgrids];
	double  thickness_list[numgrids];
	double  bed_list[numgrids];
	double  dragcoefficient_list[numgrids];
	double  drag_p,drag_q;
	int     drag_type;

	/* gaussian points: */
	int     num_area_gauss,ig;
	double* gauss_weights  =  NULL;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double  gauss_weight;
	double  gauss_coord[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/*matrices: */
	double  Jdet;
	double  P_terms[numdof]={0.0};
	double  l1l2l3[numgrids];
	double  scalar;


	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve inputs :*/
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	//recover material parameters
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();

	/*retrieve some parameters: */
	this->parameters->FindParam(&dt,DtEnum);


	/*Recover inputs: */
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);
	inputs->GetParameterValues(&dragcoefficient_list[0],&gaussgrids[0][0],3,DragCoefficientEnum);
	inputs->GetParameterValues(&bed_list[0],&gaussgrids[0][0],3,BedEnum);
	inputs->GetParameterValues(&thickness_list[0],&gaussgrids[0][0],3,ThicknessEnum);
	inputs->GetParameterValue(&drag_p,DragPEnum);
	inputs->GetParameterValue(&drag_q,DragQEnum);

	/*Build alpha2_list used by drag stiffness matrix*/
	Friction* friction=NewFriction();
	
	/*Initialize all fields: */
	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");
	
	friction->element_type=(char*)xmalloc((strlen("3d")+1)*sizeof(char));
	strcpy(friction->element_type,"3d");

	friction->gravity=matpar->GetG();
	friction->rho_ice=matpar->GetRhoIce();
	friction->rho_water=matpar->GetRhoWater();
	friction->K=&dragcoefficient_list[0];
	friction->bed=&bed_list[0];
	friction->thickness=&thickness_list[0];
	friction->vx=&vx_list[0];
	friction->vy=&vy_list[0];
	friction->p=drag_p;
	friction->q=drag_q;

	/*Compute alpha2_list: */
	FrictionGetAlpha2(&alpha2_list[0],friction);

	/*Erase friction object: */
	DeleteFriction(&friction);

	/* Compute basal friction */
	for(i=0;i<numgrids;i++){
		basalfriction_list[i]= alpha2_list[i]*(pow(vx_list[i],(double)2.0)+pow(vy_list[i],(double)2.0));
	}
	
	/* Ice/ocean heat exchange flux on ice shelf base */
	GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	for (ig=0; ig<num_area_gauss; ig++){
		gauss_weight=*(gauss_weights+ig);
		gauss_coord[0]=*(first_gauss_area_coord+ig); 
		gauss_coord[1]=*(second_gauss_area_coord+ig);
		gauss_coord[2]=*(third_gauss_area_coord+ig);

		//Get the Jacobian determinant
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss_coord);

		/*Get nodal functions values: */
		GetNodalFunctions(&l1l2l3[0], gauss_coord);

		/*Get geothermal flux and basal friction */
		inputs->GetParameterValue(&geothermalflux_value, &gauss_coord[0],GeothermalFluxEnum);
		GetParameterValue(&basalfriction,&basalfriction_list[0],gauss_coord);

		/*Calculate scalar parameter*/
		scalar=gauss_weight*Jdet*(basalfriction+geothermalflux_value)/(heatcapacity*rho_ice);
		if(dt){
			scalar=dt*scalar;
		}

		for(i=0;i<3;i++){
			P_terms[i]+=scalar*l1l2l3[i];
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::Du {{{1*/
void Tria::Du(Vec du_g,int analysis_type,int sub_analysis_type){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double dux_list[numgrids];
	double duy_list[numgrids];
	double weights_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/* parameters: */
	double  obs_velocity_mag,velocity_mag;
	double  dux,duy;
	double  meanvel, epsvel;

	/*element vector : */
	double  due_g[numdof]={0.0};
	double  due_g_gaussian[numdof];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/*relative and algorithmic fitting: */
	double scalex=0;
	double scaley=0;
	double scale=0;
	double S=0;
	double fit=-1;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Recover input data: */
	inputs->GetParameterValues(&obs_vx_list[0],&gaussgrids[0][0],3,VxObsEnum);
	inputs->GetParameterValues(&obs_vy_list[0],&gaussgrids[0][0],3,VyObsEnum);

	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyEnum);
	
	inputs->GetParameterValues(&weights_list[0],&gaussgrids[0][0],3,WeightsEnum);
	
	inputs->GetParameterValue(&fit,FitEnum);
	if(fit==3){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}

	/*Get Du at the 3 nodes (integration of the linearized function)
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 *       d J                  dJ_i
	 * DU= - --- = sum_{i=1}^3  - ---  Phi_i = sum_{i=1}^3 DU_i Phi_i
	 *       d u                  du_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(fit==0){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 *        dJ             2
		 * DU = - -- = (u   - u )
		 *        du     obs
		 */
		for (i=0;i<numgrids;i++){
			dux_list[i]=obs_vx_list[i]-vx_list[i];
			duy_list[i]=obs_vy_list[i]-vy_list[i];
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 *
		 *        dJ     \bar{v}^2
		 * DU = - -- = ------------- (u   - u )
		 *        du   (u   + eps)^2    obs
		 *               obs
		 */
		for (i=0;i<numgrids;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			dux_list[i]=scalex*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scaley*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit:
		 *                        
		 *                 [        vel + eps     ] 2
		 * J = 4 \bar{v}^2 | log ( -----------  ) |  
		 *                 [       vel   + eps    ]
		 *                            obs
		 *
		 *        dJ                 2 * log(...)
		 * DU = - -- = - 4 \bar{v}^2 -------------  u
		 *        du                 vel^2 + eps
		 *            
		 */
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],2)+pow(vy_list[i],2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],2)+pow(obs_vy_list[i],2))+epsvel; //epsvel to avoid observed velocity being nil.
			scale=-8*pow(meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			dux_list[i]=scale*vx_list[i];
			duy_list[i]=scale*vy_list[i];
		}
	}
	else if(fit==3){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 *
		 *        dJ      1       1 
		 * DU = - -- = - --- ----------- * 2 (u - u   )
		 *        du      S  2 sqrt(...)           obs
		 */
		for (i=0;i<numgrids;i++){
			scale=1.0/(S*sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))+epsvel);
			dux_list[i]=scale*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scale*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(fit==4){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 *        dJ                              1      u                             1
		 * DU = - -- = - \bar{v}^2 log(u...) --------- ----  ~ - \bar{v}^2 log(u...) ------
		 *        du                         |u| + eps  |u|                           u + eps
		 */
		for (i=0;i<numgrids;i++){
			dux_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)) * 1/(vx_list[i]+epsvel));
			duy_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)) * 1/(vy_list[i]+epsvel));
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("%s%g","unsupported type of fit: ",fit);
	}

	/*Apply weights to DU*/
	for (i=0;i<numgrids;i++){
		dux_list[i]=weights_list[i]*dux_list[i];
		duy_list[i]=weights_list[i]*duy_list[i];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Build due_g_gaussian vector: we have three cases here, according to which type of misfit we are using. */

		/*Compute absolute(x/y) at gaussian point: */
		GetParameterValue(&dux, &dux_list[0],gauss_l1l2l3);
		GetParameterValue(&duy, &duy_list[0],gauss_l1l2l3);

		/*compute Du*/
		for (i=0;i<numgrids;i++){
			due_g_gaussian[i*NDOF2+0]=dux*Jdet*gauss_weight*l1l2l3[i]; 
			due_g_gaussian[i*NDOF2+1]=duy*Jdet*gauss_weight*l1l2l3[i]; 
		}

		/*Add due_g_gaussian vector to due_g: */
		for( i=0; i<numdof; i++){
			due_g[i]+=due_g_gaussian[i];
		}
	}

	/*Add due_g to global vector du_g: */
	VecSetValues(du_g,numdof,doflist,(const double*)due_g,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::Enum {{{1*/
int Tria::Enum(void){

	return TriaEnum;

}
/*}}}*/
/*FUNCTION Tria::GetArea {{{1*/
double Tria::GetArea(void){

	double area=0;
	const int    numgrids=3;
	double xyz_list[numgrids][3];
	double x1,y1,x2,y2,x3,y3;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];
 
	return x2*y3 - y2*x3 + x1*y2 - y1*x2 + x3*y1 - y3*x1;
}
/*}}}*/
/*FUNCTION Tria::GetAreaCoordinate {{{1*/
double Tria::GetAreaCoordinate(double x, double y, int which_one){

	double area=0;
	const int    numgrids=3;
	double xyz_list[numgrids][3];
	double x1,y1,x2,y2,x3,y3;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();

	/*Get area: */
	area=this->GetArea();

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];

	if(which_one==1){
		/*Get first area coordinate = det(x-x3  x2-x3 ; y-y3   y2-y3)/area*/
		return ((x-x3)*(y2-y3)-(x2-x3)*(y-y3))/area;
	}
	else if(which_one==2){
		/*Get second area coordinate = det(x1-x3  x-x3 ; y1-y3   y-y3)/area*/
		return ((x1-x3)*(y-y3)-(x-x3)*(y1-y3))/area;
	}
	else if(which_one==3){
		/*Get third  area coordinate 1-area1-area2: */
		return 1-((x-x3)*(y2-y3)-(x2-x3)*(y-y3))/area -((x1-x3)*(y-y3)-(x-x3)*(y1-y3))/area;
	}
	else ISSMERROR("%s%i%s\n"," error message: area coordinate ",which_one," done not exist!");
}
/*}}}*/
/*FUNCTION Tria::GetB {{{1*/

void Tria::GetB(double* B, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dh/dx    0    ]
	 *                [   0    dh/dy  ]
	 *                [ 1/2*dh/dy  1/2*dh/dx  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double dh1dh3[NDOF2][numgrids];


	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list, gauss_l1l2l3);

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh3[0][i]; //B[0][NDOF2*i]=dh1dh3[0][i];
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i)=0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh3[1][i];
		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh3[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh3[0][i]; 
	}
}
/*}}}*/
/*FUNCTION Tria::GetB_prog {{{1*/

void Tria::GetB_prog(double* B_prog, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B  matrix. B=[B1 B2 B3] where Bi is of size 3*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ h ]
	 *                [ h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B_prog has been allocated already, of size: 2x(NDOF1*numgrids)
	 */

	int i;
	const int NDOF1=1;
	const int numgrids=3;

	double l1l2l3[numgrids];


	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctions(&l1l2l3[0],gauss_l1l2l3);

	/*Build B_prog: */
	for (i=0;i<numgrids;i++){
		*(B_prog+NDOF1*numgrids*0+NDOF1*i)=l1l2l3[i];
		*(B_prog+NDOF1*numgrids*1+NDOF1*i)=l1l2l3[i];
	}
}
/*}}}*/
/*FUNCTION Tria::GetBedList {{{1*/
void  Tria::GetBedList(double* bedlist){
	
	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	
	inputs->GetParameterValues(bedlist,&gaussgrids[0][0],3,BedEnum);

}
/*}}}*/
/*FUNCTION Tria::GetBPrime {{{1*/

void Tria::GetBPrime(double* Bprime, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_prime=[ 2*dh/dx dh/dy ]
	 *                       [ dh/dx  2*dh/dy]
	 *                       [dh/dy dh/dx]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */
	
	int i;
	const int NDOF2=2;
	const int numgrids=3;

	/*Same thing in the actual coordinate system: */
	double dh1dh3[NDOF2][numgrids];


	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss_l1l2l3);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime+NDOF2*numgrids*0+NDOF2*i)=2*dh1dh3[0][i]; 
		*(Bprime+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh3[1][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i)=dh1dh3[0][i]; 
		*(Bprime+NDOF2*numgrids*1+NDOF2*i+1)=2*dh1dh3[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i)=dh1dh3[1][i]; 
		*(Bprime+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh3[0][i]; 
	}
}
/*}}}*/
/*FUNCTION Tria::GetBPrime_prog {{{1*/

void Tria::GetBPrime_prog(double* Bprime_prog, double* xyz_list, double* gauss_l1l2l3){

	/*Compute B'  matrix. B'=[B1' B2' B3'] where Bi' is of size 3*NDOF2. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_prime=[ dh/dx ]
	 *                       [ dh/dy ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B' has been allocated already, of size: 3x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF1=1;
	const int NDOF2=2;
	const int numgrids=3;

	/*Same thing in the actual coordinate system: */
	double dh1dh3[NDOF2][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list,gauss_l1l2l3);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime_prog+NDOF1*numgrids*0+NDOF1*i)=dh1dh3[0][i]; 
		*(Bprime_prog+NDOF1*numgrids*1+NDOF1*i)=dh1dh3[1][i]; 
	}
}
/*}}}*/
/*FUNCTION Tria::GetDofList {{{1*/
void  Tria::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	for(i=0;i<3;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Tria::GetDofList1 {{{1*/
void  Tria::GetDofList1(int* doflist){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	int i;
	for(i=0;i<3;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Tria::Id {{{1*/
int    Tria::Id(){ return id; }
/*}}}*/
/*FUNCTION Tria::GetJacobian {{{1*/
void Tria::GetJacobian(double* J, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	const int NDOF2=2;
	const int numgrids=3;
	double x1,y1,x2,y2,x3,y3;
	
	x1=*(xyz_list+numgrids*0+0);
	y1=*(xyz_list+numgrids*0+1);
	x2=*(xyz_list+numgrids*1+0);
	y2=*(xyz_list+numgrids*1+1);
	x3=*(xyz_list+numgrids*2+0);
	y3=*(xyz_list+numgrids*2+1);


	*(J+NDOF2*0+0)=0.5*(x2-x1);
	*(J+NDOF2*1+0)=SQRT3/6.0*(2*x3-x1-x2);
	*(J+NDOF2*0+1)=0.5*(y2-y1);
	*(J+NDOF2*1+1)=SQRT3/6.0*(2*y3-y1-y2);
}
/*}}}*/
/*FUNCTION Tria::GetJacobianDeterminant2d {{{1*/
void Tria::GetJacobianDeterminant2d(double*  Jdet, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3;
	
	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);


	*Jdet=SQRT3/6.0*((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1));


	if(Jdet<0){
		ISSMERROR("negative jacobian determinant!");
	}
	
}
/*}}}*/
/*FUNCTION Tria::GetJacobianDeterminant3d {{{1*/
void Tria::GetJacobianDeterminant3d(double*  Jdet, double* xyz_list,double* gauss_l1l2l3){

	/*The Jacobian determinant is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double x1,x2,x3,y1,y2,y3,z1,z2,z3;
	
	x1=*(xyz_list+3*0+0);
	y1=*(xyz_list+3*0+1);
	z1=*(xyz_list+3*0+2);
	x2=*(xyz_list+3*1+0);
	y2=*(xyz_list+3*1+1);
	z2=*(xyz_list+3*1+2);
	x3=*(xyz_list+3*2+0);
	y3=*(xyz_list+3*2+1);
	z3=*(xyz_list+3*2+2);


	*Jdet=SQRT3/6.0*pow(pow(((y2-y1)*(z3-z1)-(z2-z1)*(y3-y1)),2.0)+pow(((z2-z1)*(x3-x1)-(x2-x1)*(z3-z1)),2.0)+pow(((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1)),2.0),0.5);

	if(Jdet<0){
		ISSMERROR("negative jacobian determinant!");
	}
	
}
/*}}}*/
/*FUNCTION Tria::GetJacobianInvert {{{1*/
void Tria::GetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss_l1l2l3){

	double Jdet;
	const int NDOF2=2;
	const int numgrids=3;

	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(Jinv, xyz_list, gauss_l1l2l3);

	/*Invert Jacobian matrix: */
	MatrixInverse(Jinv,NDOF2,NDOF2,NULL,0,&Jdet);

}
/*}}}*/
/*FUNCTION Tria::GetL {{{1*/

void Tria::GetL(double* L, double* xyz_list, double* gauss_l1l2l3,int numdof){

	/*Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the actual coordinate system
	 * by: 
	 *       numdof=1: 
	 *       Li=h;
	 *       numdof=2:
	 *       Li=[ h    0    ]
	 *                [   0   h  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume L has been allocated already, of size: numgrids (numdof=1), or numdofx(numdof*numgrids) (numdof=2)
	 */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double l1l2l3[3];


	/*Get l1l2l3 in actual coordinate system: */
	GetNodalFunctions(l1l2l3, gauss_l1l2l3);

#ifdef _DELUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  h=%lf \n",i,l1l2l3[i]);
	}
#endif

	/*Build L: */
	if(numdof==1){
		for (i=0;i<numgrids;i++){
			L[i]=l1l2l3[i]; 
		}
	}
	else{
		for (i=0;i<numgrids;i++){
			*(L+numdof*numgrids*0+numdof*i)=l1l2l3[i]; //L[0][NDOF2*i]=dh1dh3[0][i];
			*(L+numdof*numgrids*0+numdof*i+1)=0;
			*(L+numdof*numgrids*1+numdof*i)=0;
			*(L+numdof*numgrids*1+numdof*i+1)=l1l2l3[i];
		}
	}
}
/*}}}*/
/*FUNCTION Tria::GetMatPar {{{1*/
void* Tria::GetMatPar(){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	return matpar;
}
/*}}}*/
/*FUNCTION Tria::GetNodalFunctions {{{1*/
void Tria::GetNodalFunctions(double* l1l2l3, double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2l3[0]=gauss_l1l2l3[0];

	/*Second nodal function: */
	l1l2l3[1]=gauss_l1l2l3[1];

	/*Third nodal function: */
	l1l2l3[2]=gauss_l1l2l3[2];

}
/*}}}*/
/*FUNCTION Tria::GetNodalFunctionsDerivatives {{{1*/
void Tria::GetNodalFunctionsDerivatives(double* dh1dh3,double* xyz_list, double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system: */

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double dh1dh3_ref[NDOF2][numgrids];
	double Jinv[NDOF2][NDOF2];


	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReference(&dh1dh3_ref[0][0], gauss_l1l2l3); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss_l1l2l3);

	/*Build dh1dh3: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh3+numgrids*0+i)=Jinv[0][0]*dh1dh3_ref[0][i]+Jinv[0][1]*dh1dh3_ref[1][i];
		*(dh1dh3+numgrids*1+i)=Jinv[1][0]*dh1dh3_ref[0][i]+Jinv[1][1]*dh1dh3_ref[1][i];
	}

}
/*}}}*/
/*FUNCTION Tria::GetNodalFunctionsDerivativesReference {{{1*/
void Tria::GetNodalFunctionsDerivativesReference(double* dl1dl3,double* gauss_l1l2l3){
	
	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	const int NDOF2=2;
	const int numgrids=3;

	/*First nodal function: */
	*(dl1dl3+numgrids*0+0)=-0.5; 
	*(dl1dl3+numgrids*1+0)=-1.0/(2.0*SQRT3);

	/*Second nodal function: */
	*(dl1dl3+numgrids*0+1)=0.5;
	*(dl1dl3+numgrids*1+1)=-1.0/(2.0*SQRT3);

	/*Third nodal function: */
	*(dl1dl3+numgrids*0+2)=0;
	*(dl1dl3+numgrids*1+2)=1.0/SQRT3;

}
/*}}}*/
/*FUNCTION Tria::GetNodes {{{1*/
void  Tria::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=NULL;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	
	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Tria::GetOnBed {{{1*/
bool Tria::GetOnBed(){
	
	bool onbed;
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);

	return onbed;
}
/*}}}*/
/*FUNCTION Tria::GetParameterDerivativeValue {{{1*/
void Tria::GetParameterDerivativeValue(double* p, double* plist,double* xyz_list, double* gauss_l1l2l3){
	 
	const int NDOF2=2;
	const int numgrids=3;
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter derivative value at gaussian 
	 * point specified by gauss_l1l2l3:
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *   dp/dx=plist[0]*dh1/dx+plist[1]*dh2/dx+plist[2]*dh3/dx
	 *
	 * p is a vector of size 2x1 already allocated.
	 */
	
	double dh1dh3[NDOF2][numgrids]; //nodal derivative functions in actual coordinate system.

	/*Get dh1dh2dh3 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh3[0][0],xyz_list, gauss_l1l2l3);

	*(p+0)=plist[0]*dh1dh3[0][0]+plist[1]*dh1dh3[0][1]+plist[2]*dh1dh3[0][2];
	*(p+1)=plist[0]*dh1dh3[1][0]+plist[1]*dh1dh3[1][1]+plist[2]*dh1dh3[1][2];

}
/*}}}*/
/*FUNCTION Tria::GetParameterValue {{{1*/
void Tria::GetParameterValue(double* pp, double* plist, double* gauss_l1l2l3){
	
	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter value at gaussian 
	 * point specifie by gauss_l1l2l3: */
	
	/*nodal functions: */
	double l1l2l3[3];

	/*output: */
	double p;

	GetNodalFunctions(l1l2l3, gauss_l1l2l3);

	p=l1l2l3[0]*plist[0]+l1l2l3[1]*plist[1]+l1l2l3[2]*plist[2];

	/*Assign output pointers:*/
	*pp=p;
}
/*}}}*/
/*FUNCTION Tria::GetShelf {{{1*/
bool   Tria::GetShelf(){
	/*inputs: */
	bool shelf;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	return shelf;
}
/*}}}*/
/*FUNCTION Tria::GetStrainRate {{{1*/
void Tria::GetStrainRate(double* epsilon, double* velocity, double* xyz_list, double* gauss_l1l2l3){

	int i;
	const int NDOF2=2;
	const int numgrids=3;

	double B[3][NDOF2*numgrids];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss_l1l2l3);

	/*Multiply B by velocity, to get strain rate: */
	MatrixMultiply( &B[0][0],3,NDOF2*numgrids,0,
			              velocity,NDOF2*numgrids,1,0,
						  epsilon,0);

}
/*}}}*/
/*FUNCTION Tria::GetThicknessList {{{1*/
void Tria::GetThicknessList(double* thickness_list){

	const int numgrids=3;
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};
	inputs->GetParameterValues(thickness_list,&gaussgrids[0][0],3,ThicknessEnum);

}
/*}}}*/
/*FUNCTION Tria::Gradj {{{1*/
void  Tria::Gradj(Vec grad_g,int analysis_type,int sub_analysis_type,char* control_type){

	/*inputs: */
	bool onwater;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, grad = 0: */
	if(onwater)return;

	if (strcmp(control_type,"drag")==0){
		GradjDrag( grad_g,analysis_type,sub_analysis_type);
	}
	else if (strcmp(control_type,"B")==0){
		GradjB( grad_g,analysis_type,sub_analysis_type);
	}
	else ISSMERROR("%s%s","control type not supported yet: ",control_type);
}
/*}}}*/
/*FUNCTION Tria::GradjB{{{1*/
void  Tria::GradjB(Vec grad_g,int analysis_type,int sub_analysis_type){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF1=1;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh3[NDOF2][numgrids];

	/* grid data: */
	double B[numgrids];

	int    dofs1[1]={0};
	int    dofs2[2]={0,1};

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];

	/*element vector at the gaussian points: */
	double  grade_g[numgrids]={0.0};
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/* parameters: */
	double  viscosity_complement;
	double  dvx[NDOF2];
	double  dvy[NDOF2]; 
	double  dadjx[NDOF2];
	double  dadjy[NDOF2];
	double  vx,vy;
	double  lambda,mu;
	double  thickness;
	int     dofs[1]={0};
	double  dB[NDOF2]; 
	double  B_gauss;
	
	/*parameters: */
	double  cm_noisedmp;
	double  cm_mindmp_slope;
	double  cm_mindmp_value;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	this->parameters->FindParam(&cm_mindmp_value,CmMinDmpValueEnum);
	this->parameters->FindParam(&cm_mindmp_slope,CmMinDmpSlopeEnum);
	this->parameters->FindParam(&cm_maxdmp_value,CmMaxDmpValueEnum);
	this->parameters->FindParam(&cm_maxdmp_slope,CmMaxDmpSlopeEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Get thickness: */
		inputs->GetParameterValue(&thickness, gauss_l1l2l3,ThicknessEnum);

		/*Get strain rate, if velocity has been supplied: */
		inputs->GetStrainRate(&epsilon[0],&xyz_list[0][0],gauss_l1l2l3,VxEnum,VyEnum);

		/*Get viscosity complement: */
		matice->GetViscosityComplement(&viscosity_complement, &epsilon[0]);

		/*Get dvx, dvy, dadjx and dadjx: */
		inputs->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],&gauss_l1l2l3[0],VxEnum);
		inputs->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],&gauss_l1l2l3[0],VyEnum);
		inputs->GetParameterDerivativeValue(&dadjx[0],&xyz_list[0][0],&gauss_l1l2l3[0],AdjointxEnum);
		inputs->GetParameterDerivativeValue(&dadjy[0],&xyz_list[0][0],&gauss_l1l2l3[0],AdjointyEnum);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get B derivative: dB/dx */
		inputs->GetParameterDerivativeValue(&dB[0],&xyz_list[0][0],&gauss_l1l2l3[0],RheologyBEnum);
		inputs->GetParameterValue(&B_gauss, gauss_l1l2l3,RheologyBEnum);

		/*Build gradje_g_gaussian vector (actually -dJ/dB): */
		for (i=0;i<numgrids;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=-viscosity_complement*thickness*( (2*dvx[0]+dvy[1])*2*dadjx[0]+(dvx[1]+dvy[0])*(dadjx[1]+dadjy[0])+(2*dvy[1]+dvx[0])*2*dadjy[1])*Jdet*gauss_weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]-=cm_noisedmp*Jdet*gauss_weight*(dh1dh3[0][i]*dB[0]+dh1dh3[1][i]*dB[1]);

			//min dampening
			if(B_gauss<cm_mindmp_value){ 
				grade_g_gaussian[i]+= cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(B_gauss>cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

		}

		/*Add grade_g_gaussian to grade_g: */
		for( i=0; i<numgrids;i++) grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector grad_g: */
	VecSetValues(grad_g,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Tria::GradjDrag {{{1*/
void  Tria::GradjDrag(Vec grad_g,int analysis_type,int sub_analysis_type){


	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh3[NDOF2][numgrids];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double adjx_list[numgrids];
	double adjy_list[numgrids];
	double thickness_list[numgrids];
	double bed_list[numgrids];
	double dragcoefficient_list[numgrids];
	double drag_p;
	double drag_q;
	int    drag_type;

	double drag;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* parameters: */
	double  dk[NDOF2]; 
	double  vx,vy;
	double  lambda,mu;
	double  bed,thickness,Neff;
	
	/*drag: */
	double alpha_complement_list[numgrids];
	double alpha_complement;

	/*element vector at the gaussian points: */
	double  grade_g[numgrids]={0.0};
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/*inputs: */
	bool shelf;

	/*parameters: */
	double  cm_noisedmp;
	double  cm_mindmp_slope;
	double  cm_mindmp_value;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	this->parameters->FindParam(&cm_mindmp_value,CmMinDmpValueEnum);
	this->parameters->FindParam(&cm_mindmp_slope,CmMinDmpSlopeEnum);
	this->parameters->FindParam(&cm_maxdmp_value,CmMaxDmpValueEnum);
	this->parameters->FindParam(&cm_maxdmp_slope,CmMaxDmpSlopeEnum);


	/*Get out if shelf*/
	if(shelf)return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/*Recover inputs: */
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);
	inputs->GetParameterValues(&thickness_list[0],&gaussgrids[0][0],3,ThicknessEnum);
	inputs->GetParameterValues(&bed_list[0],&gaussgrids[0][0],3,BedEnum);
	inputs->GetParameterValues(&dragcoefficient_list[0],&gaussgrids[0][0],3,DragCoefficientEnum);
	inputs->GetParameterValue(&drag_p,DragPEnum);
	inputs->GetParameterValue(&drag_q,DragQEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Build alpha_complement_list: */
		if (drag_type==2){
		
			/*Allocate friction object: */
			Friction* friction=NewFriction();
			
			/*Initialize all fields: */
			friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
			strcpy(friction->element_type,"2d");
			
			
			friction->gravity=matpar->GetG();
			friction->rho_ice=matpar->GetRhoIce();
			friction->rho_water=matpar->GetRhoWater();
			friction->K=&dragcoefficient_list[0];
			friction->bed=&bed_list[0];
			friction->thickness=&thickness_list[0];
			friction->vx=&vx_list[0];
			friction->vy=&vy_list[0];
			friction->p=drag_p;
			friction->q=drag_q;

			
			if(friction->p!=1) ISSMERROR("non-linear friction not supported yet in control methods!");

			/*Compute alpha complement: */
			FrictionGetAlphaComplement(&alpha_complement_list[0],friction);

			/*Erase friction object: */
			DeleteFriction(&friction);
		}
		else{
			alpha_complement_list[0]=0;
			alpha_complement_list[1]=0;
			alpha_complement_list[2]=0;
		}
	
		/*Recover alpha_complement and k: */
		GetParameterValue(&alpha_complement, &alpha_complement_list[0],gauss_l1l2l3);
		inputs->GetParameterValue(&drag, gauss_l1l2l3,DragCoefficientEnum);

		/*recover lambda and mu: */
		inputs->GetParameterValue(&lambda, gauss_l1l2l3,AdjointxEnum);
		inputs->GetParameterValue(&mu, gauss_l1l2l3,AdjointyEnum);
			
		/*recover vx and vy: */
		inputs->GetParameterValue(&vx, gauss_l1l2l3,VxEnum);
		inputs->GetParameterValue(&vy, gauss_l1l2l3,VyEnum);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);
		
		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get k derivative: dk/dx */
		inputs->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],&gauss_l1l2l3[0],DragCoefficientEnum);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<numgrids;i++){

			//standard term dJ/dki
			grade_g_gaussian[i]=-2*drag*alpha_complement*((lambda*vx+mu*vy))*Jdet*gauss_weight*l1l2l3[i];

			//noise dampening d/dki(1/2*(dk/dx)^2)
			grade_g_gaussian[i]+=-cm_noisedmp*Jdet*gauss_weight*(dh1dh3[0][i]*dk[0]+dh1dh3[1][i]*dk[1]);
			
			//min dampening
			if(drag<cm_mindmp_value){ 
				grade_g_gaussian[i]+=cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(drag>cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}
		}
		
		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<numgrids; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector grad_g: */
	VecSetValues(grad_g,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);
	
	cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::GradjDragStokes {{{1*/
void  Tria::GradjDragStokes(Vec grad_g,int analysis_type,int sub_analysis_type){

	int i;

	/* node data: */
	const int    numgrids=3;
	const int    NDOF2=2;
	double       xyz_list[numgrids][3];
	int          doflist1[numgrids];
	double       dh1dh3[NDOF2][numgrids];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vz_list[numgrids];
	double drag;
	double  thickness_list[numgrids];
	double  bed_list[numgrids];
	double  dragcoefficient_list[numgrids];
	double  drag_p,drag_q;
	double alpha_complement_list[numgrids];
	double alpha_complement;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* parameters: */
	double  vx,vy,vz;
	double  lambda,mu,xi;
	double  bed,thickness,Neff;
	double  surface_normal[3];
	double  bed_normal[3];
	double  dk[NDOF2]; 

	/*element vector at the gaussian points: */
	double  grade_g[numgrids]={0.0};
	double  grade_g_gaussian[numgrids];

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l2l3[3];

	/* strain rate: */
	double epsilon[3]; /* epsilon=[exx,eyy,exy];*/

	/*inputs: */
	bool shelf;
	int  drag_type;

	/*parameters: */
	double  cm_noisedmp;
	double  cm_mindmp_slope;
	double  cm_mindmp_value;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	this->parameters->FindParam(&cm_mindmp_value,CmMinDmpValueEnum);
	this->parameters->FindParam(&cm_mindmp_slope,CmMinDmpSlopeEnum);
	this->parameters->FindParam(&cm_maxdmp_value,CmMaxDmpValueEnum);
	this->parameters->FindParam(&cm_maxdmp_slope,CmMaxDmpSlopeEnum);

	/*Get out if shelf*/
	if(shelf)return;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList1(&doflist1[0]);

	/*Build alpha_complement_list: */
	if (drag_type==2){

		/*Allocate friction object: */
		Friction* friction=NewFriction();

		inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxAverageEnum);
		inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyAverageEnum);
		inputs->GetParameterValues(&vz_list[0],&gaussgrids[0][0],3,VzAverageEnum);
		inputs->GetParameterValues(&dragcoefficient_list[0],&gaussgrids[0][0],3,DragCoefficientEnum);
		inputs->GetParameterValues(&bed_list[0],&gaussgrids[0][0],3,BedEnum);
		inputs->GetParameterValues(&thickness_list[0],&gaussgrids[0][0],3,ThicknessEnum);
		inputs->GetParameterValue(&drag_p,DragPEnum);
		inputs->GetParameterValue(&drag_q,DragQEnum);


		/*Initialize all fields: */
		friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
		strcpy(friction->element_type,"2d");

		friction->gravity=matpar->GetG();
		friction->rho_ice=matpar->GetRhoIce();
		friction->rho_water=matpar->GetRhoWater();
		friction->K=&dragcoefficient_list[0];
		friction->bed=&bed_list[0];
		friction->thickness=&thickness_list[0];
		friction->vx=&vx_list[0];
		friction->vy=&vy_list[0];
		friction->p=drag_p;
		friction->q=drag_q;


		if(friction->p!=1) ISSMERROR("non-linear friction not supported yet in control methods!");

		/*Compute alpha complement: */
		FrictionGetAlphaComplement(&alpha_complement_list[0],friction);

		/*Erase friction object: */
		DeleteFriction(&friction);
	}
	else{
		alpha_complement_list[0]=0;
		alpha_complement_list[1]=0;
		alpha_complement_list[2]=0;
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 4);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/*Recover alpha_complement and k: */
		GetParameterValue(&alpha_complement, &alpha_complement_list[0],gauss_l1l2l3);
		inputs->GetParameterValue(&drag, &gauss_l1l2l3[0],DragCoefficientEnum);

		/*recover lambda mu and xi: */
		inputs->GetParameterValue(&lambda, &gauss_l1l2l3[0],AdjointxEnum);
		inputs->GetParameterValue(&mu, &gauss_l1l2l3[0],AdjointyEnum);
		inputs->GetParameterValue(&xi, &gauss_l1l2l3[0],AdjointzEnum);

		/*recover vx vy and vz: */
		inputs->GetParameterValue(&vx, &gauss_l1l2l3[0],VxEnum);
		inputs->GetParameterValue(&vy, &gauss_l1l2l3[0],VyEnum);
		inputs->GetParameterValue(&vz, &gauss_l1l2l3[0],VzEnum);

		/*Get normal vector to the bed */
		SurfaceNormal(&surface_normal[0],xyz_list);

		bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
		bed_normal[1]=-surface_normal[1];
		bed_normal[2]=-surface_normal[2];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss_l1l2l3);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss_l1l2l3);

		/*Get k derivative: dk/dx */
		inputs->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],&gauss_l1l2l3[0],DragCoefficientEnum);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<numgrids;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=(
						-lambda*(2*drag*alpha_complement*(vx - vz*bed_normal[0]*bed_normal[2]))
						-mu    *(2*drag*alpha_complement*(vy - vz*bed_normal[1]*bed_normal[2]))
						-xi    *(2*drag*alpha_complement*(-vx*bed_normal[0]*bed_normal[2]-vy*bed_normal[1]*bed_normal[2]))
						)*Jdet*gauss_weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]+= - cm_noisedmp*Jdet*gauss_weight*(dh1dh3[0][i]*dk[0]+dh1dh3[1][i]*dk[1]);

			//min dampening
			if(drag<cm_mindmp_value){ 
				grade_g_gaussian[i]+= cm_mindmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}

			//max dampening
			if(drag>cm_maxdmp_value){ 
				grade_g_gaussian[i]+= - cm_maxdmp_slope*Jdet*gauss_weight*l1l2l3[i];
			}
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<numgrids; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	/*Add grade_g to global vector grad_g: */
	VecSetValues(grad_g,numgrids,doflist1,(const double*)grade_g,ADD_VALUES);

cleanup_and_return: 
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Tria::MassFlux {{{1*/
double Tria::MassFlux( double* segment,double* ug){

	int i;

	const int    numgrids=3;
	const int    numdofs=2;
	int          numberofdofspernode;
	double mass_flux=0;
	int    doflist[numgrids*numdofs];
	double vx_list[3];
	double vy_list[3];
	double xyz_list[numgrids][3];
	double gauss_1[3];
	double gauss_2[3];
	double normal[2];
	double length;
	double x1,y1,x2,y2;
	double h1,h2;
	double vx1,vx2,vy1,vy2;
	double rho_ice;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();

	/*Get material parameters :*/
	rho_ice=matpar->GetRhoIce();

	/*First off, check that this segment belongs to this element: */
	if ((int)*(segment+4)!=this->id)ISSMERROR("%s%i%s%i","error message: segment with id ",(int)*(segment+4)," does not belong to element with id:",this->id);

	/*Recover segment node locations: */
	x1=*(segment+0); y1=*(segment+1); x2=*(segment+2); y2=*(segment+3);

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/*recover velocity at three element nodes: */
	this->GetDofList(&doflist[0],&numberofdofspernode);
	for(i=0;i<3;i++){
		vx_list[i]=ug[doflist[numberofdofspernode*i+0]];
		vy_list[i]=ug[doflist[numberofdofspernode*i+1]];
	}

	/*get area coordinates of 0 and 1 locations: */
	for(i=0;i<3;i++){
		gauss_1[i]=this->GetAreaCoordinate(x1,y1,i+1);
		gauss_2[i]=this->GetAreaCoordinate(x2,y2,i+1);
	}

	/*get normal of segment: */
	normal[0]=cos(atan2(x1-x2,y2-y1));
	normal[1]=sin(atan2(x1-x2,y2-y1));

	/*get length of segment: */
	length=sqrt(pow(x2-x1,2.0)+pow(y2-y1,2));

	/*get thickness and velocity at two segment extremities: */
	inputs->GetParameterValue(&h1, &gauss_1[0],ThicknessEnum);
	inputs->GetParameterValue(&h2, &gauss_2[0],ThicknessEnum);
	inputs->GetParameterValue(&vx1, &gauss_1[0],VxEnum);
	inputs->GetParameterValue(&vx2, &gauss_2[0],VxEnum);
	inputs->GetParameterValue(&vy1, &gauss_1[0],VyEnum);
	inputs->GetParameterValue(&vy2, &gauss_2[0],VyEnum);

	mass_flux= rho_ice*length*(  
				(ONETHIRD*(h1-h2)*(vx1-vx2)+0.5*h2*(vx1-vx2)+0.5*(h1-h2)*vx2+h2*vx2)*normal[0]+
				(ONETHIRD*(h1-h2)*(vy1-vy2)+0.5*h2*(vy1-vy2)+0.5*(h1-h2)*vy2+h2*vy2)*normal[1]
				);
	return mass_flux;
}
/*}}}*/
/*FUNCTION Tria::Misfit {{{1*/
double Tria::Misfit(int analysis_type,int sub_analysis_type){

	int i;

	/* output: */
	double Jelem=0;

	/* node data: */
	const int    numgrids=3;
	const int    numdof=2*numgrids;
	const int    NDOF2=2;
	int          dofs1[1]={0};
	int          dofs2[2]={0,1};
	double       xyz_list[numgrids][3];

	/* grid data: */
	double vx_list[numgrids];
	double vy_list[numgrids];
	double obs_vx_list[numgrids];
	double obs_vy_list[numgrids];
	double misfit_list[numgrids];
	double weights_list[numgrids];

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* gauss_weights           =  NULL;
	double  gauss_weight;
	double  gauss_l1l2l3[3];
	double  gaussgrids[numgrids][numgrids]={{1,0,0},{0,1,0},{0,0,1}};

	/* parameters: */
	double  velocity_mag,obs_velocity_mag;
	double  misfit;

	/* Jacobian: */
	double Jdet;

	/*relative and logarithmic control method :*/
	double scalex=1;
	double scaley=1;
	double S=0;
	double fit=-1;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*inputs: */
	bool onwater;
	
	double  meanvel, epsvel;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If on water, return 0: */
	if(onwater)return 0;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/* Recover input data: */
	inputs->GetParameterValue(&fit,FitEnum);
	if(fit==3){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}
	inputs->GetParameterValues(&obs_vx_list[0],&gaussgrids[0][0],3,VxObsEnum);
	inputs->GetParameterValues(&obs_vy_list[0],&gaussgrids[0][0],3,VyObsEnum);
	inputs->GetParameterValues(&vx_list[0],&gaussgrids[0][0],3,VxEnum);
	inputs->GetParameterValues(&vy_list[0],&gaussgrids[0][0],3,VyEnum);
	inputs->GetParameterValues(&weights_list[0],&gaussgrids[0][0],3,WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	
	/* Compute Misfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(fit==0){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 */
		for (i=0;i<numgrids;i++){
			misfit_list[i]=0.5*(pow((vx_list[i]-obs_vx_list[i]),(double)2)+pow((vy_list[i]-obs_vy_list[i]),(double)2));
		}
	}
	else if(fit==1){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 */
		for (i=0;i<numgrids;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),(double)2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),(double)2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			misfit_list[i]=0.5*(scalex*pow((vx_list[i]-obs_vx_list[i]),2)+scaley*pow((vy_list[i]-obs_vy_list[i]),2));
		}
	}
	else if(fit==2){
		/*We are using a logarithmic misfit:
		*                        
		*                 [        vel + eps     ] 2
		* J = 4 \bar{v}^2 | log ( -----------  ) |  
		*                 [       vel   + eps    ]
		*                            obs
		*/
		for (i=0;i<numgrids;i++){
			velocity_mag=sqrt(pow(vx_list[i],(double)2)+pow(vy_list[i],(double)2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],(double)2)+pow(obs_vy_list[i],(double)2))+epsvel; //epsvel to avoid observed velocity being nil.
			misfit_list[i]=4*pow(meanvel,(double)2)*pow(log(velocity_mag/obs_velocity_mag),(double)2);
		}
	}
	else if(fit==3){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 */
		for (i=0;i<numgrids;i++){
			misfit_list[i]=sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))/S;
		}
	}
	else if(fit==4){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 */
		for (i=0;i<numgrids;i++){
			misfit_list[i]=0.5*pow(meanvel,(double)2)*(
			  pow(log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)),(double)2) +
			  pow(log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)),(double)2) );
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("%s%g","unsupported type of fit: ",fit);
	}

	/*Apply weights to misfits*/
	for (i=0;i<numgrids;i++){
		misfit_list[i]=weights_list[i]*misfit_list[i];
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussTria( &num_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &gauss_weights, 2);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){
		/*Pick up the gaussian point: */
		gauss_weight=*(gauss_weights+ig);
		gauss_l1l2l3[0]=*(first_gauss_area_coord+ig); 
		gauss_l1l2l3[1]=*(second_gauss_area_coord+ig);
		gauss_l1l2l3[2]=*(third_gauss_area_coord+ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss_l1l2l3);

		/*Compute misfit at gaussian point: */
		GetParameterValue(&misfit, &misfit_list[0],gauss_l1l2l3);

		/*compute Misfit*/
		Jelem+=misfit*Jdet*gauss_weight;
	}

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&gauss_weights);
		
	/*Return: */
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::MyRank {{{1*/
int    Tria::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Tria::Tria::SetClone {{{1*/
void  Tria::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Tria::SurfaceNormal{{{1*/

void Tria::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}
/*}}}*/
/*FUNCTION Tria::SurfaceArea {{{1*/
double Tria::SurfaceArea(int analysis_type,int sub_analysis_type){

	int i;

	/* output: */
	double S;

	/* node data: */
	int numgrids=3;
	double xyz_list[numgrids][3];
	double v13[3];
	double v23[3];
	double normal[3];

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*inputs: */
	bool onwater;
	int  fit;

	/*retrieve inputs :*/
	inputs->GetParameterValue(&fit,FitEnum);
	inputs->GetParameterValue(&onwater,ElementOnWaterEnum);

	/*If fit!=3, do not compute surface: */
	if(fit!=3)return 0;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();

	/*If on water, return 0: */
	if(onwater)return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	S = 0.5 * sqrt(pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2));

	/*Return: */
	return S;
}
/*}}}*/
