/*!\file: CreateParametersQmu.cpp
 * \brief general driver for creating parameters dataset
 */ 

#include "../../DataSet/DataSet.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../../include/macros.h"
#include "../../MeshPartitionx/MeshPartitionx.h"
#include "../ModelProcessorx.h"

void CreateParametersQmu(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle){
	
	int i,j,k;
	
	DataSet* parameters = NULL;
	Param*   param = NULL;
	int      count;
	int      second_count;
	
	int*     part=NULL;
	double*  dpart=NULL;

	char**  responsedescriptors=NULL;
	char**  variabledescriptors=NULL;
	char*   descriptor=NULL;
	char*   tag=NULL;
	double* dakota_parameter=NULL;

	//qmu files
	char* qmuinname=NULL;
	char* qmuerrname=NULL;
	char* qmuoutname=NULL;
	extern int my_rank;
				
	
	/*parameters for mass flux: */
	double* qmu_mass_flux_segments=NULL;
	double* my_qmu_mass_flux_segments=NULL;
	int num_qmu_mass_flux_segments=0;
	int my_num_qmu_mass_flux_segments=0;
	
	/*parameters for misfit: */
	double* u_g_obs=NULL;

	#ifdef _SERIAL_
		mxArray* pfield=NULL;
		mxArray* pfield2=NULL;
	#endif
	
	/*recover parameters : */
	parameters=*pparameters;

	count=parameters->Size();

	//qmu analysis?
	count++;
	param= new Param(count,"qmu_analysis",DOUBLE);
	param->SetDouble(iomodel->qmu_analysis);
	parameters->AddObject(param);


	if(iomodel->qmu_analysis){

		//name of qmu input, error and output files
		qmuinname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.in")+1)*sizeof(char));
		sprintf(qmuinname,"%s%s",iomodel->name,".qmu.in");
		
		count++;
		param= new Param(count,"qmuinname",STRING);
		param->SetString(qmuinname);
		parameters->AddObject(param);

		qmuoutname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.out")+1)*sizeof(char));
		sprintf(qmuoutname,"%s%s",iomodel->name,".qmu.out");
		
		count++;
		param= new Param(count,"qmuoutname",STRING);
		param->SetString(qmuoutname);
		parameters->AddObject(param);

		qmuerrname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.err")+1)*sizeof(char));
		sprintf(qmuerrname,"%s%s",iomodel->name,".qmu.err");
		
		count++;
		param= new Param(count,"qmuerrname",STRING);
		param->SetString(qmuerrname);
		parameters->AddObject(param);

		//npart
		count++;
		param= new Param(count,"qmu_npart",DOUBLE);
		param->SetDouble(iomodel->qmu_npart);
		parameters->AddObject(param);

		/*Deal with variables for qmu iomodeling: */
		variabledescriptors=(char**)xmalloc(iomodel->numberofvariables*sizeof(char*));


		/*Fetch descriptors: logic varies if we are running parallel or serial. In parallel, qmumarshall 
		 * took care of marshalling all the variable descriptors, so it's easy. In serial mode, 
		 * the variables are in md.variables(md.ivar), as a strucuture: */

		#ifdef _SERIAL_
		pfield=mxGetField(iomodel_handle,0,"variabledescriptors");
		for(i=0;i<iomodel->numberofvariables;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData(&descriptor,pfield2);
			variabledescriptors[i]=descriptor;
		}
		#else
		tag=(char*)xmalloc((strlen("variabledescriptori")+1)*sizeof(char));
		for(i=0;i<iomodel->numberofvariables;i++){
			sprintf(tag,"%s%i","variabledescriptor",i);
			IoModelFetchData(&descriptor,iomodel_handle,tag);
			variabledescriptors[i]=descriptor;
		}
		#endif


		/*Ok, we have all the variable descriptors. Build a parameter with it: */
		count++;
		param= new Param(count,"variabledescriptors",STRINGARRAY);
		param->SetStringArray(variabledescriptors,iomodel->numberofvariables);
		parameters->AddObject(param);


		/*Deal with responses and partition for qmu iomodeling: */
		responsedescriptors=(char**)xmalloc(iomodel->numberofresponses*sizeof(char*));

		/*Fetch descriptors: */
		#ifdef _SERIAL_
		pfield=mxGetField(iomodel_handle,0,"responsedescriptors");
		for(i=0;i<iomodel->numberofresponses;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData(&descriptor,pfield2);
			responsedescriptors[i]=descriptor;
		}
		#else
		xfree((void**)&tag);
		tag=(char*)xmalloc((strlen("responsedescriptori")+1)*sizeof(char));

		for(i=0;i<iomodel->numberofresponses;i++){
			sprintf(tag,"%s%i","responsedescriptor",i);
			IoModelFetchData(&descriptor,iomodel_handle,tag);
			responsedescriptors[i]=descriptor;
		}
		#endif

		/*Ok, we have all the response descriptors. Build a parameter with it: */
		count++;
		param= new Param(count,"responsedescriptors",STRINGARRAY);
		param->SetStringArray(responsedescriptors,iomodel->numberofresponses);
		parameters->AddObject(param);

		/*partition vertices in iomodel->qmu_npart parts, unless a partition is already present: */
		IoModelFetchData(&dpart,NULL,NULL,iomodel_handle,"part");

		if(!dpart){

			/*Partition elements and vertices and nodes: */
			Partitioning(&iomodel->my_elements, &iomodel->my_vertices, &iomodel->my_nodes, &iomodel->my_bordervertices, iomodel, iomodel_handle);

			dpart=(double*)xmalloc(iomodel->numberofvertices*sizeof(double));
			for(i=0;i<iomodel->numberofvertices;i++)dpart[i]=iomodel->my_vertices[i];
		}

		count++;
		param= new Param(count,"qmu_part",DOUBLEVEC);
		param->SetDoubleVec(dpart,iomodel->numberofvertices,1);
		parameters->AddObject(param);


		/*Ok, now if any of the variables input from Dakota are distributed, we are going to need the parameters: */
		for(i=0;i<iomodel->numberofvariables;i++){

			descriptor=variabledescriptors[i];

			if ((strcmp(descriptor,"thickness")==0) ||
				(strcmp(descriptor,"drag")     ==0)
				){

				//Fetch data: 
				IoModelFetchData(&dakota_parameter,NULL,NULL,iomodel_handle,descriptor);

				//Add parameter
				count++;
				param= new Param(count,descriptor,DOUBLEVEC);
				param->SetDoubleVec(dakota_parameter,iomodel->numberofvertices,1);
				parameters->AddObject(param);

				//free data
				xfree((void**)&dakota_parameter);

			}
		}

		/*Deal with data needed for some responses: */
		for(i=0;i<iomodel->numberofresponses;i++){
			char* descriptor=responsedescriptors[i];
			if (strcmp(descriptor,"mass_flux")==0){

				/*We need the qmu_mass_flux_segments to be able to compute the mass flux: */
				IoModelFetchData(&qmu_mass_flux_segments,&num_qmu_mass_flux_segments,NULL,iomodel_handle,"qmu_mass_flux_segments");

				#ifdef _PARALLEL_

					/*Only if partitioning exist do we care about the segments: */
					if(iomodel->my_elements){

						/*Use the element partitioning vector from the iomodel to down select qmu_mass_flux_segments to only segments that are relevant 
						 * to this cpu: */
						my_num_qmu_mass_flux_segments=0;
						for(j=0;j<num_qmu_mass_flux_segments;j++){
							if (  iomodel->my_elements[(int)(*(qmu_mass_flux_segments+5*j+4))-1])my_num_qmu_mass_flux_segments++;
						}

					
						if(my_num_qmu_mass_flux_segments){
							my_qmu_mass_flux_segments=(double*)xcalloc(5*my_num_qmu_mass_flux_segments,sizeof(double));
							second_count=0;
							for(j=0;j<num_qmu_mass_flux_segments;j++){
								if (iomodel->my_elements[(int)*(qmu_mass_flux_segments+5*j+4)-1]){
									for(k=0;k<5;k++)*(my_qmu_mass_flux_segments+5*second_count+k)=*(qmu_mass_flux_segments+5*j+k);
									second_count++;
								}
							}
						}

						count++;
						param= new Param(count,"qmu_mass_flux_segments",DOUBLEMAT);
						param->SetDoubleMat(my_qmu_mass_flux_segments,my_num_qmu_mass_flux_segments,5);
						parameters->AddObject(param);

					}
					
				#else

					count++;
					param= new Param(count,"qmu_mass_flux_segments",DOUBLEMAT);
					param->SetDoubleMat(qmu_mass_flux_segments,num_qmu_mass_flux_segments,5);
					parameters->AddObject(param);

				#endif

				xfree((void**)&qmu_mass_flux_segments);
				xfree((void**)&my_qmu_mass_flux_segments);
			}
			if (strcmp(descriptor,"misfit")==0){

				/*We need the observed velocity: */
				IoModelFetchData(&iomodel->vx_obs,NULL,NULL,iomodel_handle,"vx_obs");
				IoModelFetchData(&iomodel->vy_obs,NULL,NULL,iomodel_handle,"vy_obs");

				/*Now, recover fit: */
				IoModelFetchData(&iomodel->fit,NULL,NULL,iomodel_handle,"fit");

				u_g_obs=(double*)xcalloc(iomodel->numberofvertices*2,sizeof(double));
				if(iomodel->vx_obs)for(i=0;i<iomodel->numberofvertices;i++)u_g_obs[2*i+0]=iomodel->vx_obs[i]/iomodel->yts;
				if(iomodel->vy_obs)for(i=0;i<iomodel->numberofvertices;i++)u_g_obs[2*i+1]=iomodel->vy_obs[i]/iomodel->yts;

				count++;
				param= new Param(count,"velocity_obs",DOUBLEVEC);
				param->SetDoubleVec(u_g_obs,2*iomodel->numberofvertices,2);
				parameters->AddObject(param);

				count++;
				param= new Param(count,"fit",DOUBLEVEC);
				param->SetDoubleVec(iomodel->fit,iomodel->nsteps);
				parameters->AddObject(param);

				xfree((void**)&iomodel->vx_obs);
				xfree((void**)&iomodel->vy_obs);
				xfree((void**)&iomodel->fit);
				xfree((void**)&u_g_obs);

			}
		}

		/*Free data: */
		xfree((void**)&tag);
		for(i=0;i<iomodel->numberofresponses;i++){
			char* descriptor=responsedescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&responsedescriptors);

		for(i=0;i<iomodel->numberofvariables;i++){
			char* descriptor=variabledescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&variabledescriptors);

		xfree((void**)&iomodel->elements);
		xfree((void**)&iomodel->elements2d);
		xfree((void**)&part);
		xfree((void**)&dpart);
		xfree((void**)&qmuinname);
		xfree((void**)&qmuerrname);
		xfree((void**)&qmuoutname);
	}

	/*Assign output pointer: */
	*pparameters=parameters;
}
